# Use IR to Generate Couler Program

## Background

We are planning to migrate SQLFlow submitters to use Couler: https://github.com/sql-machine-learning/sqlflow/pull/1146/.

In this design, we'll only have `codegen_couler.go` to generate programs to run a SQL program, which will be translated into multiple couler "steps". Each step, we can execute a SQL statement, run a training python program or run prediction, etc.

## Generate Couler Program Using IR

For example, to run the training steps, we intend to use couler API call: `couler.{xgboost,tensorflow,elasticdl}.train(model_def, data)` to generate a training step. The `train` function must take lots of parameters to run the training job as the SQLFlow statement describes, see [here](https://github.com/sql-machine-learning/sqlflow/blob/develop/python/runtime/tensorflow/train.py#L52) as an example.

To implement the single `codegen_couler.go` to support generate code that can run either TensorFlow/XGBoost/ElasticDL/ALPS programs, we have below two choices:

1. `couler.{tensorflow/xgboost/elasticdl}.train` have different arguments defined, so we can do:

    ```go
        if ir.ModelType == "TensorFlow":
            tfFiller := TFFiller{
                Estimator: generateTFEstimatorCode(ir),
                FeatureColumns: generateFeatureColumnsCode(ir),
                FieldDescs: generateFieldDescsCode(ir),
                ...
            }
            tfTrainTemplate = `couler.tensorflow.train(estimator="{{.Estimator}}", FeatureColumns="""{{.FeatureColumns}}""", FieldDescs={{.FieldDescs}})`
            // Do template rendering here.
        else if ir.ModelType == "XGBoost":
            ...
    ```
2. `couler.{tensorflow/xgboost/elasticdl}.train` accept a parameter called `train_config` which is a python dictionary that can be generated by SQLFlow IR.

    ```go
        filler := Filler{
            ModelType: ir.ModelType,
            TrainConfigCode : generateTrainConfig(ir),
        }
        tfTrainTemplate = `couler.{{.ModelType}}.train(train_config={{.TrainConfigCode}})`
        // Do template rendering here.
    ```
    **Note** that the `train_config` dictionary is not "engine specific", so that it could be used by any engine. A sample `train_config` should look like:

    ```python
        train_config = {
            "DataSource": "mysql://user:pass@192.168.1.1:3306",
            "Select": "SELECT * FROM iris.train",
            "ValidationSelect": "SELECT * FROM iris.test",
            "Estimator": "XGBOOST.gbtree",
            "Attributes": {"train.num_boost_round": 30},
            "Features": {"sepal_length": {"type": "numeric", "shape": [1], "field_meta":{"name":"sepal_length", "dtype": "float32", "delimiter": "", "is_sparse": False}}...},
            "Label": {"class": {"type": "numeric", "shape": [1], "field_meta": ...}}
        }
    ```

## The Design Choice

We intend to use the solution **No.2** described above for these reasons.

1. If a data scientist needs to add a new type of engine (SVM, SKLearn, PyTorch, etc.) other than TensorFlow/XGBoost/ElasticDL, he/she can use python only to define a `couler.{new_engine}.train` function without modifying the Go code in SQLFlow.
1. `codegen_couler.go` have less code.
1. All submitter unit tests can run in couler.
