/*
 * Copyright 2013-2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.sleuth.autoconfig.instrument.messaging;

import org.junit.jupiter.api.Test;

import org.springframework.boot.autoconfigure.AutoConfigurations;
import org.springframework.boot.test.context.FilteredClassLoader;
import org.springframework.boot.test.context.runner.ApplicationContextRunner;
import org.springframework.cloud.function.context.FunctionCatalog;
import org.springframework.cloud.sleuth.autoconfig.TraceNoOpAutoConfiguration;
import org.springframework.cloud.sleuth.instrument.messaging.TracingChannelInterceptor;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.context.annotation.Configuration;

import static org.assertj.core.api.Assertions.assertThat;

class TraceSpringIntegrationAutoConfigurationTests {

	private final ApplicationContextRunner contextRunner = new ApplicationContextRunner()
			.withPropertyValues("spring.sleuth.noop.enabled=true")
			.withConfiguration(AutoConfigurations.of(TraceNoOpAutoConfiguration.class,
					TraceSpringMessagingAutoConfiguration.class, TraceSpringIntegrationAutoConfiguration.class));

	@Test
	void should_not_create_tracing_channel_interceptor_when_function_on_the_classpath() {
		this.contextRunner.run(context -> assertThat(context).doesNotHaveBean(TracingChannelInterceptor.class));
	}

	@Test
	void should_create_tracing_channel_interceptor_when_function_not_on_the_classpath() {
		this.contextRunner.withClassLoader(new FilteredClassLoader(FunctionCatalog.class))
				.run(context -> assertThat(context).hasSingleBean(TracingChannelInterceptor.class));
	}

	@Test
	void should_create_tracing_channel_interceptor_when_function_on_the_classpath_but_enable_binding_is_set() {
		this.contextRunner.withUserConfiguration(WithEnabledBinding.class)
				.run(context -> assertThat(context).hasSingleBean(TracingChannelInterceptor.class));
	}

	@Test
	void should_create_tracing_channel_interceptor_when_function_on_the_classpath_no_enable_binding_is_set_and_property_set() {
		this.contextRunner.withSystemProperties("spring.sleuth.integration.enabled=true")
				.run(context -> assertThat(context).hasSingleBean(TracingChannelInterceptor.class));
	}

	@Configuration(proxyBeanMethods = false)
	@EnableBinding
	static class WithEnabledBinding {

	}

}
