<?php declare(strict_types=1);

namespace Sprain\Tests\SwissQrBill\QrCode;

use PHPUnit\Framework\TestCase;
use Sprain\SwissQrBill\QrCode\Exception\UnsupportedFileExtensionException;
use Sprain\SwissQrBill\QrCode\QrCode;
use Sprain\Tests\SwissQrBill\TestCompactSvgQrCodeTrait;

final class QrCodeTest extends TestCase
{
    use TestCompactSvgQrCodeTrait;

    /**
     * @dataProvider supportedExtensionsProvider
     */
    public function testSupportedFileExtensions(string $extension): void
    {
        $qrCode = QrCode::create('This is a test code');
        $testfile = __DIR__ . '/../TestData/testfile.' . $extension;

        if (!is_writable(dirname($testfile))) {
            $this->markTestSkipped();
        }

        $qrCode->writeFile($testfile);
        $this->assertTrue(file_exists($testfile));
        unlink($testfile);
    }

    public function supportedExtensionsProvider(): array
    {
        return [
            ['svg'],
            ['png']
        ];
    }

    /**
     * @dataProvider unsupportedExtensionsProvider
     */
    public function testUnsupportedFileExtensions(?string $extension): void
    {
        $this->expectException(UnsupportedFileExtensionException::class);

        $qrCode = QrCode::create('This is a test code');
        $qrCode->writeFile(__DIR__ . '/../TestData/testfile.' . $extension);
    }

    public function unsupportedExtensionsProvider(): array
    {
        return [
            ['eps'],
            ['jpg'],
            ['gif'],
            [''],
            [null]
        ];
    }

    /**
     * @dataProvider dataUriProvider
     */
    public function testDataUri(string $code, string $dataUri, string $format): void
    {
        $qrCode = QrCode::create($code);
        $this->assertEquals(
            $dataUri,
            $qrCode->getDataUri($format)
        );
    }

    public function dataUriProvider()
    {
        return [
            # PNGs do not create the same output in all environments
            # [
            #     'code' => 'This is a test code',
            #     'dataUri' => 'data:image/png;base64,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',
            #     QrCode::FILE_FORMAT_PNG
            # ],
            [
                'code' => 'This is a test code',
                'dataUri' => $this->getCompact()
                    ? 'data:image/svg+xml;base64,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'
                    : 'data:image/svg+xml;base64,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',
                QrCode::FILE_FORMAT_SVG
            ]
        ];
    }

    /**
     * @dataProvider stringProvider
     */
    public function testString(string $code, string $string, string $format): void
    {
        $qrCode = QrCode::create($code);

        $this->assertEquals(
            $string,
            $qrCode->getAsString($format)
        );
    }

    public function stringProvider()
    {
        return [
            # PNGs do not create the same output in all environments
            # [
            #     'code' => 'This is a test code',
            #     'string' => file_get_contents(__DIR__ . '/../TestData/QrCode/string.png'),
            #     'format' => QrCode::FILE_FORMAT_PNG
            # ],
            [
                'code' => 'This is a test code',
                'string' => file_get_contents(__DIR__ . '/../TestData/QrCode/string' . $this->getCompact() . '.svg'),
                'format' => QrCode::FILE_FORMAT_SVG
            ]
        ];
    }

    /**
     * @dataProvider replacementCharactersProvider
     */
    public function testItReplacesUnsupportedCharacters(string $providedString, array $replacements, string $expectedString): void
    {
        $qrCode = QrCode::create($providedString, null, $replacements);

        $this->assertEquals(
            $expectedString,
            $qrCode->getText()
        );
    }

    public function replacementCharactersProvider(): array
    {
        return [
            'replaceSpecificUnsupportedCharacters' => [
                'providedString' => '«This is a test!»',
                'replacements' => [
                    '«' => '"',
                    '»' => '"'
                ],
                'expectedString' => '"This is a test!"'
            ],
            'ignoreReplacementsOfSupportedCharacters' => [
                'providedString' => '«This is a test!»',
                'replacements' => [
                    't' => 'a',
                ],
                'expectedString' => 'This is a test!'
            ],
        ];
    }

    /**
     * @dataProvider unsupportedCharactersProvider
     */
    public function testItRemovesUnsupportedCharacters(string $providedString, string $expectedString): void
    {
        $qrCode = QrCode::create($providedString);

        $this->assertEquals(
            $expectedString,
            $qrCode->getText()
        );
    }

    public function unsupportedCharactersProvider(): array
    {
        return [
            'keepAllAllowedCharacters' => [
                'providedString' => 'a-zA-Z0-9.,;:\'+-/()?*[]{}|`´~!"#%&<>÷=@_$£^àáâäçèéêëìíîïñòóôöùúûüýßÀÁÂÄÇÈÉÊËÌÍÎÏÒÓÔÖÙÚÛÜÑ',
                'expectedString' => 'a-zA-Z0-9.,;:\'+-/()?*[]{}|`´~!"#%&<>÷=@_$£^àáâäçèéêëìíîïñòóôöùúûüýßÀÁÂÄÇÈÉÊËÌÍÎÏÒÓÔÖÙÚÛÜÑ'
            ],
            'removeUnallowedCharacters' => [
                'providedString' => '«This is a test!»',
                'expectedString' => 'This is a test!'
            ],
        ];
    }
}
