// -- Copyright (c) 2016-present, Facebook, Inc.
// -- All rights reserved.
// https://github.com/facebook/duckling/edit/master/Duckling/Time/EN/Corpus.hs
// This source code is licensed under the BSD-style license
import test from 'tape'
import spacetime from 'spacetime'
import nlp from '../_lib.js'
const isArray = function (arr) {
  return Object.prototype.toString.call(arr) === '[object Array]'
}

const fmt = (iso) => (iso ? spacetime(iso).format('iso-short') : '')

 // eslint-disable-next-line
const mk = function (y, m, d, h, sec, mil) {
  if (isArray(y) && isArray(m)) {
    return [mk(y) + mk(m)]
  }
  let s = spacetime.now().year(y)
  if (m) {
    s = s.month(m - 1)
  }
  if (d) {
    s = s.date(d)
  }
  if (h) {
    s = s.hour(h)
  }
  if (sec) {
    s = s.second(sec)
  }
  if (mil) {
    s = s.millisecond(mil)
  }
  return fmt(s)
}

let arr = [
  [mk([2013, 2, 15, 0, 0, 0]), ['2/15', 'on 2/15', '2 / 15', '2-15', '2 - 15']],
  [mk([1974, 10, 31, 0, 0, 0]), ['10/31/1974', '10/31/74', '10-31-74', '10.31.1974', '31/Oct/1974', '31st Oct 1974']],
  [mk([2013, 4, 25, 16, 0, 0]), ['4/25 at 4:00pm', '4/25 at 16h00', '4/25 at 16h']],
  [
    mk([2013, 11, 28, 0, 0, 0]),
    [
      'Thanksgiving Day',
      'thanksgiving day',
      'thanksgiving',
      'thanksgiving 2013',
      'this thanksgiving',
      'next thanksgiving day',
      'thanksgiving in 9 months',
      'thanksgiving 9 months from now',
    ],
  ],
  [
    mk([2014, 11, 27, 0, 0, 0]),
    ['Thanksgiving Day', 'thanksgiving of next year', 'thanksgiving in a year', 'thanksgiving 2014'],
  ],
  [
    mk([2012, 11, 22, 0, 0, 0]),
    [
      'Thanksgiving Day',
      'last thanksgiving',
      'thanksgiving, 2012',
      'thanksgiving 3 months ago',
      'thanksgiving 1 year ago',
    ],
  ],
  [mk([2016, 11, 24, 0, 0, 0]), ['Thanksgiving Day', 'thanksgiving 2016', 'thanksgiving in 3 years']],
  [mk([2017, 11, 23, 0, 0, 0]), ['Thanksgiving Day', 'thanksgiving 2017']],
  [mk([2013, 3, 8, 0, 0, 0]), ['3 fridays from now', 'three fridays from now']],
  [mk([2013, 2, 12, 4, 30, 0]), ['now', 'right now', 'just now']],
  [mk([2013, 2, 12, 0, 0, 0]), ['today', 'at this time']],
  [mk([2013, 2, 1, 0, 0, 0]), ['2/2013']],
  [mk([2014, 1, 1, 0, 0, 0]), ['in 2014']],
  [mk([2013, 2, 11, 0, 0, 0]), ['yesterday']],
  [mk([2013, 2, 13, 0, 0, 0]), ['tomorrow', 'tomorrows']],
  [mk([2013, 2, 18, 0, 0, 0]), ['monday', 'mon.', 'this monday', 'Monday, Feb 18', 'Mon, February 18']],
  [mk([2013, 2, 19, 0, 0, 0]), ['tuesday', 'Tuesday the 19th', 'Tuesday 19th']],
  [mk([2013, 8, 15, 0, 0, 0]), ['Thu 15th']],
  [mk([2013, 2, 14, 0, 0, 0]), ['thursday', 'thu', 'thu.']],
  [mk([2013, 2, 15, 0, 0, 0]), ['friday', 'fri', 'fri.']],
  [mk([2013, 2, 16, 0, 0, 0]), ['saturday', 'sat', 'sat.']],
  [mk([2013, 2, 17, 0, 0, 0]), ['sunday', 'sun', 'sun.']],
  [mk([2013, 3, 1, 0, 0, 0]), ['the 1st of march', 'first of march', 'the first of march', 'march first']],
  [mk([2013, 3, 2, 0, 0, 0]), ['the 2nd of march', 'second of march', 'the second of march']],
  [mk([2013, 3, 3, 0, 0, 0]), ['march 3', 'the third of march']],
  [mk([2013, 3, 15, 0, 0, 0]), ['the ides of march']],
  [
    mk([2015, 3, 3, 0, 0, 0]),
    ['march 3 2015', 'march 3rd 2015', 'march third 2015', '3/3/2015', '3/3/15', '2015-3-3', '2015-03-03'],
  ],
  [
    mk([2013, 2, 15, 0, 0, 0]),
    [
      'on the 15th',
      'the 15th of february',
      '15 of february',
      'february the 15th',
      'february 15',
      '15th february',
      'February 15',
    ],
  ],
  [mk([2013, 8, 8, 0, 0, 0]), ['Aug 8']],
  [mk([2014, 3, 1, 0, 0, 0]), ['March in 1 year', 'March in a year']],
  [mk([2014, 7, 18, 0, 0, 0]), ['Fri, Jul 18', 'Jul 18, Fri']],
  [mk([2014, 10, 1, 0, 0, 0]), ['October 2014', '2014-10', '2014/10']],
  [mk([2015, 4, 14, 0, 0, 0]), ['14april 2015', 'April 14, 2015', '14th April 15']],
  [mk([2013, 2, 19, 0, 0, 0]), ['next tuesday', 'around next tuesday']],
  [mk([2013, 2, 22, 0, 0, 0]), ['friday after next']],
  [mk([2013, 3, 1, 0, 0, 0]), ['next March']],
  [mk([2014, 3, 1, 0, 0, 0]), ['March after next']],
  [mk([2013, 2, 10, 0, 0, 0]), ['Sunday, Feb 10']],
  [mk([2013, 2, 13, 0, 0, 0]), ['Wed, Feb13']],
  [mk([2013, 2, 11, 0, 0, 0]), ['this week', 'current week']],
  [mk([2013, 2, 18, 0, 0, 0]), ['coming week']],
  [mk([2013, 2, 4, 0, 0, 0]), ['last week', 'past week', 'previous week']],
  [mk([2013, 2, 18, 0, 0, 0]), ['next week', 'the following week', 'around next week', 'upcoming week']],
  [mk([2013, 1, 1, 0, 0, 0]), ['last month']],
  [mk([2013, 3, 1, 0, 0, 0]), ['next month']],
  [mk([2013, 3, 20, 0, 0, 0]), ['20 of next month', '20th of the next month', '20th, of next month']],
  [mk([2013, 2, 20, 0, 0, 0]), ['20th of the current month', '20 of this month']],
  [mk([2013, 1, 20, 0, 0, 0]), ['20th of the previous month']],
  [mk([2013, 1, 1, 0, 0, 0]), ['this quarter', 'this qtr']],
  [mk([2013, 4, 1, 0, 0, 0]), ['next quarter', 'next qtr']],
  [mk([2013, 7, 1, 0, 0, 0]), ['third quarter', '3rd quarter', 'third qtr', '3rd qtr', 'the 3rd qtr']],
  [mk([2018, 10, 1, 0, 0, 0]), ['4th quarter 2018', '4th qtr 2018', 'the 4th qtr of 2018', '18q4', '2018Q4']],
  [mk([2012, 1, 1, 0, 0, 0]), ['last year', 'last yr']],
  [mk([2013, 1, 1, 0, 0, 0]), ['this year', 'current year', 'this yr']],
  [mk([2014, 1, 1, 0, 0, 0]), ['next year', 'next yr']],
  [mk([2014, 1, 1, 0, 0, 0]), ['in 2014 A.D.', 'in 2014 AD']],
  [mk([-2014, 1, 1, 0, 0, 0]), ['in 2014 B.C.', 'in 2014 BC']],
  [mk([14, 1, 1, 0, 0, 0]), ['in 14 a.d.']],
  [mk([2013, 2, 10, 0, 0, 0]), ['last sunday', 'sunday from last week', "last week's sunday"]],
  [mk([2013, 2, 5, 0, 0, 0]), ['last tuesday']],
  [mk([2013, 2, 20, 0, 0, 0]), ['next wednesday']],
  [mk([2013, 2, 20, 0, 0, 0]), ['wednesday of next week', 'wednesday next week', 'wednesday after next']],
  [mk([2013, 2, 22, 0, 0, 0]), ['friday after next']],
  [mk([2013, 2, 11, 0, 0, 0]), ['monday of this week']],
  [mk([2013, 2, 12, 0, 0, 0]), ['tuesday of this week']],
  [mk([2013, 2, 13, 0, 0, 0]), ['wednesday of this week']],
  [mk([2013, 2, 14, 0, 0, 0]), ['the, after tomorrow']],
  [mk([2013, 2, 14, 17, 0, 0]), ['day after tomorrow 5pm']],
  [mk([2013, 2, 10, 0, 0, 0]), ['the, before yesterday']],
  [mk([2013, 2, 10, 8, 0, 0]), ['day before yesterday 8am']],
  [mk([2013, 3, 25, 0, 0, 0]), ['last Monday of March']],
  [mk([2014, 3, 30, 0, 0, 0]), ['last Sunday of March 2014']],
  [mk([2013, 10, 3, 0, 0, 0]), ['third, of october']],
  [mk([2014, 10, 6, 0, 0, 0]), ['first week of october 2014']],
  [
    mk([2018, 12, 10, 0, 0, 0]),
    [('third last week of 2018', 'the third last week of 2018', 'the 3rd last week of 2018')],
  ],
  [mk([2018, 10, 15, 0, 0, 0]), ['2nd last week of October 2018', 'the second last week of October 2018']],
  [mk([2013, 5, 27, 0, 0, 0]), ['fifth last, of May', 'the 5th last, of May']],
  [mk([2013, 10, 7, 0, 0, 0]), ['the week of october 6th']],
  [mk([2013, 10, 7, 0, 0, 0]), ['the week of october 7th']],
  [mk([2015, 10, 31, 0, 0, 0]), ['last, of october 2015', 'last, in october 2015']],
  [mk([2014, 9, 22, 0, 0, 0]), ['last week of september 2014']],
  [mk([2013, 10, 1, 0, 0, 0]), ['first tuesday of october', 'first tuesday in october']],
  [mk([2014, 9, 16, 0, 0, 0]), ['third tuesday of september 2014']],
  [mk([2014, 10, 1, 0, 0, 0]), ['first wednesday of october 2014']],
  [mk([2014, 10, 8, 0, 0, 0]), ['second wednesday of october 2014']],
  [mk([2015, 1, 13, 0, 0, 0]), ['third tuesday after christmas 2014']],
  [
    mk([2013, 2, 13, 3, 0, 0]),
    [
      'at 3am',
      '3 in the AM',
      'at 3 AM',
      '3 oclock am',
      'at three am',
      'this morning at 3',
      '3 in the morning',
      'at 3 in the morning',
      'early morning @ 3',
    ],
  ],
  [mk([2013, 2, 12, 10, 0, 0]), ['this morning @ 10', 'this morning at 10am']],
  [mk([2013, 2, 13, 3, 18, 0]), ['3:18am', '3:18a', '3h18']],
  [mk([2016, 2, 1, 7, 0, 0]), ['at 7 in 3 years']],
  [
    mk([2013, 2, 12, 15, 0, 0]),
    [
      'at 3pm',
      '@ 3pm',
      '3PM',
      '3pm',
      '3 oclock pm',
      "3 o'clock in the afternoon",
      '3ish pm',
      '3pm approximately',
      'at about 3pm',
      'at 3p',
      'at 3p.',
    ],
  ],
  [mk([2013, 2, 12, 15, 0, 0]), ['15h00', 'at 15h00', '15h', 'at 15h']],
  [
    mk([2013, 2, 12, 15, 15, 0]),
    [
      'at 15 past 3pm',
      'a quarter past 3pm',
      'for a quarter past 3pm',
      '3:15 in the afternoon',
      '15:15',
      '15h15',
      '3:15pm',
      '3:15PM',
      '3:15p',
      'at 3 15',
      '15 minutes past 3pm',
      '15 minutes past 15h',
    ],
  ],
  [
    mk([2013, 2, 12, 15, 20, 0]),
    [
      'at 20 past 3pm',
      '3:20 in the afternoon',
      '3:20 in afternoon',
      'twenty after 3pm',
      '3:20p',
      '15h20',
      'at three twenty',
      '20 minutes past 3pm',
      'this afternoon at 3:20',
      'tonight @ 3:20',
    ],
  ],
  [
    mk([2013, 2, 12, 15, 30, 0]),
    [
      'at half past three pm',
      'half past 3 pm',
      '15:30',
      '15h30',
      '3:30pm',
      '3:30PM',
      '330 p.m.',
      '3:30 p m',
      '3:30',
      'half three',
      '30 minutes past 3 pm',
    ],
  ],
  [
    mk([2013, 2, 12, 12, 15, 0]),
    [
      'at 15 past noon',
      'a quarter past noon',
      'for a quarter past noon',
      '12:15 in the afternoon',
      '12:15',
      '12h15',
      '12:15pm',
      '12:15PM',
      '12:15p',
      'at 12 15',
      '15 minutes past noon',
    ],
  ],
  [mk([2013, 2, 12, 9, 59, 0]), ['nine fifty nine a m']],
  [mk([2013, 2, 12, 15, 23, 24]), ['15:23:24']],
  [mk([2013, 2, 12, 11, 45, 0]), ['a quarter to noon', '11:45am', '11h45', '15 to noon']],
  [
    mk([2013, 2, 12, 13, 15, 0]),
    ['a quarter past 1pm', 'for a quarter past 1pm', '1:15pm', '13h15', '15 minutes from 1pm'],
  ],
  [mk([2013, 2, 12, 14, 15, 0]), ['a quarter past 2pm', 'for a quarter past 2pm']],
  [mk([2013, 2, 12, 20, 15, 0]), ['a quarter past 8pm', 'for a quarter past 8pm']],
  [
    mk([2013, 2, 12, 20, 0, 0]),
    [
      '8 tonight',
      "tonight at 8 o'clock",
      'eight tonight',
      '8 this evening',
      'at 8 in the evening',
      'in the evening at eight',
    ],
  ],
  [mk([2013, 9, 20, 19, 30, 0]), ['at 7:30 PM on Fri, Sep 20', 'at 19h30 on Fri, Sep 20']],
  [mk([2013, 2, 16, 9, 0, 0]), ['at 9am on Saturday', 'Saturday morning at 9']],
  [mk([2013, 2, 16, 9, 0, 0]), ['on Saturday for 9am']],
  [mk([2014, 7, 18, 19, 0, 0]), ['Fri, Jul 18, 2014 07:00 PM', 'Fri, Jul 18, 2014 19h00', 'Fri, Jul 18, 2014 19h']],
  [mk([2013, 2, 12, 4, 30, 1]), ['in a sec', 'one second from now', 'in 1"']],
  [mk([2013, 2, 12, 4, 31, 0]), ['in a minute', 'in one minute', "in 1'"]],
  [
    mk([2013, 2, 12, 4, 32, 0]),
    ['in 2 minutes', 'in 2 more minutes', '2 minutes from now', 'in a couple of minutes', 'in a pair of minutes'],
  ],
  [mk([2013, 2, 12, 4, 33, 0]), ['in three minutes', 'in a few minutes']],
  [mk([2013, 2, 12, 5, 30, 0]), ['in 60 minutes']],
  [mk([2013, 2, 12, 4, 45, 0]), ['in a quarter of an hour', 'in 1/4h', 'in 1/4 h', 'in 1/4 hour']],
  [mk([2013, 2, 12, 5, 0, 0]), ['in half an hour', 'in 1/2h', 'in 1/2 h', 'in 1/2 hour']],
  [mk([2013, 2, 12, 5, 15, 0]), ['in three-quarters of an hour', 'in 3/4h', 'in 3/4 h', 'in 3/4 hour']],
  [mk([2013, 2, 12, 7, 0, 0]), ['in 2.5 hours', 'in 2 and an half hours']],
  [mk([2013, 2, 12, 5, 30, 0]), ['in one hour', 'in 1h']],
  [mk([2013, 2, 12, 6, 30, 0]), ['in a couple hours', 'in a couple of hours']],
  [mk([2013, 2, 12, 7, 30, 0]), ['in a few hours', 'in few hours']],
  [mk([2013, 2, 13, 4, 30, 0]), ['in 24 hours']],
  [mk([2013, 2, 13, 4, 0, 0]), ['in a day', 'a, from now']],
  [mk([2013, 2, 13, 4, 30, 0]), ['a, from right now']],
  [mk([2016, 2, 12, 0, 0, 0]), ['3 years from today']],
  [mk([2013, 3, 1, 0, 0, 0]), ['3 fridays from now', 'three fridays from now']],
  [mk([2013, 2, 24, 0, 0, 0]), ['2 sundays from now', 'two sundays from now']],
  [mk([2013, 3, 12, 0, 0, 0]), ['4 tuesdays from now', 'four tuesdays from now']],
  [mk([2013, 2, 19, 4, 0, 0]), ['in 7 days']],
  [mk([2013, 2, 19, 17, 0, 0]), ['in 7 days at 5pm']],
  [mk([2017, 2, 1, 17, 0, 0]), ['in 4 years at 5pm']],
  [mk([2013, 2, 19, 0, 0, 0]), ['in 1 week', 'in a week']],
  [mk([2013, 2, 12, 5, 0, 0]), ['in about half an hour']],
  [mk([2013, 2, 5, 4, 0, 0]), ['7 days ago']],
  [mk([2013, 1, 29, 4, 0, 0]), ['14 days Ago', 'a fortnight ago']],
  [mk([2013, 2, 5, 0, 0, 0]), ['a week ago', 'one week ago', '1 week ago']],
  [mk([2013, 1, 31, 0, 0, 0]), ['2 thursdays back', '2 thursdays ago']],
  [mk([2013, 1, 22, 0, 0, 0]), ['three weeks ago']],
  [mk([2012, 11, 12, 0, 0, 0]), ['three months ago']],
  [
    mk([2013, 2, 4, 0, 0, 0]),
    [
      'the first Monday of this month',
      'the first Monday of the month',
      'the first Monday in this month',
      'first Monday in the month',
    ],
  ],
  [mk([2011, 2, 1, 0, 0, 0]), ['two years ago']],
  [mk([2013, 2, 19, 4, 0, 0]), ['7 days hence']],
  [mk([2013, 2, 26, 4, 0, 0]), ['14 days hence', 'a fortnight hence']],
  [mk([2013, 2, 19, 0, 0, 0]), ['a week hence', 'one week hence', '1 week hence']],
  [mk([2013, 3, 5, 0, 0, 0]), ['three weeks hence']],
  [mk([2013, 5, 12, 0, 0, 0]), ['three months hence']],
  [mk([2015, 2, 1, 0, 0, 0]), ['two years hence']],
  [mk([2013, 12, 25, 0, 0, 0]), ['one year After christmas', 'a year from Christmas']],
  [mk([2013, 12, 18, 0, 0, 0]), ['for 10 days from 18th Dec', 'from 18th Dec for 10 days', '18th Dec for 10 days']],
  [
    mk([2013, 2, 12, 16, 0, 0]),
    ["for 30' starting from 4pm", 'from 4pm for thirty minutes', '4pm for 30 mins', '16h for 30 mins'],
  ],
  [mk([2013, 6, 21, 0, 0, 0], [2013, 9, 24, 0, 0, 0]), ['this Summer', 'current summer']],
  [mk([2012, 12, 21, 0, 0, 0], [2013, 3, 21, 0, 0, 0]), ['this winter']],
  [mk([2012, 12, 21, 0, 0, 0], [2013, 3, 19, 0, 0, 0]), ['this season', 'current seasons']],
  [mk([2012, 9, 23, 0, 0, 0], [2012, 12, 20, 0, 0, 0]), ['last season', 'past seasons', 'previous seasons']],
  [mk([2013, 3, 20, 0, 0, 0], [2013, 6, 20, 0, 0, 0]), ['next season']],
  [mk([2013, 2, 11, 18, 0, 0], [2013, 2, 12, 0, 0, 0]), ['last night', 'yesterday evening']],
  [mk([2013, 2, 11, 21, 0, 0], [2013, 2, 12, 0, 0, 0]), ['late last night']],
  [mk([2013, 12, 25, 0, 0, 0]), ['Christmas', 'xmas', 'christmas', 'christmas day']],
  [mk([2013, 12, 25, 18, 0, 0]), ['Christmas', 'xmas at 6 pm']],
  [
    mk([2013, 12, 25, 0, 0, 0], [2013, 12, 25, 12, 0, 0]),
    ['Christmas', 'morning of xmas', 'morning of christmas 2013', 'morning of this christmas day'],
  ],
  [mk([2013, 12, 31, 0, 0, 0]), ["New Year's Eve", "new year's eve", 'new years eve']],
  [mk([2014, 1, 1, 0, 0, 0]), ["New Year's Day", "new year's day", 'new years day']],
  [mk([2013, 2, 14, 0, 0, 0]), ["Valentine's Day", "valentine's day", 'valentine day']],
  [mk([2013, 7, 4, 0, 0, 0]), ['4th of July', '4 of july']],
  [mk([2013, 10, 31, 0, 0, 0]), ['Halloween', 'halloween', 'next halloween', 'Halloween 2013']],
  [mk([2013, 11, 29, 0, 0, 0]), ['Black Friday', 'black friday', 'black friday of this year', 'black friday 2013']],
  [mk([2017, 11, 24, 0, 0, 0]), ['Black Friday', 'black friday 2017']],
  [mk([2013, 10, 16, 0, 0, 0]), ["Boss's Day", "boss's day", "boss's", 'boss day', "next boss's day"]],
  [mk([2016, 10, 17, 0, 0, 0]), ["Boss's Day", "boss's, 2016"]],
  [mk([2021, 10, 15, 0, 0, 0]), ["Boss's Day", "boss's, 2021"]],
  [
    mk([2014, 1, 20, 0, 0, 0]),
    [
      "Martin Luther King's Day",
      'MLK day',
      'next Martin Luther King day',
      "next Martin Luther King's day",
      'next Martin Luther Kings day',
      'this MLK day',
    ],
  ],
  [mk([2013, 1, 21, 0, 0, 0]), ["Martin Luther King's Day", 'last MLK Jr. day', 'MLK, 2013']],
  [
    mk([2012, 1, 16, 0, 0, 0]),
    ["Martin Luther King's Day", 'MLK, of last year', 'MLK, 2012', 'Civil Rights, of last year'],
  ],
  [mk([2013, 11, 1, 0, 0, 0]), ['World Vegan Day', 'world vegan day']],
  [mk([2013, 3, 31, 0, 0, 0]), ['Easter Sunday', 'easter', 'easter 2013']],
  [mk([2012, 4, 8, 0, 0, 0]), ['Easter Sunday', 'last easter']],
  [mk([2013, 4, 1, 0, 0, 0]), ['Easter Monday', 'easter mon']],
  [mk([2010, 4, 4, 0, 0, 0]), ['Easter Sunday', 'easter 2010', 'Easter Sunday two thousand ten']],
  [mk([2013, 4, 3, 0, 0, 0]), ['three days after Easter']],
  [mk([2013, 3, 28, 0, 0, 0]), ['Maundy Thursday', 'Maundy Thursday', 'Covenant thu', 'Thu of Mysteries']],
  [mk([2013, 5, 19, 0, 0, 0]), ['Pentecost', 'Pentecost', 'white sunday 2013']],
  [mk([2013, 5, 20, 0, 0, 0]), ['Whit Monday', 'whit monday', 'Monday of the Holy Spirit']],
  [mk([2013, 3, 24, 0, 0, 0]), ['Palm Sunday', 'palm sunday', 'branch sunday 2013']],
  [mk([2013, 5, 26, 0, 0, 0]), ['Trinity Sunday', 'trinity sunday']],
  [mk([2013, 2, 12, 0, 0, 0]), ['Shrove Tuesday', 'pancake, 2013', 'mardi gras']],
  [
    mk([2013, 3, 17, 0, 0, 0]),
    ["St Patrick's Day", "st patrick's, 2013", "st paddy's day", "saint paddy's day", 'saint patricks day'],
  ],
  [mk([2018, 2, 14, 0, 0, 0], [2018, 4, 1, 0, 0, 0]), ['Lent', 'lent 2018']],
  [mk([2018, 4, 8, 0, 0, 0]), ['Orthodox Easter Sunday', 'orthodox easter 2018']],
  [mk([2020, 4, 17, 0, 0, 0]), ['Orthodox Good Friday', 'orthodox good friday 2020', 'orthodox great friday 2020']],
  [mk([2018, 2, 19, 0, 0, 0]), ['Clean Monday', 'clean monday 2018', 'orthodox shrove monday two thousand eighteen']],
  [mk([2018, 3, 31, 0, 0, 0]), ['Lazarus Saturday', 'lazarus saturday 2018']],
  [mk([2018, 2, 19, 0, 0, 0], [2018, 3, 31, 0, 0, 0]), ['Great Lent', 'great fast 2018']],
  [mk([2013, 2, 12, 18, 0, 0], [2013, 2, 13, 0, 0, 0]), ['this evening', 'today evening', 'tonight']],
  [mk([2013, 2, 8, 18, 0, 0], [2013, 2, 11, 0, 0, 0]), ['this past weekend']],
  [mk([2013, 2, 13, 18, 0, 0], [2013, 2, 14, 0, 0, 0]), ['tomorrow evening']],
  [mk([2013, 2, 13, 12, 0, 0], [2013, 2, 13, 14, 0, 0]), ['tomorrow lunch', 'tomorrow at lunch']],
  [mk([2013, 2, 11, 18, 0, 0], [2013, 2, 12, 0, 0, 0]), ['yesterday evening']],
  [mk([2013, 2, 15, 18, 0, 0], [2013, 2, 18, 0, 0, 0]), ['this week-end']],
  [mk([2013, 2, 18, 0, 0, 0], [2013, 2, 18, 12, 0, 0]), ['monday mOrnIng']],
  [
    mk([2013, 2, 18, 0, 0, 0], [2013, 2, 18, 9, 0, 0]),
    ['monday early in the morning', 'monday early morning', 'monday in the early hours of the morning'],
  ],
  [mk([2013, 2, 12, 21, 0, 0], [2013, 2, 13, 0, 0, 0]), ['late tonight', 'late tonite']],
  [
    mk([2013, 2, 15, 0, 0, 0], [2013, 2, 15, 12, 0, 0]),
    ['february the 15th in the morning', '15 of february in the morning', 'morning of the 15th of february'],
  ],
  [mk([2013, 2, 12, 4, 29, 58], [2013, 2, 12, 4, 30, 0]), ['last 2 seconds', 'last two seconds']],
  [mk([2013, 2, 12, 4, 30, 1], [2013, 2, 12, 4, 30, 4]), ['next 3 seconds', 'next three seconds']],
  [mk([2013, 2, 12, 4, 28, 0], [2013, 2, 12, 4, 30, 0]), ['last 2 minutes', 'last two minutes']],
  [mk([2013, 2, 12, 4, 31, 0], [2013, 2, 12, 4, 34, 0]), ['next 3 minutes', 'next three minutes']],
  [mk([2013, 2, 12, 3, 0, 0], [2013, 2, 12, 4, 0, 0]), ['last 1 hour', 'last one hour']],
  [mk([2013, 2, 12, 5, 0, 0], [2013, 2, 12, 8, 0, 0]), ['next 3 hours', 'next three hours']],
  [mk([2013, 2, 10, 0, 0, 0], [2013, 2, 12, 0, 0, 0]), ['last 2 days', 'last two days', 'past 2 days']],
  [mk([2013, 2, 13, 0, 0, 0], [2013, 2, 16, 0, 0, 0]), ['next 3 days', 'next three days']],
  [mk([2013, 2, 13, 0, 0, 0], [2013, 2, 16, 0, 0, 0]), ['next few days']],
  [mk([2013, 1, 28, 0, 0, 0], [2013, 2, 11, 0, 0, 0]), ['last 2 weeks', 'last two weeks', 'past 2 weeks']],
  [mk([2013, 2, 18, 0, 0, 0], [2013, 3, 11, 0, 0, 0]), ['next 3 weeks', 'next three weeks']],
  [mk([2012, 12, 1, 0, 0, 0], [2013, 2, 1, 0, 0, 0]), ['last 2 months', 'last two months']],
  [mk([2013, 3, 1, 0, 0, 0], [2013, 6, 1, 0, 0, 0]), ['next 3 months', 'next three months']],
  [mk([2011, 1, 1, 0, 0, 0], [2013, 1, 1, 0, 0, 0]), ['last 2 years', 'last two years']],
  [mk([2014, 1, 1, 0, 0, 0], [2017, 1, 1, 0, 0, 0]), ['next 3 years', 'next three years']],
  [
    mk([2013, 7, 13, 0, 0, 0], [2013, 7, 16, 0, 0, 0]),
    ['July 13-15', 'July 13 to 15', 'July 13 thru 15', 'July 13 through 15', 'July 13 - July 15'],
  ],
  [
    mk([2013, 7, 13, 0, 0, 0], [2013, 7, 16, 0, 0, 0]),
    [
      'from July 13-15',
      'from 13 to 15 July',
      'from 13th to 15th July',
      'from the 13 to 15 July',
      'from the 13th to 15th July',
      'from the 13th to the 15th July',
      'from the 13 to the 15 July',
    ],
  ],
  [
    mk([2013, 7, 13, 0, 0, 0], [2013, 7, 16, 0, 0, 0]),
    [
      'from 13 to 15 of July',
      'from 13th to 15 of July',
      'from 13 to 15th of July',
      'from 13th to 15th of July',
      'from 13 to the 15 of July',
      'from 13th to the 15 of July',
      'from 13 to the 15th of July',
      'from 13th to the 15th of July',
      'from the 13 to the 15 of July',
      'from the 13th to the 15 of July',
      'from the 13 to the 15th of July',
      'from the 13th to the 15th of July',
    ],
  ],
  [mk([2013, 8, 8, 0, 0, 0], [2013, 8, 13, 0, 0, 0]), ['Aug 8 - Aug 12']],
  [mk([2013, 2, 12, 9, 30, 0], [2013, 2, 12, 11, 1, 0]), ['9:30 - 11:00', '9h30 - 11h00', '9h30 - 11h']],
  [mk([2013, 2, 12, 13, 30, 0], [2013, 2, 12, 15, 1, 0]), ['9:30 - 11:00 CST', '9h30 - 11h00 CST', '9h30 - 11h CST']],
  [
    mk([2013, 2, 12, 13, 0, 0], [2013, 2, 12, 16, 1, 0]),
    ['15:00 GMT - 18:00 GMT', '15h00 GMT - 18h00 GMT', '15h GMT - 18h GMT'],
  ],
  [mk([2015, 3, 28, 17, 0, 0], [2015, 3, 29, 21, 0, 1]), ['2015-03-28 17:00:00/2015-03-29 21:00:00']],
  [
    mk([2013, 2, 14, 9, 30, 0], [2013, 2, 14, 11, 1, 0]),
    [
      'from 9:30 - 11:00 on Thursday',
      'between 9:30 and 11:00 on thursday',
      'between 9h30 and 11h00 on thursday',
      '9:30 - 11:00 on Thursday',
      '9h30 - 11h00 on Thursday',
      'later than 9:30 but before 11:00 on Thursday',
      'Thursday from 9:30 to 11:00',
      'from 9:30 untill 11:00 on thursday',
      'Thursday from 9:30 untill 11:00',
      '9:30 till 11:00 on Thursday',
    ],
  ],
  [mk([2013, 2, 13, 1, 0, 0], [2013, 2, 13, 2, 31, 0]), ['tomorrow in between 1-2:30 ish']],
  [mk([2013, 2, 12, 15, 0, 0], [2013, 2, 12, 17, 0, 0]), ['3-4pm', 'from 3 to 4 in the PM', 'around 3-4pm']],
  [
    mk([2013, 2, 12, 15, 30, 0], [2013, 2, 12, 18, 1, 0]),
    [
      '3:30 to 6 PM',
      '3:30-6 p.m.',
      '3:30-6:00pm',
      '15h30-18h',
      'from 3:30 to six p.m.',
      'from 3:30 to 6:00pm',
      'later than 3:30pm but before 6pm',
      'between 3:30pm and 6 pm',
    ],
  ],
  [mk([2013, 2, 12, 15, 0, 0], [2013, 2, 12, 18, 0, 1]), ['3pm - 6:00:00pm']],
  [mk([2013, 2, 12, 8, 0, 0], [2013, 2, 12, 14, 0, 0]), ['8am - 1pm']],
  [mk([2013, 2, 14, 9, 0, 0], [2013, 2, 14, 12, 0, 0]), ['Thursday from 9a to 11a', 'this Thu 9-11am']],
  [mk([2013, 2, 12, 11, 30, 0], [2013, 2, 12, 13, 31, 0]), ['11:30-1:30']],
  [mk([2013, 9, 21, 13, 30, 0]), ['1:30 PM on Sat, Sep 21']],
  [mk([2013, 2, 12, 4, 30, 0], [2013, 2, 26, 0, 0, 0]), ['Within 2 weeks']],
  [mk([2013, 2, 12, 4, 30, 0], [2013, 2, 12, 14, 0, 0]), ['by 2:00pm']],
  [mk([2013, 2, 12, 4, 30, 0], [2013, 2, 13, 0, 0, 0]), ['by EOD']],
  [
    mk([2013, 2, 12, 4, 30, 0], [2013, 3, 1, 0, 0, 0]),
    ['by EOM', 'by the EOM', 'by end of the month', 'by the end of month'],
  ],
  [
    mk([2013, 2, 21, 0, 0, 0], [2013, 3, 1, 0, 0, 0]),
    ['EOM', 'the EOM', 'at the EOM', 'the end of the month', 'end of the month', 'at the end of month'],
  ],
  [
    mk([2013, 2, 1, 0, 0, 0], [2013, 2, 11, 0, 0, 0]),
    [
      'BOM',
      'the BOM',
      'at the BOM',
      'beginning of the month',
      'the beginning of the month',
      'at the beginning of month',
    ],
  ],
  [mk([2013, 2, 12, 4, 30, 0], [2013, 4, 1, 0, 0, 0]), ['by the end of next month']],
  [mk([2013, 2, 12, 13, 0, 0]), ['4pm CET']],
  [
    mk([2013, 2, 14, 6, 0, 0]),
    [
      'Thursday 8:00 GMT',
      'Thursday 8:00 gmt',
      'Thursday 8h00 GMT',
      'Thursday 8h00 gmt',
      'Thursday 8h GMT',
      'Thursday 8h gmt',
      'Thu at 8 GMT',
      'Thu at 8 gmt',
      'Thursday 9 am BST',
      'Thursday 9 am (BST)',
    ],
  ],
  [
    mk([2013, 2, 14, 14, 0, 0]),
    [
      'Thursday 8:00 PST',
      'Thursday 8:00 pst',
      'Thursday 8h00 PST',
      'Thursday 8h00 pst',
      'Thursday 8h PST',
      'Thursday 8h pst',
      'Thu at 8 am PST',
      'Thu at 8 am pst',
      'Thursday at 9:30pm ist',
    ],
  ],
  [mk([2013, 2, 12, 14, 0, 0]), ['today at 2pm', 'at 2pm', 'this afternoon at 2', 'this evening at 2', 'tonight at 2']],
  [mk([2013, 2, 13, 15, 0, 0]), ['3pm tomorrow']],
  [mk([2013, 2, 12, 5, 30, 0]), ['today in one hour']],
  [mk([2013, 2, 12, 12, 0, 0], [2013, 2, 12, 19, 0, 0]), ['in the afternoon']],
  [mk([2013, 2, 12, 8, 0, 0], [2013, 2, 12, 19, 0, 0]), ['8am until 6']],
  [mk([2013, 2, 12, 13, 30, 0]), ['at 1:30pm', '1:30pm', 'at 13h30', '13h30']],
  [mk([2013, 2, 12, 4, 45, 0]), ['in 15 minutes', "in 15'", 'in 15']],
  [mk([2013, 2, 12, 13, 0, 0], [2013, 2, 12, 17, 0, 0]), ['after lunch']],
  [mk([2013, 2, 12, 15, 0, 0], [2013, 2, 12, 21, 0, 0]), ['after school']],
  [mk([2013, 2, 12, 10, 30, 0]), ['10:30', 'approximately 1030']],
  [mk([2013, 2, 12, 0, 0, 0], [2013, 2, 12, 12, 0, 0]), ['this morning']],
  [mk([2013, 2, 18, 0, 0, 0]), ['next monday']],
  [mk([2013, 2, 12, 12, 0, 0]), ['at 12pm', 'at noon', 'midday', 'the midday', 'mid day']],
  [
    mk([2013, 2, 13, 0, 0, 0]),
    ['at 12am', 'at midnight', 'this morning at 12', 'this evening at 12', 'this afternoon at 12'],
  ],
  [mk([2013, 2, 13, 9, 0, 0]), ['9 tomorrow morning', '9 tomorrow']],
  [mk([2013, 2, 13, 21, 0, 0]), ['9 tomorrow evening']],
  [mk([2013, 3, 1, 0, 0, 0]), ['March', 'in March', 'during March']],
  [
    mk([2013, 2, 13, 17, 0, 0]),
    [
      'tomorrow afternoon at 5',
      'at 5 tomorrow afternoon',
      'at 5pm tomorrow',
      'tomorrow at 5pm',
      'tomorrow evening at 5',
    ],
  ],
  [mk([2013, 2, 13, 12, 0, 0], [2013, 2, 13, 19, 0, 0]), ['tomorrow afternoon', 'tomorrow afternoonish']],
  [mk([2013, 2, 13, 13, 0, 0], [2013, 2, 13, 15, 0, 0]), ['1pm-2pm tomorrow']],
  [mk([2013, 3, 1, 0, 0, 0]), ['on the first', 'the 1st']],
  [mk([2013, 2, 12, 10, 30, 0]), ['at 1030', 'around 1030', 'ten thirty am']],
  [mk([2013, 2, 12, 19, 30, 0]), ['at 730 in the evening', 'seven thirty p.m.']],
  [mk([2013, 2, 13, 1, 50, 0]), ['tomorrow at 150ish']],
  [mk([2013, 2, 12, 23, 0, 0]), ['tonight at 11', 'this evening at 11', 'this afternoon at 11', 'tonight at 11pm']],
  [mk([2013, 10, 7, 0, 0, 0]), ['the closest Monday to Oct 5th']],
  [mk([2013, 9, 30, 0, 0, 0]), ['the second closest Mon to October fifth']],
  [mk([2013, 3, 1, 0, 0, 0], [2013, 3, 11, 0, 0, 0]), ['early March']],
  [mk([2013, 3, 11, 0, 0, 0], [2013, 3, 21, 0, 0, 0]), ['mid March']],
  [mk([2013, 3, 21, 0, 0, 0], [2013, 4, 1, 0, 0, 0]), ['late March']],
  [
    mk([2013, 10, 25, 18, 0, 0], [2013, 10, 28, 0, 0, 0]),
    ['last weekend of October', 'last week-end in October', 'last week end of October'],
  ],
  [mk([2013, 2, 11, 0, 0, 0], [2013, 2, 17, 0, 0, 0]), ['all week']],
  [mk([2013, 2, 12, 0, 0, 0], [2013, 2, 17, 0, 0, 0]), ['rest of the week']],
  [mk([2013, 7, 26, 18, 0, 0], [2013, 7, 29, 0, 0, 0]), ['last wkend of July']],
  [mk([2017, 10, 27, 18, 0, 0], [2017, 10, 30, 0, 0, 0]), ['last weekend of October 2017']],
  [mk([2013, 8, 27, 0, 0, 0], [2013, 8, 30, 0, 0, 0]), ['August 27th - 29th', 'from August 27th - 29th']],
  [mk([2013, 10, 23, 0, 0, 0], [2013, 10, 27, 0, 0, 0]), ['23rd to 26th Oct']],
  [mk([2013, 9, 1, 0, 0, 0], [2013, 9, 9, 0, 0, 0]), ['1-8 september']],
  [mk([2013, 9, 12, 0, 0, 0], [2013, 9, 17, 0, 0, 0]), ['12 to 16 september']],
  [mk([2013, 8, 19, 0, 0, 0], [2013, 8, 22, 0, 0, 0]), ['19th To 21st aug']],
  [mk([2013, 4, 21, 0, 0, 0], [2013, 5, 1, 0, 0, 0]), ['end of April', 'at the end of April']],
  [mk([2014, 1, 1, 0, 0, 0], [2014, 1, 11, 0, 0, 0]), ['beginning of January', 'at the beginning of January']],
  [mk([2012, 9, 1, 0, 0, 0], [2013, 1, 1, 0, 0, 0]), ['end of 2012', 'at the end of 2012']],
  [mk([2017, 1, 1, 0, 0, 0], [2017, 4, 1, 0, 0, 0]), ['beginning of 2017', 'at the beginning of 2017']],
  [
    mk([2013, 1, 1, 0, 0, 0], [2013, 4, 1, 0, 0, 0]),
    ['beginning of year', 'the beginning of the year', 'the BOY', 'BOY'],
  ],
  [
    mk([2013, 2, 12, 4, 30, 0], [2014, 1, 1, 0, 0, 0]),
    ['by EOY', 'by the EOY', 'by end of the year', 'by the end of year'],
  ],
  [
    mk([2013, 9, 1, 0, 0, 0], [2014, 1, 1, 0, 0, 0]),
    ['EOY', 'the EOY', 'at the EOY', 'the end of the year', 'end of the year', 'at the end of year'],
  ],
  [
    mk([2013, 2, 11, 0, 0, 0], [2013, 2, 14, 0, 0, 0]),
    [
      'beginning of this week',
      'beginning of current week',
      'at the beginning of this week',
      'at the beginning of current week',
    ],
  ],
  [mk([2013, 2, 18, 0, 0, 0], [2013, 2, 21, 0, 0, 0]), ['beginning of coming week', 'at the beginning of coming week']],
  [
    mk([2013, 2, 4, 0, 0, 0], [2013, 2, 7, 0, 0, 0]),
    [
      'beginning of last week',
      'beginning of past week',
      'beginning of previous week',
      'at the beginning of last week',
      'at the beginning of past week',
      'at the beginning of previous week',
    ],
  ],
  [
    mk([2013, 2, 18, 0, 0, 0], [2013, 2, 21, 0, 0, 0]),
    [
      'beginning of next week',
      'beginning of the following week',
      'beginning of around next week',
      'at the beginning of next week',
      'at the beginning of the following week',
      'at the beginning of around next week',
    ],
  ],
  [
    mk([2013, 2, 15, 0, 0, 0], [2013, 2, 18, 0, 0, 0]),
    ['end of this week', 'end of current week', 'at the end of this week', 'at the end of current week'],
  ],
  [mk([2013, 2, 22, 0, 0, 0], [2013, 2, 25, 0, 0, 0]), ['end of coming week', 'at the end of coming week']],
  [
    mk([2013, 2, 8, 0, 0, 0], [2013, 2, 11, 0, 0, 0]),
    [
      'end of last week',
      'end of past week',
      'end of previous week',
      'at the end of last week',
      'at the end of past week',
      'at the end of previous week',
    ],
  ],
  [
    mk([2013, 2, 22, 0, 0, 0], [2013, 2, 25, 0, 0, 0]),
    [
      'end of next week',
      'end of the following week',
      'end of around next week',
      'at the end of next week',
      'at the end of the following week',
      'at the end of around next week',
    ],
  ],
  [mk([2014, 1, 31, 0, 0, 0]), ['Chinese New Year', 'chinese new year', "chinese lunar new year's day"]],
  [
    mk([2013, 2, 10, 0, 0, 0]),
    ['Chinese New Year', 'last chinese new year', "last chinese lunar new year's day", 'last chinese new years'],
  ],
  [mk([2018, 2, 16, 0, 0, 0]), ['Chinese New Year', "chinese new year's, 2018"]],
  [mk([2018, 9, 18, 0, 0, 0]), ['Yom Kippur', 'yom kippur 2018']],
  [mk([2018, 9, 30, 0, 0, 0]), ['Shemini Atzeret', 'shemini atzeret 2018']],
  [mk([2018, 10, 1, 0, 0, 0]), ['Simchat Torah', 'simchat torah 2018']],
  [mk([2018, 7, 21, 0, 0, 0]), ["Tisha B'Av", "tisha b'av 2018"]],
  [mk([2018, 4, 18, 0, 0, 0]), ["Yom Ha'atzmaut", 'yom haatzmaut 2018']],
  [mk([2017, 5, 13, 0, 0, 0]), ['Lag BaOmer', "lag b'omer 2017"]],
  [mk([2018, 4, 11, 0, 0, 0]), ['Yom HaShoah', 'Yom Hashoah 2018', 'Holocaust, 2018']],
  [
    mk([2018, 9, 9, 0, 0, 0], [2018, 9, 12, 0, 0, 0]),
    ['Rosh Hashanah', 'rosh hashanah 2018', 'rosh hashana 2018', 'rosh hashanna 2018'],
  ],
  [
    mk([2018, 12, 2, 0, 0, 0], [2018, 12, 10, 0, 0, 0]),
    ['Hanukkah', 'Chanukah 2018', 'hanukah 2018', 'hannukkah 2018'],
  ],
  [mk([2018, 3, 30, 0, 0, 0], [2018, 4, 8, 0, 0, 0]), ['Passover', 'passover 2018']],
  [mk([2018, 9, 23, 0, 0, 0], [2018, 10, 2, 0, 0, 0]), ['Sukkot', 'feast of the ingathering 2018', 'succos 2018']],
  [mk([2018, 5, 19, 0, 0, 0], [2018, 5, 22, 0, 0, 0]), ['Shavuot', 'shavuot 2018']],
  [mk([2017, 11, 30, 0, 0, 0]), ['Mawlid', 'mawlid al-nabawi 2017']],
  [mk([1950, 7, 16, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 1950']],
  [mk([1975, 10, 6, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 1975']],
  [mk([1988, 5, 16, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 1988']],
  [mk([2018, 6, 15, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 2018']],
  [mk([2034, 12, 12, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 2034']],
  [mk([2046, 8, 4, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 2046']],
  [mk([2050, 6, 21, 0, 0, 0]), ['Eid al-Fitr', 'Eid al-Fitr 2050']],
  [
    mk([2018, 8, 21, 0, 0, 0]),
    ['Eid al-Adha', 'Eid al-Adha 2018', 'id ul-adha 2018', 'sacrifice feast 2018', 'Bakr Id 2018'],
  ],
  [
    mk([1980, 10, 19, 0, 0, 0]),
    ['Eid al-Adha', 'Eid al-Adha 1980', 'id ul-adha 1980', 'sacrifice feast 1980', 'Bakr Id 1980'],
  ],
  [
    mk([1966, 4, 1, 0, 0, 0]),
    ['Eid al-Adha', 'Eid al-Adha 1966', 'id ul-adha 1966', 'sacrifice feast 1966', 'Bakr Id 1966'],
  ],
  [
    mk([1974, 1, 3, 0, 0, 0]),
    ['Eid al-Adha', 'Eid al-Adha 1974', 'id ul-adha 1974', 'sacrifice feast 1974', 'Bakr Id 1974'],
  ],
  [mk([2017, 6, 22, 0, 0, 0]), ['Laylat al-Qadr', 'laylat al kadr 2017', 'night of measures 2017']],
  [mk([2018, 6, 11, 0, 0, 0]), ['Laylat al-Qadr', 'laylat al-qadr 2018', 'night of power 2018']],
  [mk([2018, 9, 11, 0, 0, 0]), ['Islamic New Year', 'Islamic New Year 2018', 'Amun Jadid 2018']],
  [mk([2017, 9, 30, 0, 0, 0]), ['Ashura', 'day of Ashura 2017']],
  [mk([2018, 1, 30, 0, 0, 0]), ['Tu BiShvat', 'tu bishvat 2018']],
  [mk([2017, 6, 23, 0, 0, 0]), ["Jumu'atul-Wida", 'Jamat Ul-Vida 2017', "Jumu'atul-Wida 2017"]],
  [mk([2018, 6, 8, 0, 0, 0]), ["Jumu'atul-Wida", 'Jamat Ul-Vida 2018', "Jumu'atul-Wida 2018"]],
  [mk([2018, 4, 13, 0, 0, 0]), ["Isra and Mi'raj", "isra and mi'raj 2018", "the prophet's ascension 2018"]],
  [mk([2019, 4, 3, 0, 0, 0]), ["Isra and Mi'raj", 'the night journey 2019', 'ascension to heaven 2019']],
  [mk([1950, 6, 17, 0, 0, 0], [1950, 7, 16, 0, 0, 0]), ['Ramadan', 'Ramadan 1950']],
  [mk([1977, 8, 15, 0, 0, 0], [1977, 9, 14, 0, 0, 0]), ['Ramadan', 'Ramadan 1977']],
  [mk([2018, 5, 16, 0, 0, 0], [2018, 6, 15, 0, 0, 0]), ['Ramadan', 'Ramadan 2018']],
  [mk([2034, 11, 12, 0, 0, 0], [2034, 12, 12, 0, 0, 0]), ['Ramadan', 'Ramadan 2034']],
  [mk([2046, 7, 5, 0, 0, 0], [2046, 8, 4, 0, 0, 0]), ['Ramadan', 'Ramadan 2046']],
  [mk([2050, 5, 22, 0, 0, 0], [2050, 6, 21, 0, 0, 0]), ['Ramadan', 'Ramadan 2050']],
  [mk([2017, 10, 17, 0, 0, 0]), ['Dhanteras', 'dhanatrayodashi in 2017']],
  [mk([2019, 10, 25, 0, 0, 0]), ['Dhanteras', 'dhanteras 2019']],
  [mk([2019, 10, 26, 0, 0, 0]), ['Naraka Chaturdashi', 'kali chaudas 2019', 'choti diwali two thousand nineteen']],
  [mk([2019, 10, 27, 0, 0, 0]), ['Diwali', 'diwali 2019', 'Deepavali in 2019', 'Lakshmi Puja six years hence']],
  [mk([2019, 10, 29, 0, 0, 0]), ['Bhai Dooj', 'bhai dooj 2019']],
  [mk([2019, 11, 2, 0, 0, 0]), ['Chhath', 'chhath 2019', 'dala puja 2019', 'Surya Shashthi in 2019']],
  [mk([2021, 10, 12, 0, 0, 0]), ['Maha Saptami', 'Maha Saptami 2021']],
  [mk([2018, 10, 18, 0, 0, 0]), ['Vijayadashami', 'Dussehra 2018', 'vijayadashami in five years']],
  [mk([2018, 10, 9, 0, 0, 0], [2018, 10, 19, 0, 0, 0]), ['Navaratri', 'navaratri 2018', 'durga puja in 2018']],
  [mk([2018, 10, 27, 0, 0, 0]), ['Karva Chauth', 'karva chauth 2018', 'karva chauth in 2018']],
  [mk([2018, 7, 14, 0, 0, 0]), ['Ratha-Yatra', 'ratha-yatra 2018']],
  [mk([2018, 8, 26, 0, 0, 0]), ['Raksha Bandhan', 'rakhi 2018']],
  [mk([2020, 4, 6, 0, 0, 0]), ['Mahavir Jayanti', 'mahavir jayanti 2020', 'mahaveer janma kalyanak 2020']],
  [mk([2020, 2, 21, 0, 0, 0]), ['Maha Shivaratri', 'maha shivaratri 2020']],
  [mk([2018, 2, 10, 0, 0, 0]), ['Dayananda Saraswati Jayanti', 'saraswati jayanti 2018']],
  [mk([2018, 1, 14, 0, 0, 0]), ['Thai Pongal', 'pongal 2018', 'makara sankranthi 2018']],
  [mk([2018, 1, 13, 0, 0, 0]), ['Boghi', 'bogi pandigai 2018']],
  [mk([2018, 1, 15, 0, 0, 0]), ['Mattu Pongal', 'maattu pongal 2018']],
  [mk([2018, 1, 16, 0, 0, 0]), ['Kaanum Pongal', 'kaanum pongal 2018', 'kanni pongal 2018']],
  [mk([2019, 1, 15, 0, 0, 0]), ['Thai Pongal', 'makar sankranti 2019', 'maghi in 2019']],
  [mk([2018, 4, 14, 0, 0, 0]), ['Vaisakhi', 'Vaisakhi 2018', 'baisakhi in 2018', 'Vasakhi 2018', 'vaishakhi 2018']],
  [mk([2018, 8, 24, 0, 0, 0]), ['Thiru Onam', 'onam 2018', 'Thiru Onam 2018', 'Thiruvonam 2018']],
  [mk([2019, 2, 10, 0, 0, 0]), ['Vasant Panchami', 'vasant panchami in 2019', 'basant panchami 2019']],
  [mk([2019, 3, 20, 0, 0, 0]), ['Holika Dahan', 'chhoti holi 2019', 'holika dahan 2019', 'kamudu pyre 2019']],
  [mk([2019, 8, 23, 0, 0, 0]), ['Krishna Janmashtami', 'krishna janmashtami 2019', 'gokulashtami 2019']],
  [mk([2019, 3, 21, 0, 0, 0]), ['Holi', 'holi 2019', 'dhulandi 2019', 'phagwah 2019']],
  [mk([2018, 8, 17, 0, 0, 0]), ['Parsi New Year', 'Parsi New Year 2018', 'Jamshedi Navroz 2018']],
  [mk([2022, 8, 16, 0, 0, 0]), ['Parsi New Year', 'jamshedi Navroz 2022', 'parsi new year 2022']],
  [
    mk([2013, 4, 26, 0, 0, 0], [2013, 4, 29, 0, 0, 0]),
    ['Global Youth Service Day', 'GYSD 2013', 'global youth service day'],
  ],
  [mk([2013, 5, 24, 0, 0, 0]), ['Vesak', 'vesak', 'vaisakha', 'Buddha day', 'Buddha Purnima']],
  [mk([2013, 3, 23, 20, 30, 0], [2013, 3, 23, 21, 31, 0]), ['Earth Hour', 'earth hour']],
  [mk([2016, 3, 19, 20, 30, 0], [2016, 3, 19, 21, 31, 0]), ['Earth Hour', 'earth hour 2016']],
  [mk([2013, 2, 23, 0, 0, 0]), ['Purim', 'purim']],
  [mk([2013, 2, 24, 0, 0, 0]), ['Shushan Purim', 'Shushan Purim']],
  [
    mk([2014, 1, 7, 0, 0, 0]),
    [
      'Guru Gobind Singh Jayanti',
      'guru gobind singh birthday',
      'guru gobind singh jayanti 2014',
      'guru gobind singh jayanti',
      'Guru Govind Singh Jayanti',
    ],
  ],
  [mk([2018, 4, 27, 0, 0, 0]), ["King's Day", 'Koningsdag 2018', 'koningsdag 2018', "king's, 2018", "King's, 2018"]],
  [mk([2014, 4, 26, 0, 0, 0]), ["King's Day", 'Koningsdag 2014', 'koningsdag 2014', "King's, 2014", "king's, 2014"]],
  [
    mk([2018, 5, 9, 0, 0, 0]),
    ['Rabindra Jayanti', 'rabindra jayanti 2018', 'Rabindranath Jayanti 2018', 'Rabindra Jayanti 2018'],
  ],
  [
    mk([2019, 5, 9, 0, 0, 0]),
    ['Rabindra Jayanti', 'rabindra jayanti 2019', 'Rabindranath Jayanti 2019', 'Rabindra Jayanti 2019'],
  ],
  [
    mk([2018, 1, 31, 0, 0, 0]),
    ['Guru Ravidass Jayanti', 'guru Ravidas jayanti 2018', 'Guru Ravidass birthday 2018', 'guru ravidass Jayanti 2018'],
  ],
  [
    mk([2019, 2, 19, 0, 0, 0]),
    ['Guru Ravidass Jayanti', 'Guru Ravidass Jayanti 2019', 'Guru Ravidas Birthday 2019', 'guru ravidas jayanti 2019'],
  ],
  [mk([2019, 10, 13, 0, 0, 0]), ['Pargat Diwas', 'valmiki jayanti 2019', 'Valmiki Jayanti 2019', 'pargat diwas 2019']],
  [
    mk([2018, 10, 24, 0, 0, 0]),
    ['Pargat Diwas', 'maharishi valmiki jayanti 2018', 'pargat diwas 2018', 'Pargat Diwas 2018'],
  ],
  [mk([2019, 9, 2, 0, 0, 0]), ['Ganesh Chaturthi', 'ganesh chaturthi 2019']],
  [mk([2020, 4, 2, 0, 0, 0]), ['Rama Navami', 'rama navami 2020']],
  [
    mk([2018, 3, 18, 0, 0, 0]),
    [
      'Ugadi',
      'Ugadi 2018',
      'ugadi 2018',
      'yugadi 2018',
      'Yugadi 2018',
      'samvatsaradi 2018',
      'chaitra sukladi 2018',
      'chaitra sukhladi 2018',
    ],
  ],
  [mk([2012, 12, 25, 0, 0, 0]), ['Christmas', 'the closest xmas to today']],
  [mk([2013, 12, 25, 0, 0, 0]), ['Christmas', 'the second closest xmas to today']],
  [mk([2011, 12, 25, 0, 0, 0]), ['Christmas', 'the 3rd closest xmas to today']],
  [mk([2013, 10, 25, 0, 0, 0]), ['last friday of october', 'last friday in october']],
  [
    mk([2013, 2, 25, 0, 0, 0]),
    [
      'upcoming two weeks',
      'upcoming two week',
      'upcoming 2 weeks',
      'upcoming 2 week',
      'two upcoming weeks',
      'two upcoming week',
      '2 upcoming weeks',
      '2 upcoming week',
    ],
  ],
  [
    mk([2013, 2, 14, 0, 0, 0]),
    [
      'upcoming two days',
      'upcoming two day',
      'upcoming 2 days',
      'upcoming 2 day',
      'two upcoming days',
      'two upcoming day',
      '2 upcoming days',
      '2 upcoming day',
    ],
  ],
  [
    mk([2013, 4, 1, 0, 0, 0]),
    [
      'upcoming two months',
      'upcoming two month',
      'upcoming 2 months',
      'upcoming 2 month',
      'two upcoming months',
      'two upcoming month',
      '2 upcoming months',
      '2 upcoming month',
    ],
  ],
  [
    mk([2013, 7, 1, 0, 0, 0]),
    [
      'upcoming two quarters',
      'upcoming two quarter',
      'upcoming 2 quarters',
      'upcoming 2 quarter',
      'two upcoming quarters',
      'two upcoming quarter',
      '2 upcoming quarters',
      '2 upcoming quarter',
    ],
  ],
  [
    mk([2015, 1, 1, 0, 0, 0]),
    [
      'upcoming two years',
      'upcoming two year',
      'upcoming 2 years',
      'upcoming 2 year',
      'two upcoming years',
      'two upcoming year',
      '2 upcoming years',
      '2 upcoming year',
    ],
  ],
  [mk([2013, 2, 13, 13, 40, 0]), ['20 minutes to 2pm tomorrow']],
  [mk([2013, 1, 7, 0, 0, 0]), ['first monday of last month']],
  [mk([2013, 1, 1, 0, 0, 0]), ['first tuesday of last month']],
  [mk([2013, 1, 14, 0, 0, 0]), ['second monday of last month']],
  [mk([2013, 2, 23, 0, 0, 0]), ['next saturday']],
  [mk([2013, 2, 18, 0, 0, 0]), ['next monday']],
]

test('duckling-tests', (t) => {
  arr.forEach((a) => {
    a[1].forEach((str) => {
      let context = {
        today: [2013, 2, 12],
      }
      let found = nlp(str).dates(context).json()[0] || {}
      found.dates = found.dates || {}
      t.ok(fmt(found.dates.start), str)
      // t.equal(fmt(found.date.start), a[0], str)
    })
  })
  t.end()
})
