/**
* pattern_en_packed
*
* Variable containing polarity, subjectivity, and intensity scores for selected opinion-based words in English, compressed with efrt 2.7.0
*
* @author Scott Cram
*
* Original version:
* @author Steven Loria
* @see {@link https://github.com/sloria/TextBlob/blob/dev/textblob/en/en-sentiment.xml|en-sentiment.xml} 
* 
* From xml file above, multiple occurences of the same word were replaced with single instance, and values were averaged to four decimal places.
* Resulting data was trimmed so as to only contain polarity, subjectivity and intensity scores.
* All apostrophes removed.
* Added all words from negation array below (with dummy values same as "wont", since they won't be used.)
* Converted to JSON object.
* Format for entries: "word": ["i1.0", "p0.0", "s0.1"]
* i denotes intensity score
* p denotes polarity score
* s denotes subjectivity score
* 
* Alternative versions:
* @see {@link https://github.com/NaturalNode/natural/blob/master/lib/natural/sentiment/English/pattern-sentiment-en.json|pattern-sentiment-en.json (Another English language version)}
* @see {@link https://github.com/NaturalNode/natural/blob/master/lib/natural/sentiment/Dutch/pattern-sentiment-nl.json|pattern-sentiment-nl.json (Dutch language)}
* @see {@link https://github.com/NaturalNode/natural/blob/master/lib/natural/sentiment/French/pattern-sentiment-fr.json|pattern-sentiment-fr.json (French language)}
* @see {@link https://github.com/NaturalNode/natural/blob/master/lib/natural/sentiment/Italian/pattern-sentiment-it.json|pattern-sentiment-it.json (Italian language)}
*/
export default "i1.0¦0:N2;1:MG;2:N5;3:MX;4:M7;5:MQ;6:MP;7:M3;8:MY;9:KB;A:M1;B:JE;C:LP;D:L2;E:IX;F:II;G:KX;H:JO;I:IO;J:MW;K:LC;L:L0;aJTbIOcFQdE7eD0fBNgAShA7i8Wj8Rk8Pl7Xm6Sn6Io63p4Zqu4Vr46s22t1Hu0Fv09wPyM;aNellM4oungM;!er,iD;aawwnnnn,rn;a00eWhUiRoMro3tf;manJ0nPodKHrMu3Jw;kmanli89sNthM;!leJLwDFy;e,t;deCDky,t;de,lNnMse,tI;!ni3s;d,li3;addupwitdM2iMo6;ms9te;aOiN0lMre2st2Nt;coJLl-M;advis0intLVo3V;k,ltFB;cGSnPrOsM;nt,teM;!d,s;m,y;!ts;aPeterCiNoDDulM;gC5nG0;bCCcMoFLsu1t1vLL;io7tim;cu5JgA7pMrGst;id,orif5;gl0LltiK8nPppErOsMttE;eMu1;d LZfBleIW;bCinEK;a0Fb0Cc09d08e05f01grL5hYiVknown,likeI7nUorigF5pPreaOsNtMusu1w0;hink4race4;alt0chool0ettK8tirr0;d,liH;aL9lPrM;eNoM;ceDKpKD;cedeLZdi8B;ay4easa2;ec5Hotic0;l5Ompor8NnMq9Rvers1;spir0teM;lliFArrupt0;aNeM;altEHsiDP;mpG5pGG;aiOoMruitfB;cJErM;gett4tunateG5;r,t04;ngaGNvJ4xM;cell0pM;eUlHQ;eserv0ign78;ared-f68h85ivil,oMrit9ut;mMntroEWok0;fFKmGR;eNlMraH6;emLHinH8;fitKliHT;b6dultBPffeMnswFQppeJEsham0vow0waIA;ct0;iFLyM;!-duckJD;a05e01hXiVoSrNwMyp9;ist0o-dimen4I;aPeOiDKoub1AuM;e,tM;hfB;men8Hn9C;diESg5pp0;il1UneFpNt1uM;cIZgh;!-not9Uic1;dy,gGRny,reMt3U;d,soHU;anks,eOiNoughtMrillJ;-provoGPfB;ck,n,rd;atr9mHJoret9;chn9dGen1YnOrM;min3HrM;est5Di7X;!se;leKOme,steF;a1Mc1Je19h14i0Yk0Wl0Tm0Qo0Jp0DtYuNwMympathI6;e1Fi3N;bVcRddI3ffQit4pNrMspensH3;e,prisJre1;erNportiM;ng,ve;!b,fi8Xi57natD9;e3QocaK;ceNh,kM;erBJs;e2XssM;!fB;conscGjeC0norm1seqJOt6urbC;aWeTiRrOuMyliD;mb6nEWpidM;!iI;aMetCOiG1o3utK;ightMnH6;!forGZ;ff,nkM;er,s;adNll9SreotypM;ed,ic1;faJ3y;bBHinFle,ndaKArNtM;e-suppo57ic;k,tM;i3li3;aniDeOiritNlendJ2oMrightFW;ntanD2of;ed,u1;cMnt;iMtac2N;al,f5;bEci1ftRlPphOrLuM;nd,rHWthM;ern;ist9Hom28;e,iM;c91d;!-boBH;aNiMooAW;le6W;ll6Xrt;iNoM;pE3w;ck,gF6p7A;ept9iM;ll0ttiD;ckQgnifica2lPmNnMxth-graFH;ceG7g6istEks;il8YplM;e,iH;e2ly;!eDS;aPeEit,oNriMy;eDAll;ckJd7Brt,uMweL;ld2;dy,ky,llI1m,peFrp;aUcRizurBDlfQnOrNv2SxM;u1y;io7mE8;sMtim93;aCLiA;-acKiD;!ondNrMuFR;et;!aLhaHH;mFsI0;aMum;rMtGN;ey,y;dCPfe,me,rcaHtisM;fMyfi3;i0yi3;a07eTiQoNuM;de,iH5r1sG6thF;b0Afl,hypnol,maFPse,uM;ghMnd;!aFF;ch,dicu88ghtNp-oMsk-free,veK;ff;!-E2iHF;aYcXdWfresGAgUhaDlTmSpQsNtardMwarFS;!ed,s;ourcERpM;ectMoC2;ab6fBiIR;eMut4;lBCtiA;ark4inis2Wo8;atHWe3YigG;rets,uM;l7RrgitAE;!eeGWoubt4uI9;e2ogniz4;dy,liHsS;nNre,uMw;co7ncAL;coHEdom,k;estiOiM;ck,et,rC0xM;ot5;on4;a0Ie0Dh0Bi08l03oXrOsychoNuM;bl5re,taA;lFFt5;eRiQoM;fOlix,miNpMud,vA0;aganda,er;ne2si3;es0Ui1EouG9;ceFmaLor,sBSva8;acA6ciNdi3TgTse2tMvG;e2Sty;o7se;et5iPlOor,p0AsNtentMwerCR;!i1;iAsHH;ar,it9;gMntF;na2;aPeNodM;!di3;asMonaH;a2ed;cGBin,titud9IusH9;nMo7ty,vot1;heaMk;ds;antasmagMenomen1ilosoph80ys9;or5;aPeviDppeLrM;fe8MpMsAO;etuMlex0;alCW;cDCky;infBlRrNsMthEC;sioFDt;aOtiM;al,cM;ul6F;de,llel;e,p4;bYccaXdd,ffVkUld43nCNoz90pSrRthEutOverMwn;-the-top,aMbGKexcit0whelFH;ll;dF8rageNsM;iCPt5F;d,o7;di99ig9Fthodox;enC4po6ItimM;um;!ay;beFNeM;rs;siA1;ed8KjecAsMvG;e7Stacl8M;aSeQiGNoNumM;b,eFP;!b6nvio9CrNstalg5tM;!ab6;m1wegiC;ar,cMedFgaAithErve-racBWt,vEw,xt;essaL;iGKk0meFrrF1sIturalNuM;ghIseEO;!iH;a0Ee05iZoRuMysterG;ch,g4OltiNnda4CsMzak;ic1t2;lMp6;atM;er1;dQnPrNst,tl42uth-waM;te6G;alMe,on6U;!iDR;k3Yosyllab5;!eM;rMst;a8n;dQghIlPnNsMx0;er4fiCGplac0si3;dMim1or,us;-boggDKleC4;d,i9D;!d6;aSdiPlodrama,m7HnOrNsMtaphor9xDP;meriDFs4H;ciFe;a8Lt1;caNev1ocrM;e,iI;l,tiFR;gEnMsB7;!ingAI;d,gVin,jUlSnPrOsNtM;hemat9uC1;culi3IsiFMte4K;k0ri0ti1ve4V;nerB1oNq2VyM;!-sid0;ri1;adroit,eM;!vo83;or;ic9EnifiM;ce2;a09e05iVmao,oNuMyr5;c8Ysh;c1g9lSnQo9HsePuOvMw,y1;ab6eMi3;!d,ly;d,sy;rs,s;eAMgM;!-wiAB;!ol;ab6ceUfelSghtQkOmNnguiHtMve8Z;eCAt6;it0p;ab6eM;d,ly;!-heaM;rt0;iMo3;ke;ntG;aOftNgMn6Jss1P;al,endaLib6;!iD9;dBNst;me,rge1Xst67tNughMwfBzy;!ab6ed;eMtE;!r,st;ey,iM;ll0nd;aOewiDoy,uM;stMveni6;ifi0;ckassMil,mm0pane8M;!es;c0Vd0Rll0Qm0HnPrMs2tB2;iDon5rM;eleMi5O;va2; 0Ba0Ac06d02eYfWgenGhuma25i45nUsStMvenA;eMiBMrigui3;llOn8ErM;e57naM;l,ti7C;ectu1igentM;!sia;a1YecuAGpirMta2ulting7Z;a77i3;er,oM;ce2vaA;aMlexDJuriaK;mo7nti6tuC2;luOvQxpM;eMlic4reDF;d5LrieCV;ct4;eOiNomM;it4;e,spens4vidu1;ciph6Qpende2;alc8MoMredD9ur4;heDOmpNnMrruX;sistenci5Gven5D;ar4e6L;ppo33rticu90uspicG;good tMsto8W;as8;aginaAbeci6itati88mTpM;aRePlicOoMressCP;rMssD0;ta2;at0it 7X;cc4rceM;ptCW;s31t51;ane2en7G;!eg1;eOioM;cy,tM;!ic,s;al,ntifi4;ky,on5y;aVeTiSoOuMyster9;ge,mM;an,b6oBZ;llBGnestOrriMt;b6fM;ic,yi3;!-to-god;dd9TghSlarGnd6Pstor3Bt-and-mi8Z;aMro5;lt4YrtfeBVvy;d2haTlf,ndSpQrOs2teNve2zarM;do7;!d;dMsh;!er;hazaCEle8RpM;i6Dy;-held,so8Yy;!haM;!haM;!ha;a0Ee0Bi0Al07o02rNuM;ard0ilI;aYeUiRoPrNuM;d6Veso8R;!rM;!r;ss,tesqM;ue;eNm,pMtI;pi3;f,vo7;atNeMy;k,n;!eM;r,st;ndMph5tu0Z;!iloqAU;dforsak8Yld8YoNrM;g4Ny;dMfy;!y-gooM;dy;ad,oNuM;ey;b1om;a2ft0mmic5F;nMrmCtt4;er6Ct6uiM;ne;meMrgantuCw5By;!changE;*c6Wa0Le0Ii0Al05o00rSuM;ckQllOnNrtMtu7Z;hEiBK;!ny;! of life,-M;bodi0fleBFleng2B;!ed,i3;agi6ePiMostbitt8DustratJ;end6VgNnM;gy;h1Eid;eNnMqA4sh;ch;!-thin6JstM;an85;llowi3rMur21; su7KcOeign,getNmMtu97;er,ula5;fBt4;ed,ib6;aOippa2uMy;ffMid;!y;s34t,wM;ed,le72;c43endiDftie1RlSnPrNtMx0;!ti3;m,stM;!-st10;aNeM;!-loo62;l,n7U;l0t2V;aNeb6licMma6veriDw;ito7;rfB;bVci1iTke,l4TmRnQrOsNtIultFvorM;ed,i8;cinaKt;!-out,cMthermo92;e,ic1;at5taH;iliMo7;ar;lMnt,r;!ed,s,u6R;l0rNuM;lo7;ic8F;a0Qc0Nd0Lerie,ff0KgoiHl0Im0Gn08p5qu1r06ssen05th9uropeCv03xM;a00cXhWot5pSquiRtNubeM;ra2;eOin1BrM;aMe6I;!ordi2D;nUrn1;si8;eMlo6OresS;ct0nRriM;e90mM;ent1;au0YilaraK;eMitJu43;l2Ip31sM;si9U;ct,ggM;er7X;eryday,iM;de2l;ti1;ot5ron2CstwMudi8;hi6;dRerg6LgQigm5SjoyPlighOoNtM;ertai3OhusiaHi5Z;r64ugh;te3M;!ab6ed,i3;a3YliDros77;eaMle5K;ri3;o2KpM;ir9ty;abo8UeM;ct,ga2m2O;ecAi3;gy,uM;ca2E;centr5oM;l6DnomM;ic35;rlMsy;iEy;a11e0DiXoSrQuMynam5;dsvil6e,hNllMmb,sIuuh;!s;!hhM;!h;aMeadfBi0own0unk,y;g,m52;cum28es2meHnOubNwM;dy,n;le,tfB;e wiMt;th;a00dZffXgit1mWrUsMurn1;aRbelie06court1De0LguQhone78lik0possePtM;aNinTraug3XurM;bi3;nt,st4K;ss0;stJ;bl0ppointMst7I;ed,i3me2;eMty;ct;!-witt0;e8FiM;cu7Fde2;act5nt;lect1pha3A;ad08bau07c05ep,fZlVnomina1GpSsPtaOvMxtr1;asMelop0o6W;taK;il0;erNpe7OtrM;oyOucA;vi3;lNressM;!i3;or4;iMuxe;cNghtM;ed,fB;a8io7;eOiMt;cMni8;ie2;cMn2G;atM;es;e2reM;as0;ch0;!ly,pC;iNnge6Lrk,zM;ed,z55;ly,nI;a27e21h1Li1Kl17o01rTuMyn9;lPn1LrOsNtM;e,ti3;hy;io7re2si7F;iNtM;ur1;naL;aSeRiPoOuMyi3;ci1dMel,shJ;dy,e;ok0ss;mMsp,t9;in1;aAd6Qe1A;fIp,zy;ar1Ac0Qhe74l0Om0BnSol,rNsmopolitCuMw,zy;ld2ntFrtQ;iacPpNruptM;!ib6;o6HuM;le2;eo7;c00fYsUtPvM;eNinM;ci3;nZx;ePinOrM;iv0oM;versi1;ge2;mpo3Xst4;cGecr4SiMta2um41;dNsM;te2;er4;iMusJ;de2rm0;avo-convex,eMi0Ore8;iv4pM;tiM;on1;a,e-at-4fXic0DmVpM;eTlNreheM;ns5W;ainQePiM;c4EmM;enM;taL;te,x;ed,t;l3Ote2;ercialMon;!ism;ort4;d,lect5LoM;rfBss1;ky;aVeSoPuM;elessNmM;sy;ne1Y;s0udM;-covMle1W;er0;aMvE;nMr;!ly;ssMustrophob5;icMy;!al;nem1Qviliz0;aXeTiPopOrNurM;ni3;istiCon2N;py;ck2DlNneMtch46;se;diDlM;i3y;ap,eM;rNsM;ie3Sy;fBy;llenOngeFrM;acteriHiMmi3;sm1Bt4;gi3;aQlebr38ntOrM;emoni1taM;in;er,rM;al,ic;seF;cophoWlUnTpRrPsNtMuH;c2Bh36;t-irMu1;on;diac,eM;fBle0W;ab6tivM;aKe;d3Ct;cMm;ul4;no7;a0Ie0Ei0Cl03oZrNuM;d1Hsy;aUeathtaTiRoNuM;sh0t1;adMk1F;!-M;miM;nd0;ef,gMllia2tiD;ht;ki3;insiNsh,vM;ado,e;ck;diOgg0ilerpNld,nny,otleg,rJundF;le0A;la8;ly;aSeRi29oMue;nPodM;stNthirsIy;ty;ain0;de;ak,ch;ck,nMst0ta2;d,k;gMograph5ttEzar08;!gE;autifBco24efy,hi1YlNst,ttEwitc17;er;iMov0;ev4;ck,dRlmy,nQrOsNttlM;efB;e,ic,s;barMe;iCo7;al,d0g-up;!neM;ss;b34c2Qd2Aeriform,f20g1Whw,i1Tl1Gm15n12p0Qr0Gs0At06uXvVwNxiomM;at5;aSeQfBkOwM;!wM;!w;wa32;ul;aLsoM;me;re;ail4eraMid;ge;rea8strTtM;hPiHoM;biograph9noM;mo7;st5;eNorM;itaA;nt5;aliC;mospher5rocGtMyp9;enMracA;da2tiM;on-getKve;cQhPiCkew,sumpAtM;oMu8;nis04unM;di3;en;et5;bitUchaeRduo7e2oQre14tM;eOiM;fiMst5;ci1;siC;us0;olM;og9;ic1;raL;hon5pMt;aVeSlRoQrM;eciat1IoM;acNpria8xiM;ma8;hi3;r0Usi8;aud4icaA;aNtiM;zi3;li3;llJre2;gMnoyJxG;erMry;!ed;aRbPeNuM;si3;n4rM;icC;itG;io7;tMzi3;eurNoL;ry;!iD;sh;aWcohVg08iSlNterM;na8;-arouPeNusioM;ns;g0viM;at0;nd;enMve;!aK;ti3;ol5;rMs;mi3;nt,rM;ed,heM;ad0;ed,haOiNlM;ow;le,taA;st;fTloSoremPrM;aNicC;an;id;enM;tiM;on0;at;ab6irmaAlM;ue2;ama2dZeWjectiv1ministr4orUuTvM;aReM;ntuOrM;saAte2;ti0N;ro7;us;nc0;lt;ab6i3;ab6;pt,quateM;! M;to;ictMl0;ed,i0C;adem5cSquaiRross-the-bQtNuM;a8te;iNu1;al;ng,on,ve;oaX;nt0;eOomplNuM;ra8;ish0;ssM;ib6;le;ic;horYle,oXrUsNuM;nda2;enRoNuM;rd;lu8rbJ;ed,i3;ng;te;ce;iMupt;dg0;ed;ve;re2;nt|p0.0¦0:3I;1:2U;2:35;3:3K;4:3A;5:3E;6:3B;7:2W;8:2A;9:3J;a36b2Tc28d23e1Tf1Jg1Gh1Ei17j16key,l0Zm0Pn0No0Kp0Dqui0Cr06sQtJuFvDwByA;el3Foung1;estXhi3DoA;man2UoP;aAeter4isu0;porSri7st;ltima39nApp1rb4tt1;aBcr0AiAw6;nterrupt6vers0;vow6wa1S;eDhBiny,oAradi2B;p2t0;eAiJ;atr2m2Uoret2;chn2enBn,rrA;est0Jif3;!age;ame,eNhMiKmall1oHpFtaDuA;bBch,dAspense0L;den;consci7seque8urb4;ndaArK;rd;an29ecAiritu0;if3;lButhA;ern;e,id;le8mA;iIp5;e1ort;cond,lf-acAver0;ti9;eBomant3uA;r0ssi4;al1Yce8d,guClAminisce8spe2P;atAigi7;ed,i2T;lar;et,rky;aFers1IhEi7oCrAsych29ubl3;ese8iAop1;or,va29;igna8lAssib5t0R;it2;antasmagor3iloso20ys2;lp2Crallel;bBccasi1Bn1MpAutsi1Overall;en,posi23;je2Bvi7;ak6eAorwegi4umer7;cess0Wt,xt;aGeEiCoBuAysteri7;ltip5s2;de0Yr0;dAx6;!d5;diAtaphor2x1X;c0ev0;le,nAr0Cs09themat2;oAy-sid6;ri0;aCeBike17oA;c0w;ft,ss1;rg1stCtBwA;ful;er,t1;!i9;apa0Tew0Z;mFnAr0Yt1C;calcul1NdBevCi00n1s0JternaA;l,ti0M;epende8iBomA;it1K;e,vidu0;men0Nplicit in;and-held,eartfelt,istoricAum4;!al;eBia8lob0rA;a0Teek,itty,o07;n13rm4;aIema5iGluid,oErBuA;rth1tuQ;agi5eA;eAnch;-thinki9standi9;llowi9rmAurth;er,ula3;c04naA;l,n05;bricat6ci0rthermost;arli1ccentr3mo01nGqu0rstwhi5sDurope4xtA;eAra;nArn0;si17;pecial09sA;enA;ti0;gl05oBtiA;re;rm7ugh;ai03eCiBoAynam3;cumentGmest3ub5;ffere8git0urn0;ep,finiteAnominaOxtr0;!ly;athol3entrThPinem08lass2oFrDuA;linBnni9rA;re8si0T;ary;it2oAuK;ok6ss;arMmmerInCrpoBsmopolit4untleA;ss;ra08;cCsA;ta8;nt;avo-convex,epA;tiA;on0;ci0;iBrA;isti4onX;neA;se;al,ic;aKiGlEodiDrA;iAoad-mind6ush6;ef,tA;ish;ly;ank,onAue;de;gBograA;ph3;!g1;er;ck,nd6rbar7s3;ous;boXcSdOfKgJlternaImerLpproachi9rchaeGsi4tEuBxiomA;at3;strBtobiograA;ph2;ali4;mosphAyp2;er3;olog2;ic0;te;low;loat,oremention6rA;ic4;an;ed;diCministrA;ab5;le;ctiF;adem3tA;i9u0;al;ng;ic;ve|s0.1¦a0Cb08c00dXeTfPgOhNiLlJmDobjCp8respCs4t2un1w0;ants,orR;affect0Fco00origin0HprocXw0F;e0heoretTitular,wo-dimen8;chnSrrestE;e2ileHlippLmiQo1peHt0ucceeds;artKumbP;lKur0A;amXlf-actI;hantasmagor05olitNro0sychologN;fes1p0;aganda,er;sion07;ec2;a2edica1i0;lita00nus;ti04;le,n0ture;o0y-sidZ;ri00;augh,i0oses;nguiDteraU;cy,mplicit in,s0;nt;inderTumF;ettab6rey;ew,iftie2lu1ollow0reestand0;ing;id;th;mpir2rstwhi1xt0;ernOra;le;icM;extrLisposs1ome0rownJulls;stF;essH;a6ent5lo3osmopolit2r0ulturI;o0u7;okE;an;sCud0;less;er,r7;nt,thol6;a2ig,lue,o1ru0;sh7;dily,gg6;nd5re;bo7ct5djectiv6float,ll3mato2p0;hon0preciat3;ic;ry;eg0usions;ed;ion,u0;al;ve|s0.0¦0:1O;a1Gb1Cc10d0Re0Qf0Mg0Hh0Di09j07kill1Tl03mWnUoTpRrMsFtAu6v4w2y1;ellow,ou0J;aste1estHhi1Nooden;!d,s;a1eter1Oisu0;cuum,porif1R;n2pp0Fr1seZ;b1Lin0R;explain1Li1;nterrupt1Kvers0;e3h1oilet,rapp1J;e1i7;atr1Bmat1K;en1n;!aB;adi0Te6i5outh4pani0Yt2u1;burb1Ccce0O;anda1retch1C;rd;ern;nks,xth-gra0V;cond,izur0Ever0;e4o2u1;r0ssi15;hypnol,ugha1;ge;d,ha0Nlat0C;arallel,hiloso0Rr1;esent,ior,ov07;bstacl06ffers,pposi0Wverall;e1orwegi0Y;t,xt;a6e4id3on2u1;ltipWs0Qzak;key,osyllab0Y;!dU;di1x0R;c0ev0;rAthemat0L;a1eft,oc0;sti0Rt2w1;ful;er,tF;a1ewi06;il,pa03;mitation,n1ri04t0A;consistenciPdie,i2terna1;l,tM;ti0;a3i1;storic1t-and-miQ;!al;nd-held,rd5;amecha3e2lob0r1;eek,oM;n01rm09;ng1;er;a3inan3o1rench;rm1urth;er,ula08;ci0;ngliNurope02;aily,e4i3o1;cumentBub1;le;git0urn0;fec3nominat2stroy1;!iZ;ion0;at1;es;ardiac,h9lassNo5ro4u1;lin2rs1;ive;ary;ss;m1ncavo-convex,rporaK;a,mercial1;!i1;sm;i2r1;istiIonC;nese;ack,iographKl2oilerplaDriti1;sh;ank,on1;de;bsence,cEfAlterna9merBpproachiFr5siCtmosph4u1;str2tobiogra1;ph5;ali9;erC;chae1rest;olog1;ic0;al;te;oremention3r1;ic1;an;ed;adem2ti1;ng;ic|s0.9¦0:1T;1:1N;2:1M;a1Cb14c0Qd0Le0Df07g06h03iXjUkind,lSmPnNoMpJrGsBt8un4w3;elcome,ise,orth12ro1Q;asham0befit0Pchas1UdignTexcell0f5grad0ha4necess0Hpr3;eced7opiQ;mper0ppy;aith16ruit16;a3ren0O;l3ste0W;ent0;ensi1Jp6t5u3;bnorm0Aper3rpris0;fiOior;at14ereotyp0;ectac8ontane1;a4e3ofl;doub0Jleva2source0W;nk,re;ain0Uerplex0leon00o3rolix;p3wer0L;ular;bedIptimum,verexcit0;a3o0X;me0Hugh0N;annerly,i4o3;de0Vtley;gh0Knd0E;ate0TenCicen3ong-wind0;ti1;ackass4ust3;ifi0;!es;mpat7n3ron0L;auspici1credi0Md5exp4huma3tellige2;ne;ed4lica0K;ispensa0JomiZ;ie2;a4ero0Fonest3uge;!-to-god;nRte,zard1;orge1randiloque2;a7i6l5r4ull3; of life,-fledg0;ee-thinki0Kingy,ust8;ippa2y;ll0rst-st0H;bl0ir,nt7vor0;conomic9le7n5rot05x3;hila3perienc0ubera2;raE;orm1thusi3;ast01;ct,ment3;ary;al;anger1e4o3;ne with,ubtR;bauch0ft,l4p3;loraWressi05;ica09uxe;aDhBiviliz0lassy,o8r4ut3;ti02;a5u3;d3sB;dy;fGzy;cky,ntes4r3;pule2ruptiN;taM;ill3urniU;iTy;re4st-iron,tc3;hiR;less;a8eefy,l5o4ra3;insick,sh;nny,otleg;ast0ood3;thirs3y;ty;rbar1ttle3;ful;bsoKcJdCfAgha9hw,lgBmaziHnnoyiHpp7rtesian,s5ttenIuthoritaIww3;!w3;!w;cet3tuJ;ic;alliClauda3reciaD;ble;st;firmaAra3;id;dic8ept,o6ve3;ntur1rte2;nt;ous;ri3;ng;tive;ross-the-board,u5;lu4rb0;ed;te|s0.3¦aTbScPdMeHfEgreen,haha,lDmCpAre9s2thought-provokVundeservOv1w0;haddupwitdQonky;apid,e6;e3h0ucks;a1o0;rt,uldA;m,peless;cond0rmon;a0hand;ry;al,gurgitates;ar8erson9hilosophJi0reachy;nk,ous;asculine,odern;ow,ush;ail1luff,reque0;nt;!ed,s,ure;ar3ssen1x0;cruciating2ploitE;ti0;al;ly;evelop1is0;abl0tinct;ed;hitch1onc0;eiv7rete;at;est,usy;d2lienat1xiomat0;ic;ing;ministr1ult,vers0;ative;able|s0.2¦a09b06c02d00eZfVgrief,haUiRjoy,lQmKnHoFpDrAs8t6u3w0yarn;as0Fe1iAo0;nt,u02;ll-intentionVre0D;gly-duckl03n0s3;answerTe0fortunateS;ngag01ven;hanks,on0rouM;el00;milPob06poof,ta0will;inlYte-supportO;e1obot00ui0;ns;dunda03grets,moZ;erpetualJiBl0rofiterT;atitud1od;ld,oz0;es;e1o0;!t;ithVvV;e3i1u0;ch,ltilaterSstU;nor,s0;fire,placA;diocri0lodrama,ntPtaphorO;ty;augh7egNosers;mpercepti1n0;fatuat5telligentsia;ble;dLppinDsLveL;ix2orc2rustrat0un;ed,ing0;ly;ed;conomBrudiC;evoid,idFo0;esEnt;hoppy,inemat8luelessn5o0;mplain1u0;ldB;ed,t;adn1udd0;ing;ess;glow,i6ng5re6typ3u0;rea1tist0;ic;te;ic0;al;er;nt|s0.125¦bas2f1independent,occasional,s0;ame,pecif1;lat,oreign,uture;ic|s0.5¦a1Db14c0Vd0Se0Pf0Fg0Ch0Ai07l04mZnYoWpNquiMrJs8t5un3v1w0;himsEorthwhi19;a0ulner17;gue,ri0W;a0blem1Glike0Estirr1H;b15ppea1Cwa0F;e1houghtWiny,o0ruthWyp9;p,uchi1B;n0Urminal0A;afe,h8i7k5m4t1u0;bt10ch,dd03ffoca0GrprF;a1eaAi0;ff,nks;le,rt14;all0Uooth;ept0ill16;ic0W;mplist12nce03;a0od3y;ky,llow;andom,e0ip-off,omant0Z;a0deemi0Xminisce0Kspe6;dy;ck,xot0W;alp0Le7henomen0Oinheads,lausib0Mo4r0uW;e1ima0Com0;isi0S;di0g3;ct0H;i0te0C;g0n03;na0A;ace6ppe05;bvi02ffbeat,k0pGrdina04;!ay;ot0Aumer00;a3e1ild,o0;re,st;aning0re;ful;jor,ny;arg03ess03i1o0;usy,v03;ab03k02mp,tt03ve;con0Bdentifi01lleg04mplicat0Dn0;articula06terest0;ed,i08;ahahaha,ea0ighX;ltAvy;en1immicky,luey,old0reatV;en;erXuine;a8i7la5or3r1urt0;hRiZ;agiSesh,iend0;ly; su0gettP;re;s0wZ;hy;ne,t2;miliar,rce;arliIccentrTmpty,n1xhaus0;tiR;deariQergetRjoy,ough;efiniMi1u0;ll,mb;m,shon8;heesi7o0;ar5m3n1un0;tless;ceptionDsci0;ous;e-at-8ic,mon,plimenta0;ry;se;est;a6e2i1lata0;nt;gg1tt1;liev1tt0;er;ab0;le;n0ss;al;bridg7ccompl6lcohol5pp0;ea2licati1ropria0;te;ve;li0;ng;ic;ish0;ed|s0.3333¦adequate,brief,const0extensive,first,hidden,main,older,particular,quiet,realistic,subject,vibr0;ant|s0.6¦a0Ab08cZdReMfJgIhahahahaha,iEjamm0OlBmAn8orthodox,p7r6s3un2w0;arm,ell-off,o0;manly,r02;hesitati0Gknown,notic0L;ec1hrill,t0;ark,inker;!u03;aucReaso0Aisk-free,ude;eevish,loddi0Boor;a0onviolT;turalisWuseT;ark0DinimF;a04e1ifel0o0B;ike,o06;ftist,gi04;cky,n0;c0experie05flexi02tZ;ompa0u0;ra00;ood,uard05;a1rostbittRu0;ck,ll-bodi03;lIst;colog2n0rroneBth2veryday,xtinct;ig8joy0;aTiU;ic0;al;estrucFi5r3u0;h0sty,uuh;!hh0;!h;a0iSy;ma9;aphan0ffer4slikQ;ous;a5loud-coverOo0ryiJ;mpelliIn0;ci2secr1ting0vex;ent;atK;se;p1us0;tic;tiF;izar0road-mindF;re;cBd9larmi8me6pp3r1sh0we2;en;bitr0ousB;ary;etizi4ortion9rox0;ima0;te;na0;ble;ng;dict3equate to,va0;nc2;quaint1ti0;ve;ed|s0.25¦aDcAe8fic9juveni7logicCm5odd,p3re1s0usuC;eason5o6tel3upporting;c0ligious,petitive;e8ognizab4;ast,o0;lar;arri0ix0or6;ed;le;duca0qu3vide2xact;tion2;entr1onsiste0;nt;al;mateur,ware|s0.8¦a05b04cTdNeLfHgEhaphazard,iAl8m7p6re5s2tiOun0vulgar,well-advis08young07;blink05c0focus07think00;ared-for,ontroversiVut;arcasOh1kitt04t0uckerA;abb02eadfaF;it,ock03;putVtard02;ivotQrecise;aladroit,uggy;ikZo0;lol,ud,vY;diot2mpassBn0rrit1;corrupMfuri0;atT;!s;argantuan,entNrr0;!r0;!r;*2a0ree,u2;na8rthermo0;st;ckL;ffect0xcitK;ive;azKe4i1ow0;dy;dac1ffidJrty,s0;astrCbelievFturbF;tic;fenseless,servD;a6h5o1r0;ap,itic2;l0mfort5;lec1oss0;al;tib3;arit1ild7;cophon2lcul0ndid;ab0;le;ous;loodstain3right;bhorr3lleviat2mateur1nnoy2pposite,ttention-gett0;ing;ish;ed;ent|s1.0¦0:3W;1:3V;2:3T;3:36;4:40;5:3S;6:3P;a3Ib3Ac2Kd28e1Yf1Mg1Ch15i0Wkey,l0Tm0Jn0Go0Dp07qu05r01sMtJuDvBw7yaaawwnnnn;ac05e9it2Io7tf;nde1Rr7w;st,t2K;alt2Ji45;a1Ci7;c5o1Z;g29lti2Pn7tt0H;appetizi1believ4comAexpect0f8i7predict4real1Zsalt0usu6;nspir0q16;air,or7;gett4tuna2M;fort4mon;e8hril25ire38op-notch,r7wist0;emendo2oubl0;d5r0S;aJcaHeFhoc2XiEoDpCt9u7ympaW;p7rre6spens2R;erb,por3L;ereotyp22ri2Uu7ylD;n14pid7;!i1Y;irit0lend31;ft-boil0phist2Cr3D;cke10nist00;lf7ntiment6rious1Rxy;ish;r7t31;ey,y;d,tisf7;i0yi1;a9e8i7uthl10;dicu11ve2D;al1Jfres2Vpel18tardNwar2U;nc08unc1P;estion4ir7;ky;aBerfect,leas0oAr7sychot2T;e8icel0To7;fou0Fmine3ud;c5t1F;ss0Ktenti6we0O;thet2O;bsess0n19utrage8ver7wn;-the-top,whelmi1;d,o2;a8e7ice,ostalg2Kumb;cessa2Pedl0Krve-rac22;i2Ss17;aCe9i8oron7yster5;!s;nd-bogg13ser4;a7diocre,m2Pna1Frcil0F;g7ningl0E;er;d,g8levo0Mrve0Es7;si2Kte09;ic7nifice3;!al;a8egenda2Di7mao;ght-heart0ke0S;ugh4zy;de6ll,mAn7rreleva3; good tas16appo0Aconvenie3evit4famo2gen5nova2Cs7tDven2C;ane,pir7ul1H;a0Bi1;beci2DmAp7;o8ress7;ed,i28;rta3ssS;en1J;aBeartfe0Dilar5or8um7yster0M;oro2;ri7;b25f7;ic,yi1;nd1Hp7;lQpy;iFlad,oDr7uil0D;aBeateAi9otesq8u7;d0Ye1D;ue;evo2m,ppi1;st;nd;o7ry;dy-goody,fy;a3ft0;aEeCiAlawlForc9rig7unny;hte7id;ni1;ib1O;lt03n7;al,e-loo0S;a7eb1Llicito2;rf0X;bu9i3ke,mo2r8ultl7vori09;ess;!-out;lo2;erie,goFlEspecialOvil,x7;agger07ce9ot11presAqui8tr7;aordina16e0S;si02;l9p8s7;si17;tion6;le3;aboEega3;ist0T;aFeBi8r7;eadf0Hunk;fficu8m-witt0s7;gust09traught;lt;ad9finite9li8spe7vasta02;raP;c5ghtf0B;ly;in8zz7;li1;ty;aThNoDrAu8yn7;ic6;r5s7te;hy;e8u7;ci6de,el;a0Jepy;ld,mDn9r7urt8;iac7r0O;eo2;firm0sum8vin7;ci1;ma7;te;ic6pl7;ic7;at0;a9ee7;r7sy;fNy;llen8r7;ismatVmi1;gi1;ptiva8r7;efI;ti1;aDeCleBorAr7;aZeathta8illia3ut6;nt;ki1;ed,i1;ak,ch;autifAlov0witcI;se;bWdTff4gitaRirhead0las,musi1nLpKrtiHsDtCvBw7;e8f7kwaW;ul;so7;me;id;roc5tracL;sumpKto7;nis8un7;di1;hi1;fici6st7;ic;al;pall0t;g9x5;io2;us;ry;ng;ed;ti7;ve;or4;ab7;le;r8su7;rd;upt|s0.7¦aZbYcSdMeHfEhaDiAl7m6open-mind01p5respectPs4t3u1w0;ary,orkmanlike;n0sed to;adulteratYcriticNhealthy,ilaterNplay9settlQtrace9;hrillXirX;ecret,pright4uffers;lacid,retentiU;esmerizModer8;ight,o0;l,ne0;ly;maginat8n0;calcul0expressJnoceQ;abJ;r2tN;iendi1ortun0uckM;ate;sh;ffBn2x0;haustIpens0;ive;g0joyGtertain8;ag7ross7;e4i0udsvil9;alect2s0;appoint4taste0;ful;al;creas9ficieAlight9;heap,lean4o1red2unn0;ing;here7mprehens0nfus6;ib0;le;!ly;ang-up,ehind,oundless;dama2ir1utonom0;ous;ed;nt|s0.4¦a0Bbr0Ac03d00eWfTgShRiOlMmKnFpDr9s5tries,u3v2w0you05;et,ho0Di0;de,ld,n;itVocationV;glin05n0;avow0Ebra7paid,read,school0E;atisyfiZcum,h2imilar,lDmall,o1tr0;aight,uttiY;phomorRu09;owery,rie5;elat09ight1ou0;gh,nd;-mi0iB;nd06;ea0risUut3;ky;a2e0;ar,g0;ati00;k00rr0turF;ow;anque,esNo0;d,raliziJ;ea0oI;st;diocy,mmaneQn0; stock,competePdividu8eluctNsultingly,t0;ellectu7riguiE;ahaha,umbM;ame,raph6;a1everish,i0luffy,orgetD;rm,t;rcic2tty;nlighteni8p2xpe0;ctLriment0;al;ic;ark,e1i0;rect,sappointmeD;ad,ep;ap9easel5lum4o0urreB;lor2mple1nfusi0;ng;te,x;ful;sy;ess;avado,oken;cuate,g7l4skew,ttenda3v0;ail0erage;ab0;le;nt;i1l-arou0;nd;ve;ed|i2.¦definite1especi0re0serious1;al0;ly|s0.1667¦c3dynam5fema2half,in1mundane,p0slight;a1revious;coherent,ner;le;lass1o0;ntemporary,w;ic|s0.65¦cool,emotion0lyric,norm0sweet,true;al|s0.625¦able,entire,shady,weak|s0.75¦aKcGdEeDfabricIgBi9l8origin7poetLr6s4t2vivid,wi0;ll0nn0;ing;ot4ra0;dition3gH;h0loppy,uit5;arp,eer;emark3ich;al;ame,ovely;mpecc0ndecipher0;abC;odforsakAr0;eat,itty;dgy,ndless,xcit5;etail4is0;appoint3eas3;alm,elebr1laustrophob4o0;ntriv1zy;at0;ed;eriform,gi2li1mbitious,uthent0;ic;en;le|s0.6667¦b3compete2dece2insta2s0;e0uper;nsational,rious;nt;ad,lind,old|s0.0667¦l0public,social;ast,ess|s0.05¦missing,outside,subsequent,topical,victim|s0.1429¦limited,physical|s0.375¦accessible,due,other,private,s0;lick,traightforward|s0.0769¦regular|s0.3571¦conventional,plain,simple|s0.1333¦gloom,spiritual|s0.55¦criminal,full,responsible,subconscious|p-0.0125¦absence,vacuum|p-0.05¦arreDcCgAl9m8obstacl7p6rehash,s1un0well-intention5;affect4explain4thinka2;adism,eizur5ixth-grade,t0;r1um0;ble;ange,etch0;ed;assionate,erpetually;es;inor,onk2uzak;efti3ong;argantuan,r0;ey;ardiac,hangeless;st|p-0.1¦a0Nb0Jc09d00eXfVhaUiQjaPlNmJnGpErAs6t5u3w0yarn;as0Te1haddupwitdat,ide,o0;nt,u09;re0Rt;gly-duckli0Pn0rin03sed to;answer0Qbrand0Qcook0QfortunateUknown,proc00;on09ri02wo-dimensionG;e1hou04inks,lippi0Moft-boil0Op0tereotyp0Owill;e0Moof;c0riousQ;!ondhand;e2o0;bot0Ehypnol,ugh0;!age;grets,mo07;arti7eppe0Ai0ropaganda;nk,ty;e1o0;!t;ith03v03;ent2i0onosyllab06ust0A;lita04n0;im0us;al;i0oneB;a4felo04;il,mm05;cy,mmane03n0;articuSconsistenciGe0telligentsia;lucta0xperienc02;ble;dZrdRsZveZ;everish,luff,orgetMu0;ll-bodiYrZ;mpty,x0;cruciating0pectW;ly;e5i2o1r0ulls;ag,ownT;esRnt;dQs0;believiOposs0taP;essP;fec0void;at0;es;a7e4l3o0rushLurious;m1ntingeJu0;ldI;a,mercialism;osHuelessn2;as0nt7;el0;ess;nt,re0;ful;itt2oilerp0;la0;te;er;dversa8g7i6l3phon2r0skew;bitra0e5;ry;ic;armi1l0;eg2usions;ng;nt;ed;tive|p-0.2¦autist0Gb0Ac02dTeveryday,fRgPhNiKkill0FlImEnarrow,oCp9r8s5tr4un2wa0;n,ste0;!d,s;cared-for,e0notic0CoriginWschool0C;ngagiZvM;app0AoubP;h0ophomor0Atark;a0it,owery;m,pe05;edundaOightist,ound;l1re0;achy,dictabJ;atitud1od;oz0rthodox;es;e2i0;ndXs0;fire,placYsiM;diocriAssy;ead7imp,o0;ng-windVsers;mp1n0sC;cohereBfatuatT;atieAerceptib7;arLeavy,i0umb6;ndHt-and-miP;immicFre0;en;at0ew,irm,luffy,rustratingly,ull of life;ty;e6i1riLudsvil0;le;a2ffide1s0;ablIlikI;nt;lect0phanous;al;ad,stroy0;!i1;h4loud-cov3o1ryi0;ng;c0zy;ky;er8;i0oppy;ldi0tchat;sh;o1ra0;sh,vado;gg2und0;le0;ss;ed;ic|p0.3¦aMbonny,cCdistinct,eBf8g7int6l5mesmerizHp4raAs3u1vocationKw0;ins,orth;n0seJ;blinkEcontroversiH;miEolicitous,uccess;owerGroves;augh,ifelike;ellectuDriguA;amechanger,olden;or su1r0un;esh,ingy;re;conomic8le4nlighten5xcit5;alm,leanly,o0;lo5m0;e-at-ab3p0;ell1li0;mentary;ing;le;r1ss0;al;ful;bsorb0hw,pportion0uthoritative,ww;ed|p-0.1333¦active,cow,gloom,imitation|p0.1¦0:06;aYbVcLdJeHfFgettRiElBm9n8o7p6right-mind0s3titulUun2vit02w1youX;ild,ow;blemish0expect0hesitatiVread;arcaAe2o1tate-support0urpris0;ber,ft;amOnsitV;eaky,rofessionW;ffers,ld;aEeM;a1edicatRodest,ultilaterT;nque,rk0sculine,ture;i1oud,ush;ngui1teraN;st5;dentifiDn stock;ar,i1requeO;ftieth,x0;arly,mpiricLnigmat1p1rudiOxperimentL;ic;evelop0i1;m,rect;l7o2ul1;turG;mpleIn1;c2s1;cious,ider2;eiv1ise;able;assy,e2oud1;less;ar;almy,u1;ddi1sy;ng;bridg0c6d3ir0l2mato1rous0;ry;ive;ama2jectiv1ult;al;nt;ross-the-board,tion,ua1;te;ed|p-0.0333¦excuse,toilet|p-0.7¦a5ba4crude,disastr6frustrat3grr1merciless,painful,shock3u0vulgar;gly,nchaste;!r0;!r;ed;d,rbarian;bhorrent,nger,troci0;ous|p0.5¦0:13;a0Pbe0Nc0Fe09f06g04h02iYlUmNnot12oJpGrDs6t4unexce9w1;acky,e02i2or1;kmanlike,thwhi12;n08tty;o1ruthfM;p,uc0M;a5ex4incere,ki3ophistic0Qpirit0t2u1ympath07;ppor09re;at0Iriki0Nun03ylish;ll0;u0Ay;fe,tisf1;i0yi0J;e1ive09;dee6fres0Dput0Ospect1war7;ab0PfB;henomen04ivot04l2o0Jr1;eci0Komine0J;ausPeas0;k3utstan2verwhel1;mi0B;di0A;!ay;a4e2ind-bogg07o1;re,st;aningf1mor0D;ul;gic2n1;nerly,y;!al;atest,enie06i3ov1;ab09e1;!ly;ght-heart0k05;conSn1;corrupt8geni02no2spir1teresNvenH;ationKiV;ceZvaF;andsome,e1ilariZumorZ;althy;ift0lad,oofy,r1;and,eaJippiQ;a2itGorc1;ibX;mTvoriL;l5n1xotH;deariLerg3joy2tertai1;niK;abSed,iJ;etD;aboraFegaM;aptiva7ha6o2rea1uE;tive;hereJm2n1;fideIvinciD;ic1peG;al;llengiArismat5;ti9;lievFt1;ter;bFcquaint0d8giFirhead0llevi7pp3stonis2uthent1wwww;ic;hi4;ea2ropria1;te;li1;ng;at0;or5ve1;ntur3r1;te1;nt;ous;ab2;ed;le|p-0.125¦abrupt,due,e0foreign,other,sharp;ndless,xtreme|p0.2¦aKbeefy,cHeEfast,gentOhahaBi9leg8m6p5quixotGre3st2thanks,un1very,w0;ants,hoN;paid,usuD;ainless,raight;a0sponsibK;dy,l,soI;hilosophAromisL;od0uch;!ern;al,ibF;nt0ron6;ense,imaH;!ha0;!ha0;!ha;conom1thic0;al;ic;ap1on0;secr3vex;ab4tive;bso5ccomplish4dor6ll-around,me2pp0ttendant,urea7;etiz5reci0;at2;nab0;le;ed;lu1rb0;ing;te|s0.95¦a3barbarian,c2discourteous,gawky,lov6mouth-water6rose,successful,un0;i0pleasa4;mporta3ntellige3;hicken,ontroversial;b0fflue1;sorb1unda0;nt;ing|p-0.5¦a0Wb0Rc0Dd07e04fXgVhonest-to-god,iRjackassQlOmeNnMoLpJquestion0PraIs9t8u2v1w0yaaawwnnnn;a10ei11himsBro0Jtf;ague,ulner0N;n0se0L;a3c2f1i0likeCreaBsett0F;later0Inspir0N;air,ortunate;omfort0Iut;b0Isham0K;ediXiresome,roubl0Jwist0J;ad,car7elfish,hy,i3kept2or0Rt0uffocati0A;a0ereotyp1ink3;le,rt07;ic0A;l2mp1nist0;er;li4;ly;ey,y;ndom,uncH;leona0ower04sychot0G;st0F;bsess06ffbeat,ver-the-top;ame01eed01ostalg0D;aning00dioc5re;a0ouW;me,ughZ;!es;ll2n0rreleva03;auspiciDcurWexpedie02famDsecu0;re;!egS;o0ratuitAuilY;ody-gooDry;a3eebSla1orgettRr0;ighteJostbitt00;s0wS;hy;i0ke;l0nt;!ed,s;erie,ffiFrrone1x0;aggerApensS;ous;azKi3ow2runk,u0;hhh0uuh;!h;dy;dactOfficult,st0;asteful,urbi7;a8h3o0reepy;llectibBmplic1ntrivDr0;puleFrupt;atB;ee3il1ur0;ni1;li0;ng;sy;lcul2re1su0;al;less;ab0;le;l1or0rainsick;ed;ata1ind,oodthirs0;ty;nt;bsu5ng4s0wea4;cet2h1sumpt0;ive;en;ic;ry;rd|p0.6¦aOcJdeIeGfDhAimaginaSkind,l9motley,n8own,pop7r5s3t2un1w0;arm,ell-advisD;hamperCprecedentC;hrillBi9ren9;atisyfNmilAp0;ectac2ontaneF;e0ose;doubEsourceful;ular;ice,oD;ik4mao,ovH;a0onest;n0zard8;dy;i1ull-fledg0;ed;ne-lookBrst-strB;ffecBn0;gross9thusiastic;ft,luxe,serv8;a3hari1ourte0;ous;ta0;ble;ndid,tch3;bundant,cute,dept,ffirma3m1p0stound2;precia2t;az0us0;ing;tive|p0.375¦accessible,excited,f1original,poetic,rich,s0unique;ignificant,traightforward;amiliar,riendly|p0.4¦aUbScLdetail00eJfDgBhuge,i9justifi00li8mighNn7o6p4r3s2thought1un0well-off,youngish;adulteratZsaltZ;-provokWfP;ecuQmooth,ound,prightly,teadfast,uperf9;elevaBisk-frA;erplexVr0;ecise,imary;bedie8pen-mindT;aturaliAonviole7;centiNght;mporta5n0;comparaKdispensaK;enu0uardO;ine;a3i2lippa1ortunaNr0;ee;nt;llJt;bulEnta0r-out;stic;cological,n0;gagDjoy;he4ivilizEo2r0;af0edi8ushB;ty;m0nfirmB;forta5prehensi5;ap,erf0;ul;ang-up,izar0;re;ccura6dvanc5pp4ttenti2utonom1vaila0www;ble;ous;on-gett0ve;ing;licative,osi1;ed;te|s0.6333¦accurate,outdated|p-0.4¦aXbVcOdKexHfCgBi7n6o5p4rip-off,s2t1un0worG;appealiIfocSgrad00healthy,importaMnecessary,playQstirr00;erminally,hin,irZ;ecret,hri0;eky,ll;eevToor;utd4verexcitV;ause3erve-rackiC;mp1nf0rrit5;antiJlexibJ;assive,lic0;atQ;ame,luey,odforsaK;a1rustr0uck;ati5;l1rc0;e,ic9;se;haust0tinct;ed,i0;ng;e0usty;creasFf0;enseless,icie0;nt;a5heesiest,on1rimin0;al;f2test0;ab0;le;us7;cophonous,ustic;ehind,ootleg,ro0;ken;d2l1mateur0nnoy3pproximate;ish;as,gid;dict0equate to;ed|p-0.4667¦addled,maladroit|s0.8333¦addled,bland,c2easy,gratuitous,idiotic,l0sexu1tough;oy0ucky;al;lever,onfident|p0.3333¦a2bo1extraordinary,l0mi1pregnant,quick,super,tremendous,worthy;oyal,ucky;ld;dequate,rtistic|p-0.25¦a7excess6juveni5lazy,o4p3repetit6s1u0;nbelievab4su1;entiment0hady,lick,mall;al;ast,ointless;rdinary,verboard;le;ive;eriform,l0mateur,nxious;coholic,ien|p0.8¦a7bra8e4f3great,happy,intelligent,joy,lol2proud,rofl,un1w0;elcome,in;common,forgett7;!ol;avor2ly;lect,xp0;erienc0ressi2;ed;ff1ttracti0;ve;able|p0.65¦affluent|p-0.6¦aWbScMdCfAgr9hap7in6m5numb,prolix,s2tastel8un0;befitMdignifiTf0happy,propitiK;aithTruitT;cathiLt1u0;bnormUffers;abbiJinks,upidiC;eager,uggy;conveni9explicaJfuriaG;hazPl0;ess;andiloqu6udgiE;*ckiDiendish,uck0;ed,iC;anger8e5i0one with;m-wittGr3s0;appoint0easFtraught;i8m0;ent;ty;p1s0;perate,trucG;lora5ressi3;ous;hi4o2razy,ut0ynicB;ti0;ng;ld,rrupti0;ble;cken,lly;attle2l0;ast0oodstain0;ed;ful;fraid,g2rtifici1wkw0;ard;al;hast,ita0;tive|p-0.3¦alienatXbaVcOdLeIfGiElCmBnAp6r5s2thick,u0vap9wonF;gliVn0;deservFtraceable;cum,econdary,hoddy,low,ourEtruttUuck0;er0s;!s;aucKegurgitat7ude;inheads,l1r0;etentiIisMofiterP;ac0oddO;id;a9eg8;elodrama,oralizL;aAea9os0;es;c0diocy,nsultingly;ky;anatic,orc0;ed;dgy,xploit0;at0;ive;elica1ishone0uh;st;te;lum5o0;m1nfus7riace0;ous;mon,pl0;ain0ex;ed,t;sy;d0nal;ness;ing|p0.25¦aFcBe9f8h7interest6l5marri6p4recognizable,s1thr0w0;ill2;eason4tellar,u0;pport0rre7;ing;eaceful,retty;ogic4yr8;ed;igher,ot;irst,unny;ducation0vid3xact;al;o0risp;m1nsist0;ent;ic;mbitious,vid,ware|p-0.75¦anger1claustrophob0disappoint1trag0;ic;ed|s0.85¦a2b0deadpan,fascinat1hot,passionate,solicit3thin;almy,ecom0;ing;ngered,rdu0;ous|p-0.8¦aCbAcrap,d9egoistic,filthy,grie7ha6i4m2r0stupid,unappetizEvi3worthless;an0etardC;cor6k;alev0oron2;olent;diot0mbecile,napposi1;!s;te;f,v0;ous;ebauch3oubtful;ase,l0;ech,oody;nnoy1ppall0;ed;ing|p-0.35¦a0;ppalling,rduous|p0.05¦appare1bare,ceremoni0exubera1gener0inexpressible;al;nt|s0.35¦appare0ceremonial,dista0infantile,soft;nt|p0.7¦applaudable,bCchBdelightFe8f5go4h3l2mouth-waterEoptimum,s0talentFwise;kittish,u0;cceeds,perior,rprisC;aughCovC;appiness,ero6;od,rge1;a1elicit0;ous;bl7ir,scin1;rot1xhilar0;at4;ic;arm2eery;e0right;lov1witch0;ing;ed|p0.9¦artesian,brilliant,c2dainty,i0;deal,n0; good taste,credible;ast-iron,ushy|p0.55¦astute,controversial,suitable|p-0.15¦average,ba0dark,me0naughty,pale,ruins;ss|p1.0¦awesome,bBdeli8ex6f4greateCimpress3legendary,ma1p0superb,top-notch,wond2;erfect,rice4;gnific5rvel8st0;er6;ed,ive;ault0law0;less;cell0quisite;ent;ci1ght0;ful;ous;e0reathtaking;st|p-1.¦awEbCcruel,d9evil,gr8h4ins3m2nasty,outrage1pathet6ruthless,shockiDterri0vici1worst;b6fyiC;ous;enaciAiserab4;ane,ul7;orri0ysterical;b1f0;ic;le;im,uesome;evasta1isgust0read4;ed,i2;ti1;leak,ori0;ng;ful|p0.85¦beautiful|p0.45¦becoming|p-0.1667¦bla2h1less,mundane,odd,previous,s0typical,unpredictable;light,ubject;alf,idden;ck,nd|s0.4333¦black,full-length|p0.0625¦broad,major|s0.3125¦broad|p-0.875¦brutal|s0.8667¦casual,measly|p0.35¦c0full,sweet,true;elebrated,ool|p0.2143¦certain,large,pure,smart|s0.5714¦certain,special|s0.15¦changel0m0overboard,strange,wan;ess|p-0.0667¦characteristic,d0hollow,putative;epress,ry|s0.4667¦characteristic|p0.1667¦c2dece1main,old4particular,realist5s0vibra1;teady,upernatural;nt;l0ontemporary;ass1ev0;er;ic|p0.3667¦clean|s0.3833¦clear|p0.15¦concrete,normal|s0.45¦considerable|p0.95¦consummate|p-0.1429¦conventional|s0.4167¦crisp|p-0.9¦cruddy,f5h2inhumane,outrag4re0sicken3;pellent,tard0;!s;at1orrify0;ing;ed;earful,rigid|p0.75¦dazzling,impecc0remark0successful;able|p-0.8333¦deadly|p-0.55¦deadpan,g0indecipherable;awky,rotesque|s0.0333¦depress,excuse,hollow|p-0.65¦discourteous,un0;intellige0pleasa0;nt|p-0.1556¦down|s0.2889¦down|s0.07¦drag|p-0.4333¦dramatic|p-0.2917¦dull,hard|p-0.375¦dumb,incompetent,weak|p0.4333¦easy,strong|p0.6667¦excep0lively,sensa0;tional|p-0.3167¦failure|p0.4167¦fine,gay|p-0.025¦flat|p0.0333¦full-length,social|s0.5833¦gay|s0.5417¦hard|p0.16¦high|s0.54¦high|p-0.6667¦i0;diotic,mpossible|s0.875¦insecure,outstanding,si0;gnificant,lly|s0.4286¦large|s0.2667¦leaden|p-0.0714¦limited,single|p-0.1875¦little|p0.1364¦live,new|s0.9333¦lively,uncivil|p-0.0769¦loose|s0.2692¦loose|p-0.625¦mad|p-0.3125¦mean|s0.6875¦mean|p-0.5667¦measly|s0.4545¦new|p-0.24¦parade|s0.22¦parade|p-0.2143¦plain,stiff|p0.7333¦pleasant|s0.9667¦pleasant|p-0.0833¦polar|p0.2273¦positive|s0.5455¦positive|p0.0833¦profound|p-0.2308¦raw|s0.4615¦raw|p-0.3333¦ridicul2s0tense;eri1ha0ubtle;ky,llow;ous|p0.2857¦right|s0.5357¦right|p-0.225¦sermon,tame|p-0.7143¦sick|s0.8571¦sick|s0.2143¦single|p-0.4167¦sloppy|s0.6429¦smart|p-0.1571¦sour|s0.1143¦sour|p0.3571¦special|p0.8333¦splendid|s0.7333¦strong|s0.5667¦supernatural|s0.8889¦sure|s0.225¦tame|s0.475¦thick|p-0.1786¦tight|s0.2857¦tight|p-0.3889¦tough|p-0.7333¦uncivil|p-0.075¦victim|p0.125¦vivid|i1.5¦real|i1.3¦excruciatingly,very|i1.2¦frustratingly";
