<?php
declare(strict_types = 1);

namespace Spaze\PHPStan\Rules\Disallowed\RuleErrors;

use PhpParser\Node\Expr;
use PhpParser\Node\Expr\CallLike;
use PhpParser\Node\Expr\MethodCall;
use PhpParser\Node\Expr\StaticCall;
use PhpParser\Node\Expr\Variable;
use PhpParser\Node\Identifier;
use PhpParser\Node\Name;
use PHPStan\Analyser\Scope;
use PHPStan\Reflection\ClassReflection;
use PHPStan\Reflection\MethodReflection;
use PHPStan\Rules\RuleError;
use PHPStan\ShouldNotHappenException;
use Spaze\PHPStan\Rules\Disallowed\DisallowedCall;
use Spaze\PHPStan\Rules\Disallowed\Formatter\Formatter;
use Spaze\PHPStan\Rules\Disallowed\PHPStan1Compatibility;
use Spaze\PHPStan\Rules\Disallowed\Type\TypeResolver;

class DisallowedMethodRuleErrors
{

	private DisallowedCallsRuleErrors $disallowedCallsRuleErrors;

	private TypeResolver $typeResolver;

	private Formatter $formatter;


	public function __construct(
		DisallowedCallsRuleErrors $disallowedCallsRuleErrors,
		TypeResolver $typeResolver,
		Formatter $formatter
	) {
		$this->disallowedCallsRuleErrors = $disallowedCallsRuleErrors;
		$this->typeResolver = $typeResolver;
		$this->formatter = $formatter;
	}


	/**
	 * @param Name|Expr $class
	 * @param MethodCall|StaticCall $node
	 * @param Scope $scope
	 * @param list<DisallowedCall> $disallowedCalls
	 * @return list<RuleError>
	 * @throws ShouldNotHappenException
	 */
	public function get($class, CallLike $node, Scope $scope, array $disallowedCalls): array
	{
		if ($node->name instanceof Identifier) {
			$methodName = $node->name->name;
		} elseif ($node->name instanceof Variable) {
			$methodName = $this->typeResolver->getVariableStringValue($node->name, $scope);
			if (!is_string($methodName)) {
				return [];
			}
		} else {
			return [];
		}

		$calledOnType = $this->typeResolver->getType($class, $scope);
		if (PHPStan1Compatibility::isClassString($calledOnType)->yes()) {
			$calledOnType = $calledOnType->getClassStringObjectType();
		}
		if ($calledOnType->canCallMethods()->yes() && $calledOnType->hasMethod($methodName)->yes()) {
			$method = $calledOnType->getMethod($methodName, $scope);
			$declaringClass = $method->getDeclaringClass();
			$classes = $calledOnType->getObjectClassReflections();
			$classNames = array_map(fn($class): string => $class->isAnonymous() ? 'class@anonymous' : $class->getName(), $classes);
			if (count($classNames) === 0) {
				$calledAs = null;
			} else {
				$calledAs = $this->formatter->getFullyQualified($this->formatter->formatIdentifier($classNames), $method);
			}

			$ruleErrors = $this->getRuleErrors(array_values($declaringClass->getTraits()), $method, $node, $scope, $calledAs, $disallowedCalls);
			if ($ruleErrors) {
				return $ruleErrors;
			}
			$ruleErrors = $this->getRuleErrors(array_values($declaringClass->getInterfaces()), $method, $node, $scope, $calledAs, $disallowedCalls);
			if ($ruleErrors) {
				return $ruleErrors;
			}
		} else {
			return [];
		}
		return $this->getRuleErrors([$declaringClass], $method, $node, $scope, $calledAs, $disallowedCalls);
	}


	/**
	 * @param list<ClassReflection> $classes
	 * @param list<DisallowedCall> $disallowedCalls
	 * @return list<RuleError>
	 * @throws ShouldNotHappenException
	 */
	private function getRuleErrors(array $classes, MethodReflection $method, CallLike $node, Scope $scope, ?string $calledAs, array $disallowedCalls): array
	{
		foreach ($classes as $class) {
			if ($class->hasMethod($method->getName())) {
				$declaredAs = $this->formatter->getFullyQualified($class->getDisplayName(false), $method);
				$ruleErrors = $this->disallowedCallsRuleErrors->get($node, $scope, $declaredAs, $calledAs, $class->getFileName(), $disallowedCalls, ErrorIdentifiers::DISALLOWED_METHOD);
				if ($ruleErrors) {
					return $ruleErrors;
				}
			}
		}
		return [];
	}

}
