package gop1

import (
	"bufio"
	"errors"
	"io"
	"strings"
	"time"

	"github.com/tarm/serial"
)

const (
	defaultBaudrate = 115200
	defaultTimeout  = 500
	crcDelimiter    = '\x21' // hex char code for !
)

// P1 allows you to easily read from a P1-compatible serial device. The output is
// parsed into structured data
type P1 struct {
	serialDevice io.Reader
	Incoming     chan *Telegram
}

// P1Config is the configuration to create a new P1 object with
type P1Config struct {
	USBDevice string
	Baudrate  int
	Timeout   int // in milliseconds
}

// New returns a P1 object with given configuration or error when something went
// wrong initialising the serial object
func New(config P1Config) (*P1, error) {
	if config.Baudrate <= 0 {
		config.Baudrate = defaultBaudrate
	}

	if config.Timeout <= 0 {
		config.Timeout = defaultTimeout
	}

	serialConfig := &serial.Config{
		Name:        config.USBDevice,
		Baud:        config.Baudrate,
		ReadTimeout: time.Millisecond * time.Duration(config.Timeout),
	}

	serialDevice, err := serial.OpenPort(serialConfig)
	if err != nil {
		return nil, err
	}

	return &P1{
		serialDevice: serialDevice,
		Incoming:     make(chan *Telegram),
	}, nil
}

// Start makes P1 start reading data from the serial device
func (p *P1) Start() {
	go p.readData()
}

func (p *P1) readData() {
	// open reader to serial device
	reader := bufio.NewReader(p.serialDevice)

	for {
		// telegram data is suffixed with a CRC code
		// this CRC code starts with ! so let's read until we receive that char
		message, err := reader.ReadString(crcDelimiter)
		if err != nil {
			if errors.Is(err, io.EOF) {
				break
			}

			continue
		}

		/*
			// we're currently not processing the CRC itself, but let's read that as well
			// from the CRC, we assume the first carriage return is the end of the CRC
			_, err = reader.ReadString('\x0d') // hex char for CR
			if err != nil {
				continue
			}
		*/

		lines := strings.Split(message, "\n")
		p.Incoming <- parseTelegram(lines)
	}

	close(p.Incoming)
}

// Telegram represents the structured data for one complete dump (or telegram)
// of P1 data
type Telegram struct {
	Device  string
	Objects []*TelegramObject
}

// TelegramObject is the structured representation of a sinle line in a P1 data
// dump. It can have one or more values
type TelegramObject struct {
	Type   OBISType
	Values []TelegramValue
}

// TelegramValue is one value of a P1 data line, optionally with a specific unit
// of measurement
type TelegramValue struct {
	Value string
	Unit  string
}

// OBISType represents the specific Object Identification System type of a P1
// telegram line
type OBISType string

// These are the OBIS types currently supported
const (
	OBISTypeVersionInformation            = "Version Information"
	OBISTypeDateTimestamp                 = "Date timestamp"
	OBISTypeDeviceType                    = "Device Type"
	OBISTypeEquipmentIdentifier           = "Equipment Identifier"
	OBISTypeGasEquipmentIdentifier        = "Equipment Identifier (Gas)"
	OBISTypeElectricityDeliveredTariff1   = "Electricity delivered to client (tariff 1)"
	OBISTypeElectricityDeliveredTariff2   = "Electricity delivered to client (tariff 2)"
	OBISTypeElectricityGeneratedTariff1   = "Electricity generated by client (tariff 1)"
	OBISTypeElectricityGeneratedTariff2   = "Electricity generated by client (tariff 2)"
	OBISTypeElectricityTariffIndicator    = "Electricity tariff indicator"
	OBISTypeElectricityDelivered          = "Actual electricity delivered"
	OBISTypeElectricityGenerated          = "Actual electricity generated"
	OBISTypeNumberOfPowerFailures         = "Number of power failures on any phase"
	OBISTypeNumberOfLongPowerFailures     = "Number of long power failures on any phase"
	OBISTypePowerFailureEventLog          = "Event log for long power failures"
	OBISTypeNumberOfVoltageSagsL1         = "Number of voltage sags on phase L1"
	OBISTypeNumberOfVoltageSagsL2         = "Number of voltage sags on phase L2"
	OBISTypeNumberOfVoltageSagsL3         = "Number of voltage sags on phase L3"
	OBISTypeNumberOfVoltageSwellsL1       = "Number of voltage swells on phase L1"
	OBISTypeNumberOfVoltageSwellsL2       = "Number of voltage swells on phase L2"
	OBISTypeNumberOfVoltageSwellsL3       = "Number of voltage swells on phase L3"
	OBISTypeTextMessage                   = "Text message"
	OBISTypeInstantaneousVoltageL1        = "Instantaneous voltage on phase L1"
	OBISTypeInstantaneousVoltageL2        = "Instantaneous voltage on phase L2"
	OBISTypeInstantaneousVoltageL3        = "Instantaneous voltage on phase L3"
	OBISTypeInstantaneousCurrentL1        = "Instantaneous current on phase L1"
	OBISTypeInstantaneousCurrentL2        = "Instantaneous current on phase L2"
	OBISTypeInstantaneousCurrentL3        = "Instantaneous current on phase L3"
	OBISTypeInstantaneousPowerDeliveredL1 = "Instantaneous active power delivered on phase L1"
	OBISTypeInstantaneousPowerDeliveredL2 = "Instantaneous active power delivered on phase L2"
	OBISTypeInstantaneousPowerDeliveredL3 = "Instantaneous active power delivered on phase L3"
	OBISTypeInstantaneousPowerGeneratedL1 = "Instantaneous active power generated on phase L1"
	OBISTypeInstantaneousPowerGeneratedL2 = "Instantaneous active power generated on phase L2"
	OBISTypeInstantaneousPowerGeneratedL3 = "Instantaneous active power generated on phase L3"
	OBISTypeGasDelivered                  = "Actual gas delivered"
	OBISTypeConsumerMessageCode           = "Consumer message code"
	OBISTypeBreakerState                  = "Breaker state"
	OBISTypeLimiterThreshold              = "Electricity limiter threshold"
	OBISTypeFuseThresholdL1               = "Fuse threshold on phase L1"
	OBISTypeGasValveState                 = "Gas valve state"
)
