import { Channel } from '@sinclair/smoke'
import { Test, Assert } from '../test/index.mjs'

Test.describe('Channel:Select', () => {
  Test.it('Should select on single channel', () => {
    const channel = new Channel.Channel()
    Channel.select([channel])
  })
  Test.it('Should select on multiple channels', () => {
    const channel0 = new Channel.Channel()
    const channel1 = new Channel.Channel()
    Channel.select([channel0, channel1])
  })
  Test.it('Should receive select from single channel', async () => {
    const channel = new Channel.Channel()
    const select = Channel.select([channel])
    channel.send(0)
    channel.send(1)
    channel.send(2)
    channel.send(3)
    channel.end()
    const value0 = await select.next()
    const value1 = await select.next()
    const value2 = await select.next()
    const value3 = await select.next()
    const eof = await select.next()
    Assert.isEqual(value0, 0)
    Assert.isEqual(value1, 1)
    Assert.isEqual(value2, 2)
    Assert.isEqual(value3, 3)
    Assert.isEqual(eof, null)
  })
  Test.it('Should receive select from multiple channels', async () => {
    const channel0 = new Channel.Channel()
    const channel1 = new Channel.Channel()
    const channel2 = new Channel.Channel()
    const channel3 = new Channel.Channel()
    const select = Channel.select([channel0, channel1, channel2, channel3])
    channel0.send(0)
    channel1.send(1)
    channel2.send(2)
    channel3.send(3)

    channel0.send(4)
    channel1.send(5)
    channel2.send(6)
    channel3.send(7)

    channel0.end()
    channel1.end()
    channel2.end()
    channel3.end()
    const value0 = await select.next()
    const value1 = await select.next()
    const value2 = await select.next()
    const value3 = await select.next()
    const value4 = await select.next()
    const value5 = await select.next()
    const value6 = await select.next()
    const value7 = await select.next()
    const eof = await select.next()
    Assert.isEqual(value0, 0)
    Assert.isEqual(value1, 1)
    Assert.isEqual(value2, 2)
    Assert.isEqual(value3, 3)
    Assert.isEqual(value4, 4)
    Assert.isEqual(value5, 5)
    Assert.isEqual(value6, 6)
    Assert.isEqual(value7, 7)
    Assert.isEqual(eof, null)
  })
  Test.it('Should timeout if one of multiple channels never ends', async () => {
    const channel0 = new Channel.Channel()
    const channel1 = new Channel.Channel()
    const channel2 = new Channel.Channel()
    const channel3 = new Channel.Channel()
    const select = Channel.select([channel0, channel1, channel2, channel3])
    channel0.send(0)
    channel1.send(1)
    channel2.send(2)
    channel3.send(3)

    channel0.send(4)
    channel1.send(5)
    channel2.send(6)
    channel3.send(7)

    channel0.end()
    channel1.end()
    channel2.end()
    // channel3.end() <- no end
    await select.next()
    await select.next()
    await select.next()
    await select.next()
    await select.next()
    await select.next()
    await select.next()
    await select.next()
    await Assert.shouldTimeout(async () => await select.next(), { timeout: 100 })
  })
  Test.it('Should throw if single channel errors', async () => {
    const channel = new Channel.Channel()
    const select = Channel.select([channel])
    channel.send(0)
    channel.send(1)
    channel.error(new Error('error'))
    const value0 = await select.next()
    const value1 = await select.next()
    const value2 = await select.next().catch((error) => error)
    Assert.isEqual(value0, 0)
    Assert.isEqual(value1, 1)
    Assert.isInstanceOf(value2, Error)
  })
  Test.it('Should throw if one of multiple channels errors', async () => {
    const channel0 = new Channel.Channel()
    const channel1 = new Channel.Channel()
    const select = Channel.select([channel0, channel1])
    channel0.send(0)
    channel1.send(1)
    channel0.error(new Error('error'))
    const value0 = await select.next()
    const value1 = await select.next()
    const value2 = await select.next().catch((error) => error)
    Assert.isEqual(value0, 0)
    Assert.isEqual(value1, 1)
    Assert.isInstanceOf(value2, Error)
  })
})
