package com.sendbird.uikit.activities.adapter;

import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.sendbird.android.SendbirdChat;
import com.sendbird.android.message.Emoji;
import com.sendbird.android.message.Reaction;
import com.sendbird.uikit.activities.viewholder.BaseViewHolder;
import com.sendbird.uikit.databinding.SbViewEmojiBinding;
import com.sendbird.uikit.interfaces.OnItemClickListener;
import com.sendbird.uikit.internal.ui.reactions.EmojiView;
import com.sendbird.uikit.internal.ui.viewholders.EmojiMoreViewHolder;
import com.sendbird.uikit.internal.ui.viewholders.EmojiViewHolder;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Adapters provides a binding from an {@link Emoji} data set to views that are displayed within a RecyclerView.
 *
 * since 1.1.0
 */
public class EmojiListAdapter extends BaseAdapter<Emoji, BaseViewHolder<Emoji>> {
    private static final int VIEW_EMOJI = 0;
    private static final int VIEW_EMOJI_MORE = 1;

    @NonNull
    private final List<Emoji> emojiList;
    @NonNull
    private final Map<String, List<String>> reactionUserMap = new HashMap<>();
    @Nullable
    private OnItemClickListener<String> emojiClickListener;
    @Nullable
    private View.OnClickListener moreButtonClickListener;
    private final boolean showMoreButton;

    /**
     * Constructor
     *
     * @param emojiList The {@link List<Emoji>} that contains the data needed for this adapter
     * @param reactionList The {@link List<Reaction>} that contains the data needed for this adapter
     * @param showMoreButton <code>true</code> if the more button is showed,
     *                       <code>false</code> otherwise.
     * since 1.1.0
     */
    public EmojiListAdapter(@NonNull List<Emoji> emojiList,
                            @Nullable List<Reaction> reactionList,
                            boolean showMoreButton) {
        this.emojiList = emojiList;
        if (reactionList != null) {
            for (Reaction reaction : reactionList) {
                reactionUserMap.put(reaction.getKey(), reaction.getUserIds());
            }
        }
        this.showMoreButton = showMoreButton;
    }

    /**
     * Called when RecyclerView needs a new {@link BaseViewHolder<Emoji>} of the given type to represent
     * an item.
     *
     * @param parent The ViewGroup into which the new View will be added after it is bound to
     *               an adapter position.
     * @param viewType The view type of the new View.
     *
     * @return A new {@link BaseViewHolder<Emoji>} that holds a View of the given view type.
     * @see #getItemViewType(int)
     * @see #onBindViewHolder(BaseViewHolder, int)
     * since 1.1.0
     */
    @NonNull
    @Override
    public BaseViewHolder<Emoji> onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        LayoutInflater inflater = LayoutInflater.from(parent.getContext());
        if (viewType == VIEW_EMOJI_MORE) {
            return new EmojiMoreViewHolder(new EmojiView(parent.getContext()));
        } else {
            return new EmojiViewHolder(SbViewEmojiBinding.inflate(inflater, parent, false));
        }
    }

    /**
     * Called by RecyclerView to display the data at the specified position. This method should
     * update the contents of the {@link BaseViewHolder#itemView} to reflect the item at the given
     * position.
     *
     * @param holder The {@link BaseViewHolder<Emoji>} which should be updated to represent
     *               the contents of the item at the given position in the data set.
     * @param position The position of the item within the adapter's data set.
     * since 1.1.0
     */
    @Override
    public void onBindViewHolder(@NonNull BaseViewHolder<Emoji> holder, int position) {
        Emoji current = getItem(position);
        int type = getItemViewType(position);

        if (type == VIEW_EMOJI_MORE) {
            holder.itemView.setOnClickListener(v -> {
                if (moreButtonClickListener != null) {
                    moreButtonClickListener.onClick(v);
                }
            });
        } else {
            if (!reactionUserMap.isEmpty() && current != null) {
                List<String> userIds = reactionUserMap.get(current.getKey());
                holder.itemView.setSelected(userIds != null && SendbirdChat.getCurrentUser() != null && userIds.contains(SendbirdChat.getCurrentUser().getUserId()));
            }

            holder.itemView.setOnClickListener(v -> {
                Emoji emoji = getItem(holder.getBindingAdapterPosition());
                if (emojiClickListener != null && emoji != null) {
                    emojiClickListener.onItemClick(v,
                        holder.getBindingAdapterPosition(),
                        emoji.getKey());
                }
            });
        }

        if (current == null) return;
        holder.bind(current);
    }

    /**
     * Returns the total number of items in the data set held by the adapter.
     *
     * @return The total number of items in this adapter.
     * since 1.1.0
     */
    @Override
    public int getItemCount() {
        if (showMoreButton) {
            return emojiList.size() + 1;
        } else {
            return emojiList.size();
        }
    }

    /**
     * Returns the {@link Emoji} in the data set held by the adapter.
     *
     * @param position The position of the item within the adapter's data set.
     * @return The {@link Emoji} to retrieve the position of in this adapter.
     * since 1.1.0
     */
    @Override
    @Nullable
    public Emoji getItem(int position) {
        if (position >= emojiList.size()) {
            return null;
        }
        return emojiList.get(position);
    }

    /**
     * Returns the {@link List<Emoji>} in the data set held by the adapter.
     *
     * @return The {@link List<Emoji>} in this adapter.
     * since 1.1.0
     */
    @Override
    @NonNull
    public List<Emoji> getItems() {
        return emojiList;
    }

    /**
     * Return the view type of the {@link BaseViewHolder<Emoji>} at <code>position</code> for the purposes
     * of view recycling.
     *
     * @param position position to query
     * @return integer value identifying the type of the view needed to represent the item at <code>position</code>.
     * since 1.1.0
     */
    @Override
    public int getItemViewType(int position) {
        if (showMoreButton && position >= emojiList.size()) {
            return VIEW_EMOJI_MORE;
        } else {
            return VIEW_EMOJI;
        }
    }

    /**
     * Register a callback to be invoked when the emoji is clicked and held.
     *
     * @param emojiClickListener The callback that will run
     * since 1.1.0
     */
    public void setEmojiClickListener(@Nullable OnItemClickListener<String> emojiClickListener) {
        this.emojiClickListener = emojiClickListener;
    }

    /**
     * Register a callback to be invoked when the emoji more button is clicked and held.
     *
     * @param moreButtonClickListener The callback that will run
     * since 1.1.0
     */
    public void setMoreButtonClickListener(@Nullable View.OnClickListener moreButtonClickListener) {
        this.moreButtonClickListener = moreButtonClickListener;
    }
}
