QUnit.module('Data adapters - Select - current');

var SelectData = require('select2/data/select');
var $ = require('jquery');
var Options = require('select2/options');
var selectOptions = new Options({});

QUnit.test('current gets default for single', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  data.current(function (data) {
    assert.equal(
      data.length,
      1,
      'There should only be one selected option'
    );

    var option = data[0];

    assert.equal(
      option.id,
      'One',
      'The value of the option tag should be the id'
    );

    assert.equal(
      option.text,
      'One',
      'The text within the option tag should be the text'
    );
  });
});

QUnit.test('current gets default for multiple', function (assert) {
  var $select = $('#qunit-fixture .multiple');

  var data = new SelectData($select, selectOptions);

  data.current(function (data) {
    assert.equal(
      data.length,
      0,
      'Multiple selects have no default selection.'
    );
  });
});

QUnit.test('current gets options with explicit value', function (assert) {
  var $select = $('#qunit-fixture .single');

  var $option = $('<option value="1">One</option>');
  $select.append($option);

  var data = new SelectData($select, selectOptions);

  $select.val('1');

  data.current(function (data) {
    assert.equal(
      data.length,
      1,
      'There should be one selected option'
    );

    var option = data[0];

    assert.equal(
      option.id,
      '1',
      'The option value should be the selected id'
    );

    assert.equal(
      option.text,
      'One',
      'The text should match the text for the option tag'
    );
  });
});

QUnit.test('current gets options with implicit value', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  $select.val('One');

  data.current(function (val) {
    assert.equal(
      val.length,
      1,
      'There should only be one selected value'
    );

    var option = val[0];

    assert.equal(
      option.id,
      'One',
      'The id should be the same as the option text'
    );

    assert.equal(
      option.text,
      'One',
      'The text should be the same as the option text'
    );
  });
});

QUnit.test('select works for single', function (assert) {
  var $select = $('#qunit-fixture .single-with-placeholder');

  var data = new SelectData($select, selectOptions);

  assert.equal($select.val(), 'placeholder');

  data.select({
    id: 'One',
    text: 'One'
  });

  assert.equal($select.val(), 'One');
});

QUnit.test('multiple sets the value', function (assert) {
  var $select = $('#qunit-fixture .multiple');

  var data = new SelectData($select, selectOptions);

  assert.ok(
    $select.val() == null || $select.val().length == 0,
    'nothing should be selected'
  );

  data.select({
    id: 'Two',
    text: 'Two'
  });

  assert.deepEqual($select.val(), ['Two']);
});

QUnit.test('multiple adds to the old value', function (assert) {
  var $select = $('#qunit-fixture .multiple');

  var data = new SelectData($select, selectOptions);

  $select.val(['Two']);

  assert.deepEqual($select.val(), ['Two']);

  data.select({
    id: 'One',
    text: 'One'
  });

  assert.deepEqual($select.val(), ['One', 'Two']);
});

QUnit.test('duplicates - single - same id on select triggers change',
  function (assert) {
  var $select = $('#qunit-fixture .duplicates');

  var data = new SelectData($select, data);
  var second = $('#qunit-fixture .duplicates option')[2];

  var changeTriggered = false, inputTriggered = false;

  assert.equal($select.val(), 'one');

  $select.on('change', function () {
    changeTriggered = inputTriggered;
  }).on('input', function() {
    inputTriggered = true;
  });

  data.select({
    id: 'one',
    text: 'Uno',
    element: second
  });

  assert.equal(
    $select.val(),
    'one',
    'The value never changed'
  );

  assert.ok(
    inputTriggered,
    'The input event should be triggered'
  );

  assert.ok(
    changeTriggered,
    'The change event should be triggered after the input event'
  );

  assert.ok(
    second.selected,
    'The second duplicate is selected, not the first'
  );
});

QUnit.test('duplicates - single - different id on select triggers change',
  function (assert) {
  var $select = $('#qunit-fixture .duplicates');

  var data = new SelectData($select, data);
  var second = $('#qunit-fixture .duplicates option')[2];

  var changeTriggered = false, inputTriggered = false;

  $select.val('two');

  $select.on('change', function () {
    changeTriggered = inputTriggered;
  }).on('input', function() {
    inputTriggered = true;
  });

  data.select({
    id: 'one',
    text: 'Uno',
    element: second
  });

  assert.equal(
    $select.val(),
    'one',
    'The value changed to the duplicate id'
  );

  assert.ok(
    inputTriggered,
    'The input event should be triggered'
  );

  assert.ok(
    changeTriggered,
    'The change event should be triggered after the input event'
  );

  assert.ok(
    second.selected,
    'The second duplicate is selected, not the first'
  );
});

QUnit.test('duplicates - multiple - same id on select triggers change',
function (assert) {
  var $select = $('#qunit-fixture .duplicates-multi');

  var data = new SelectData($select, data);
  var second = $('#qunit-fixture .duplicates-multi option')[2];

  var changeTriggered = false, inputTriggered = false;

  $select.val(['one']);

  $select.on('change', function () {
    changeTriggered = inputTriggered;
  }).on('input', function() {
    inputTriggered = true;
  });

  data.select({
    id: 'one',
    text: 'Uno',
    element: second
  });

  assert.deepEqual(
    $select.val(),
    ['one', 'one'],
    'The value now has duplicates'
  );

  assert.ok(
    inputTriggered,
    'The input event should be triggered'
  );

  assert.ok(
    changeTriggered,
    'The change event should be triggered after the input event'
  );

  assert.ok(
    second.selected,
    'The second duplicate is selected, not the first'
  );
});

QUnit.test('duplicates - multiple - different id on select triggers change',
function (assert) {
  var $select = $('#qunit-fixture .duplicates-multi');

  var data = new SelectData($select, data);
  var second = $('#qunit-fixture .duplicates-multi option')[2];

  var changeTriggered = false, inputTriggered = false;

  $select.val(['two']);

  $select.on('change', function () {
    changeTriggered = inputTriggered;
  }).on('input', function() {
    inputTriggered = true;
  });

  data.select({
    id: 'one',
    text: 'Uno',
    element: second
  });

  assert.deepEqual(
    $select.val(),
    ['two', 'one'],
    'The value has the new id'
  );

  assert.ok(
    inputTriggered,
    'The input event should be triggered'
  );

  assert.ok(
    changeTriggered,
    'The change event should be triggered after the input event'
  );

  assert.ok(
    second.selected,
    'The second duplicate is selected, not the first'
  );
});

QUnit.module('Data adapter - Select - query');

QUnit.test('all options are returned with no term', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  data.query({}, function (data) {
    assert.equal(
      data.results.length,
      1,
      'The number of items returned should be equal to the number of options'
    );
  });
});

QUnit.test('the matcher checks the text', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  data.query({
    term: 'One'
  }, function (data) {
    assert.equal(
      data.results.length,
      1,
      'Only the "One" option should be found'
    );
  });
});

QUnit.test('the matcher ignores case', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  data.query({
    term: 'one'
  }, function (data) {
    assert.equal(
      data.results.length,
      1,
      'The "One" option should still be found'
    );
  });
});

QUnit.test('no options may be returned with no matches', function (assert) {
  var $select = $('#qunit-fixture .single');

  var data = new SelectData($select, selectOptions);

  data.query({
    term: 'qwerty'
  }, function (data) {
    assert.equal(
      data.results.length,
      0,
      'Only matching items should be returned'
    );
  });
});

QUnit.test('optgroup tags are marked with children', function (assert) {
  var $select = $('#qunit-fixture .groups');

  var data = new SelectData($select, selectOptions);

  data.query({}, function (data) {
    assert.ok(
      'children' in data.results[0],
      'The optgroup element should have children when queried'
    );
  });
});

QUnit.test('empty optgroups are still shown when queried', function (assert) {
  var $select = $('#qunit-fixture .groups');

  var data = new SelectData($select, selectOptions);

  data.query({}, function (data) {
    assert.equal(
      data.results.length,
      2,
      'The empty optgroup element should still be returned when queried'
    );

    var item = data.results[1];

    assert.equal(
      item.text,
      'Empty',
      'The text of the empty optgroup should match the label'
    );

    assert.equal(
      item.children.length,
      0,
      'There should be no children in the empty opgroup'
    );
  });
});

QUnit.test('multiple options with the same value are returned', function (assert) {
  var $select = $('#qunit-fixture .duplicates');

  var data = new SelectData($select, selectOptions);

  data.query({}, function (data) {
    assert.equal(
      data.results.length,
      3,
      'The duplicate option should still be returned when queried'
    );

    var first = data.results[0];
    var duplicate = data.results[2];

    assert.equal(
      first.id,
      duplicate.id,
      'The duplicates should have the same id'
    );

    assert.notEqual(
      first.text,
      duplicate.text,
      'The duplicates do not have the same text'
    );
  });
});

QUnit.test('data objects use the text of the option', function (assert) {
  var $select = $('#qunit-fixture .duplicates');

  var data = new SelectData($select, selectOptions);

  var $option = $('<option>&amp;</option>');

  var item = data.item($option);

  assert.equal(item.id, '&');
  assert.equal(item.text, '&');
});

QUnit.test('select option construction accepts id=0 (zero) value', function (assert) {
  var $select = $('#qunit-fixture .single');

  var selectOptions = [{ id: 0, text: 'Zero Value'}];
  var data = new SelectData($select, selectOptions);

  var optionElem = data.option(selectOptions[0]);

  // If was "Zero Value"", then it ignored id property
  assert.equal(
    optionElem[0].value,
    '0',
    'Built option value should be "0" (zero as a string).'
  );
});

QUnit.test('select option construction accepts id="" (empty string) value',
  function (assert) {
  var $select = $('#qunit-fixture .single');

  var selectOptions = [{ id: '', text: 'Empty String'}];
  var data = new SelectData($select, selectOptions);

  var optionElem = data.option(selectOptions[0]);

  assert.equal(
    optionElem[0].value,
    '',
    'Built option value should be an empty string.'
  );
});

QUnit.test('user-defined types are normalized properly', function (assert) {
  var $select = $('#qunit-fixture .user-defined'),

  UserDefinedType = function (id, text) {
    var self = this;

    self.id = id;
    self.text = text;

    return self;
  };

  var testData = [
    'Test',
    {
        id: 4,
        text: 'item'
    },
    new UserDefinedType(1, 'aaaaaa')
  ];

  var data = new SelectData($select, selectOptions);

  var normalizedItem = data._normalizeItem(testData[0]);
  var normalizedItem2 = data._normalizeItem(testData[1]);
  var normalizedItem3 = data._normalizeItem(testData[2]);

  assert.equal(
    testData[0],
    normalizedItem.id,
    'id property should be equal to text after normalize'
  );

  assert.equal(
    testData[0],
    normalizedItem.text,
    'text property should be equal after normalize'
  );

  assert.equal(
    testData[1].id,
    normalizedItem2.id,
    'id property should be equal after normalize'
  );

  assert.equal(
    testData[1].text,
    normalizedItem2.text,
    'text property should be equal after normalize'
  );

  assert.equal(
    testData[2].id,
    normalizedItem3.id,
    'id property should be equal after normalize'
  );

  assert.equal(
    testData[2].text,
    normalizedItem3.text,
    'text property should be equal after normalize'
  );

});
