package pe

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"io"
	"microscope/utils"
	"strings"
)

func readImports(is64bit bool, virtualAddress uint32) {

	// Ottieni la sezione che corrisponde al virtualAddress importato
	section := getSectionFromVirtualAddress(uint64(virtualAddress))

	if section == nil {
		fmt.Println("Non è stato possibile ritrovare la sezione Import")
		return
	}

	// Calcola la tableOffset
	tableOffset := virtualAddress - section.VirtualAddress

	// Crea un nuovo lettore dati
	readerSection := bytes.NewReader(section.Raw)

	if readerSection == nil {
		fmt.Println("Impossibile creare un nuovo reader per il contenuto della sezione.")
		return
	}

	// Per ogni import DLL
	// Scorri la tableOffset nelle righe
	for i := tableOffset; ; i += uint32(binary.Size(ImportDirectory{})) {

		_, err := readerSection.Seek(int64(i), io.SeekStart)
		if err != nil {
			fmt.Println("Impossibile impostare l'offset per il seguente motivo : " + err.Error())
			return
		}

		importDirectory := ImportDirectory{}

		err = binary.Read(readerSection, binary.LittleEndian, &importDirectory)
		if err != nil {
			fmt.Println("Non è stato possibile leggere l'importDirectory per il seguente motivo " + err.Error())
			return
		}

		if importDirectory.NameRVA == 0 {
			break
		}

		requiredSection := getSectionFromVirtualAddress(uint64(importDirectory.NameRVA))

		if requiredSection == nil {
			fmt.Println("Non è stato possibile trovare la sezione dal relative address del nome della DLL. Binario malformato.")
			return
		}

		// Parsa il raw per trovare la prima stringa
		dllName := strings.ToLower(utils.ReadString(requiredSection.Raw[importDirectory.NameRVA-requiredSection.VirtualAddress:]))

		var padding int
		var bitMaskedFlag uint64

		if is64bit {
			padding = 8
			bitMaskedFlag = 0x8000000000000000
		} else {
			padding = 4
			bitMaskedFlag = 0x80000000
		}

		// prendi una sezione
		section = getSectionFromVirtualAddress(uint64(importDirectory.ImportAddressTableRVA))

		if section == nil {
			fmt.Println("Impossibile trovare la sezione")
			return
		}

		thunk2 := importDirectory.ImportAddressTableRVA
		importThunk := 0

		if importDirectory.ImportLookupTableRVA > section.VirtualAddress {
			importThunk = int(importDirectory.ImportLookupTableRVA - section.VirtualAddress)
		} else {
			importThunk = int(importDirectory.ImportAddressTableRVA - section.VirtualAddress)
		}

		for importThunk+padding < len(section.Raw) {
			// Recupera
			thunk1 := binary.LittleEndian.Uint32(section.Raw[importThunk : importThunk+padding])
			if thunk1 == 0 {
				break
			}
			var funcName string

			// flag che specifica la modalità d'import (prelevata dalla ILT) Import Lookup Table
			// doOrdinal = true: le import sono ordinate per un numero
			// doOrdinal = false: le import sono ordinate per la funcName
			doOrdinal := uint64(thunk1)&bitMaskedFlag > 0
			if doOrdinal {
				// Fai il parsing in ordine di numero
				ord := uint16(thunk1 & 0xffff)

				// Traduci il numero in una funzione con il nome (se esiste)
				funcName = translateOrdinal(dllName, ord)
				thunk2 += uint32(padding)
			} else {
				sectionTmp := getSectionFromVirtualAddress(uint64(thunk1) + 2)
				if sectionTmp != nil {
					v := uint32(thunk1) + 2 - sectionTmp.VirtualAddress
					funcName = utils.ReadString(sectionTmp.Raw[v:])
					thunk2 += uint32(padding)
				}

			}

			importElement := &ImportInfo{DllName: dllName, APICalled: funcName, Offset: tableOffset, Thunk: thunk1}
			fileAnalyzed.Imports = append(fileAnalyzed.Imports, importElement)

			// Continua il ciclo aggiungendo il padding
			importThunk += padding

		}
	}
}

func translateOrdinal(dllName string, ord uint16) string {

	switch dllName {
	case "ws2_32.dll":
		return translateOrdinalWS32(ord)
	case "oleaut32.dll":
		return translateOrdinalOLE32(ord)
	case "comctl32.dll":
		return translateOrdinalComCTL32(ord)
	}
	return ""
}

func translateOrdinalComCTL32(ord uint16) string {
	switch ord {
	case 2:
		return "imp_MenuHelp"
	case 3:
		return "imp_ShowHideMenuCtl"
	case 4:
		return "imp_GetEffectiveClientRect"
	case 5:
		return "imp_DrawStatusTextA"
	case 6:
		return "imp_CreateStatusWindowA"
	case 7:
		return "imp_CreateToolbar"
	case 8:
		return "imp_CreateMappedBitmap"
	case 9:
		return "imp_DPA_LoadStream"
	case 10:
		return "imp_DPA_SaveStream"
	case 11:
		return "imp_DPA_Merge"
	case 12:
		return "imp_CreatePropertySheetPage"
	case 13:
		return "imp_MakeDragList"
	case 14:
		return "imp_LBItemFromPt"
	case 15:
		return "imp_DrawInsert"
	case 16:
		return "imp_CreateUpDownControl"
	case 17:
		return "imp_InitCommonControls"
	case 18:
		return "imp_CreatePropertySheetPageA"
	case 19:
		return "imp_CreatePropertySheetPageW"
	case 20:
		return "imp_CreateStatusWindow"
	case 21:
		return "imp_CreateStatusWindowW"
	case 22:
		return "imp_CreateToolbarEx"
	case 23:
		return "imp_DestroyPropertySheetPage"
	case 24:
		return "imp_DllGetVersion"
	case 25:
		return "imp_DrawStatusText"
	case 26:
		return "imp_DrawStatusTextW"
	case 27:
		return "imp_FlatSB_EnableScrollBar"
	case 28:
		return "imp_FlatSB_GetScrollInfo"
	case 29:
		return "imp_FlatSB_GetScrollPos"
	case 30:
		return "imp_FlatSB_GetScrollProp"
	case 31:
		return "imp_FlatSB_GetScrollPropPtr"
	case 32:
		return "imp_FlatSB_GetScrollRange"
	case 33:
		return "imp_FlatSB_SetScrollInfo"
	case 34:
		return "imp_FlatSB_SetScrollPos"
	case 35:
		return "imp_FlatSB_SetScrollProp"
	case 36:
		return "imp_FlatSB_SetScrollRange"
	case 37:
		return "imp_FlatSB_ShowScrollBar"
	case 38:
		return "imp_GetMUILanguage"
	case 39:
		return "imp_ImageList_Add"
	case 40:
		return "imp_ImageList_AddIcon"
	case 41:
		return "imp_ImageList_AddMasked"
	case 42:
		return "imp_ImageList_BeginDrag"
	case 43:
		return "imp_ImageList_Copy"
	case 44:
		return "imp_ImageList_Create"
	case 45:
		return "imp_ImageList_Destroy"
	case 46:
		return "imp_ImageList_DragEnter"
	case 47:
		return "imp_ImageList_DragLeave"
	case 48:
		return "imp_ImageList_DragMove"
	case 49:
		return "imp_ImageList_DragShowNolock"
	case 50:
		return "imp_ImageList_Draw"
	case 51:
		return "imp_ImageList_DrawEx"
	case 52:
		return "imp_ImageList_DrawIndirect"
	case 53:
		return "imp_ImageList_Duplicate"
	case 54:
		return "imp_ImageList_EndDrag"
	case 55:
		return "imp_ImageList_GetBkColor"
	case 56:
		return "imp_ImageList_GetDragImage"
	case 57:
		return "imp_ImageList_GetFlags"
	case 58:
		return "imp_ImageList_GetIcon"
	case 59:
		return "imp_ImageList_GetIconSize"
	case 60:
		return "imp_ImageList_GetImageCount"
	case 61:
		return "imp_ImageList_GetImageInfo"
	case 62:
		return "imp_ImageList_GetImageRect"
	case 63:
		return "imp_ImageList_LoadImage"
	case 64:
		return "imp_ImageList_LoadImageA"
	case 65:
		return "imp_ImageList_LoadImageW"
	case 66:
		return "imp_ImageList_Merge"
	case 67:
		return "imp_ImageList_Read"
	case 68:
		return "imp_ImageList_Remove"
	case 69:
		return "imp_ImageList_Replace"
	case 70:
		return "imp_ImageList_ReplaceIcon"
	case 71:
	case 72:
	case 73:
	case 74:
		return "imp_None"
	case 75:
		return "imp_ImageList_SetBkColor"
	case 76:
		return "imp_ImageList_SetDragCursorImage"
	case 77:
		return "imp_ImageList_SetFilter"
	case 78:
		return "imp_ImageList_SetFlags"
	case 79:
		return "imp_ImageList_SetIconSize"
	case 80:
		return "imp_ImageList_SetImageCount"
	case 81:
		return "imp_ImageList_SetOverlayImage"
	case 82:
		return "imp_ImageList_Write"
	case 83:
		return "imp_InitCommonControlsEx"
	case 84:
		return "imp_InitMUILanguage"
	case 85:
		return "imp_InitializeFlatSB"
	case 86:
		return "imp_PropertySheet"
	case 87:
		return "imp_PropertySheetA"
	case 88:
		return "imp_PropertySheetW"
	case 89:
		return "imp_RegisterClassNameW"
	case 90:
		return "imp_UninitializeFlatSB"
	case 91:
		return "imp__TrackMouseEvent"
	case 151:
		return "imp_None"
	case 152:
		return "imp_FreeMRUList"
	case 153:
		return "imp_None"
	case 154:
		return "imp_None"
	case 155:
		return "imp_None"
	case 156:
		return "imp_None"
	case 157:
		return "imp_None"
	case 163:
		return "imp_None"
	case 164:
		return "imp_None"
	case 167:
		return "imp_None"
	case 169:
		return "imp_None"
	case 233:
		return "imp_None"
	case 234:
		return "imp_None"
	case 235:
		return "imp_None"
	case 236:
		return "imp_Str_SetPtrW"
	case 320:
		return "imp_DSA_Create"
	case 321:
		return "imp_DSA_Destroy"
	case 322:
		return "imp_DSA_GetItem"
	case 323:
		return "imp_DSA_GetItemPtr"
	case 324:
		return "imp_DSA_InsertItem"
	case 325:
		return "imp_DSA_SetItem"
	case 326:
		return "imp_DSA_DeleteItem"
	case 327:
		return "imp_DSA_DeleteAllItems"
	case 328:
		return "imp_DPA_Create"
	case 329:
		return "imp_DPA_Destroy"
	case 330:
		return "imp_DPA_Grow"
	case 331:
		return "imp_DPA_Clone"
	case 332:
		return "imp_DPA_GetPtr"
	case 333:
		return "imp_DPA_GetPtrIndex"
	case 334:
		return "imp_DPA_InsertPtr"
	case 335:
		return "imp_DPA_SetPtr"
	case 336:
		return "imp_DPA_DeletePtr"
	case 337:
		return "imp_DPA_DeleteAllPtrs"
	case 338:
		return "imp_DPA_Sort"
	case 339:
		return "imp_DPA_Search"
	case 340:
		return "imp_DPA_CreateEx"
	case 341:
	case 342:
	case 350:
	case 351:
	case 352:
	case 353:
	case 354:
	case 355:
	case 356:
	case 357:
	case 358:
	case 359:
	case 360:
	case 361:
	case 362:
	case 363:
	case 364:
	case 365:
	case 366:
	case 367:
	case 368:
	case 369:
	case 372:
	case 373:
	case 374:
	case 375:
	case 376:
	case 377:
	case 382:
	case 383:
	case 384:
		return "imp_None"
	case 385:
		return "imp_DPA_EnumCallback"
	case 386:
		return "imp_DPA_DestroyCallback"
	case 387:
		return "imp_DSA_EnumCallback"
	case 388:
		return "imp_DSA_DestroyCallback"
	case 389:
		return "imp_None"
	case 390:
		return "imp_None"
	case 400:
		return "imp_CreateMRUListW"
	case 401:
		return "imp_AddMRUStringW"
	case 402:
		return "imp_None"
	case 403:
		return "imp_EnumMRUListW"
	case 404:
		return "imp_None"
	case 410:
		return "imp_SetWindowSubclass"
	case 411:
		return "imp_None"
	case 412:
		return "imp_RemoveWindowSubclass"
	case 413:
		return "imp_DefSubclassProc"
	case 414:
		return "imp_None"
	case 415:
		return "imp_None"
	case 416:
		return "imp_None"
	case 417:
		return "imp_None"
	case 418:
		return "imp_None"
	case 419:
		return "imp_None"
	case 420:
		return "imp_None"
	case 421:
		return "imp_None"
	}
	return ""
}

func translateOrdinalOLE32(ord uint16) string {
	switch ord {
	case 2:
		return "imp_SysAllocString"
	case 3:
		return "imp_SysReAllocString"
	case 4:
		return "imp_SysAllocStringLen"
	case 5:
		return "imp_SysReAllocStringLen"
	case 6:
		return "imp_SysFreeString"
	case 7:
		return "imp_SysStringLen"
	case 8:
		return "imp_VariantInit"
	case 9:
		return "imp_VariantClear"
	case 10:
		return "imp_VariantCopy"
	case 11:
		return "imp_VariantCopyInd"
	case 12:
		return "imp_VariantChangeType"
	case 13:
		return "imp_VariantTimeToDosDateTime"
	case 14:
		return "imp_DosDateTimeToVariantTime"
	case 15:
		return "imp_SafeArrayCreate"
	case 16:
		return "imp_SafeArrayDestroy"
	case 17:
		return "imp_SafeArrayGetDim"
	case 18:
		return "imp_SafeArrayGetElemsize"
	case 19:
		return "imp_SafeArrayGetUBound"
	case 20:
		return "imp_SafeArrayGetLBound"
	case 21:
		return "imp_SafeArrayLock"
	case 22:
		return "imp_SafeArrayUnlock"
	case 23:
		return "imp_SafeArrayAccessData"
	case 24:
		return "imp_SafeArrayUnaccessData"
	case 25:
		return "imp_SafeArrayGetElement"
	case 26:
		return "imp_SafeArrayPutElement"
	case 27:
		return "imp_SafeArrayCopy"
	case 28:
		return "imp_DispGetParam"
	case 29:
		return "imp_DispGetIDsOfNames"
	case 30:
		return "imp_DispInvoke"
	case 31:
		return "imp_CreateDispTypeInfo"
	case 32:
		return "imp_CreateStdDispatch"
	case 33:
		return "imp_RegisterActiveObject"
	case 34:
		return "imp_RevokeActiveObject"
	case 35:
		return "imp_GetActiveObject"
	case 36:
		return "imp_SafeArrayAllocDescriptor"
	case 37:
		return "imp_SafeArrayAllocData"
	case 38:
		return "imp_SafeArrayDestroyDescriptor"
	case 39:
		return "imp_SafeArrayDestroyData"
	case 40:
		return "imp_SafeArrayRedim"
	case 41:
		return "imp_SafeArrayAllocDescriptorEx"
	case 42:
		return "imp_SafeArrayCreateEx"
	case 43:
		return "imp_SafeArrayCreateVectorEx"
	case 44:
		return "imp_SafeArraySetRecordInfo"
	case 45:
		return "imp_SafeArrayGetRecordInfo"
	case 46:
		return "imp_VarParseNumFromStr"
	case 47:
		return "imp_VarNumFromParseNum"
	case 48:
		return "imp_VarI2FromUI1"
	case 49:
		return "imp_VarI2FromI4"
	case 50:
		return "imp_VarI2FromR4"
	case 51:
		return "imp_VarI2FromR8"
	case 52:
		return "imp_VarI2FromCy"
	case 53:
		return "imp_VarI2FromDate"
	case 54:
		return "imp_VarI2FromStr"
	case 55:
		return "imp_VarI2FromDisp"
	case 56:
		return "imp_VarI2FromBool"
	case 57:
		return "imp_SafeArraySetIID"
	case 58:
		return "imp_VarI4FromUI1"
	case 59:
		return "imp_VarI4FromI2"
	case 60:
		return "imp_VarI4FromR4"
	case 61:
		return "imp_VarI4FromR8"
	case 62:
		return "imp_VarI4FromCy"
	case 63:
		return "imp_VarI4FromDate"
	case 64:
		return "imp_VarI4FromStr"
	case 65:
		return "imp_VarI4FromDisp"
	case 66:
		return "imp_VarI4FromBool"
	case 67:
		return "imp_SafeArrayGetIID"
	case 68:
		return "imp_VarR4FromUI1"
	case 69:
		return "imp_VarR4FromI2"
	case 70:
		return "imp_VarR4FromI4"
	case 71:
		return "imp_VarR4FromR8"
	case 72:
		return "imp_VarR4FromCy"
	case 73:
		return "imp_VarR4FromDate"
	case 74:
		return "imp_VarR4FromStr"
	case 75:
		return "imp_VarR4FromDisp"
	case 76:
		return "imp_VarR4FromBool"
	case 77:
		return "imp_SafeArrayGetVartype"
	case 78:
		return "imp_VarR8FromUI1"
	case 79:
		return "imp_VarR8FromI2"
	case 80:
		return "imp_VarR8FromI4"
	case 81:
		return "imp_VarR8FromR4"
	case 82:
		return "imp_VarR8FromCy"
	case 83:
		return "imp_VarR8FromDate"
	case 84:
		return "imp_VarR8FromStr"
	case 85:
		return "imp_VarR8FromDisp"
	case 86:
		return "imp_VarR8FromBool"
	case 87:
		return "imp_VarFormat"
	case 88:
		return "imp_VarDateFromUI1"
	case 89:
		return "imp_VarDateFromI2"
	case 90:
		return "imp_VarDateFromI4"
	case 91:
		return "imp_VarDateFromR4"
	case 92:
		return "imp_VarDateFromR8"
	case 93:
		return "imp_VarDateFromCy"
	case 94:
		return "imp_VarDateFromStr"
	case 95:
		return "imp_VarDateFromDisp"
	case 96:
		return "imp_VarDateFromBool"
	case 97:
		return "imp_VarFormatDateTime"
	case 98:
		return "imp_VarCyFromUI1"
	case 99:
		return "imp_VarCyFromI2"
	case 100:
		return "imp_VarCyFromI4"
	case 101:
		return "imp_VarCyFromR4"
	case 102:
		return "imp_VarCyFromR8"
	case 103:
		return "imp_VarCyFromDate"
	case 104:
		return "imp_VarCyFromStr"
	case 105:
		return "imp_VarCyFromDisp"
	case 106:
		return "imp_VarCyFromBool"
	case 107:
		return "imp_VarFormatNumber"
	case 108:
		return "imp_VarBstrFromUI1"
	case 109:
		return "imp_VarBstrFromI2"
	case 110:
		return "imp_VarBstrFromI4"
	case 111:
		return "imp_VarBstrFromR4"
	case 112:
		return "imp_VarBstrFromR8"
	case 113:
		return "imp_VarBstrFromCy"
	case 114:
		return "imp_VarBstrFromDate"
	case 115:
		return "imp_VarBstrFromDisp"
	case 116:
		return "imp_VarBstrFromBool"
	case 117:
		return "imp_VarFormatPercent"
	case 118:
		return "imp_VarBoolFromUI1"
	case 119:
		return "imp_VarBoolFromI2"
	case 120:
		return "imp_VarBoolFromI4"
	case 121:
		return "imp_VarBoolFromR4"
	case 122:
		return "imp_VarBoolFromR8"
	case 123:
		return "imp_VarBoolFromDate"
	case 124:
		return "imp_VarBoolFromCy"
	case 125:
		return "imp_VarBoolFromStr"
	case 126:
		return "imp_VarBoolFromDisp"
	case 127:
		return "imp_VarFormatCurrency"
	case 128:
		return "imp_VarWeekdayName"
	case 129:
		return "imp_VarMonthName"
	case 130:
		return "imp_VarUI1FromI2"
	case 131:
		return "imp_VarUI1FromI4"
	case 132:
		return "imp_VarUI1FromR4"
	case 133:
		return "imp_VarUI1FromR8"
	case 134:
		return "imp_VarUI1FromCy"
	case 135:
		return "imp_VarUI1FromDate"
	case 136:
		return "imp_VarUI1FromStr"
	case 137:
		return "imp_VarUI1FromDisp"
	case 138:
		return "imp_VarUI1FromBool"
	case 139:
		return "imp_VarFormatFromTokens"
	case 140:
		return "imp_VarTokenizeFormatString"
	case 141:
		return "imp_VarAdd"
	case 142:
		return "imp_VarAnd"
	case 143:
		return "imp_VarDiv"
	case 144:
		return "imp_BSTR_UserFree64"
	case 145:
		return "imp_BSTR_UserMarshal64"
	case 146:
		return "imp_DispCallFunc"
	case 147:
		return "imp_VariantChangeTypeEx"
	case 148:
		return "imp_SafeArrayPtrOfIndex"
	case 149:
		return "imp_SysStringByteLen"
	case 150:
		return "imp_SysAllocStringByteLen"
	case 151:
		return "imp_BSTR_UserSize64"
	case 152:
		return "imp_VarEqv"
	case 153:
		return "imp_VarIdiv"
	case 154:
		return "imp_VarImp"
	case 155:
		return "imp_VarMod"
	case 156:
		return "imp_VarMul"
	case 157:
		return "imp_VarOr"
	case 158:
		return "imp_VarPow"
	case 159:
		return "imp_VarSub"
	case 160:
		return "imp_CreateTypeLib"
	case 161:
		return "imp_LoadTypeLib"
	case 162:
		return "imp_LoadRegTypeLib"
	case 163:
		return "imp_RegisterTypeLib"
	case 164:
		return "imp_QueryPathOfRegTypeLib"
	case 165:
		return "imp_LHashValOfNameSys"
	case 166:
		return "imp_LHashValOfNameSysA"
	case 167:
		return "imp_VarXor"
	case 168:
		return "imp_VarAbs"
	case 169:
		return "imp_VarFix"
	case 170:
		return "imp_OaBuildVersion"
	case 171:
		return "imp_ClearCustData"
	case 172:
		return "imp_VarInt"
	case 173:
		return "imp_VarNeg"
	case 174:
		return "imp_VarNot"
	case 175:
		return "imp_VarRound"
	case 176:
		return "imp_VarCmp"
	case 177:
		return "imp_VarDecAdd"
	case 178:
		return "imp_VarDecDiv"
	case 179:
		return "imp_VarDecMul"
	case 180:
		return "imp_CreateTypeLib2"
	case 181:
		return "imp_VarDecSub"
	case 182:
		return "imp_VarDecAbs"
	case 183:
		return "imp_LoadTypeLibEx"
	case 184:
		return "imp_SystemTimeToVariantTime"
	case 185:
		return "imp_VariantTimeToSystemTime"
	case 186:
		return "imp_UnRegisterTypeLib"
	case 187:
		return "imp_VarDecFix"
	case 188:
		return "imp_VarDecInt"
	case 189:
		return "imp_VarDecNeg"
	case 190:
		return "imp_VarDecFromUI1"
	case 191:
		return "imp_VarDecFromI2"
	case 192:
		return "imp_VarDecFromI4"
	case 193:
		return "imp_VarDecFromR4"
	case 194:
		return "imp_VarDecFromR8"
	case 195:
		return "imp_VarDecFromDate"
	case 196:
		return "imp_VarDecFromCy"
	case 197:
		return "imp_VarDecFromStr"
	case 198:
		return "imp_VarDecFromDisp"
	case 199:
		return "imp_VarDecFromBool"
	case 200:
		return "imp_GetErrorInfo"
	case 201:
		return "imp_SetErrorInfo"
	case 202:
		return "imp_CreateErrorInfo"
	case 203:
		return "imp_VarDecRound"
	case 204:
		return "imp_VarDecCmp"
	case 205:
		return "imp_VarI2FromI1"
	case 206:
		return "imp_VarI2FromUI2"
	case 207:
		return "imp_VarI2FromUI4"
	case 208:
		return "imp_VarI2FromDec"
	case 209:
		return "imp_VarI4FromI1"
	case 210:
		return "imp_VarI4FromUI2"
	case 211:
		return "imp_VarI4FromUI4"
	case 212:
		return "imp_VarI4FromDec"
	case 213:
		return "imp_VarR4FromI1"
	case 214:
		return "imp_VarR4FromUI2"
	case 215:
		return "imp_VarR4FromUI4"
	case 216:
		return "imp_VarR4FromDec"
	case 217:
		return "imp_VarR8FromI1"
	case 218:
		return "imp_VarR8FromUI2"
	case 219:
		return "imp_VarR8FromUI4"
	case 220:
		return "imp_VarR8FromDec"
	case 221:
		return "imp_VarDateFromI1"
	case 222:
		return "imp_VarDateFromUI2"
	case 223:
		return "imp_VarDateFromUI4"
	case 224:
		return "imp_VarDateFromDec"
	case 225:
		return "imp_VarCyFromI1"
	case 226:
		return "imp_VarCyFromUI2"
	case 227:
		return "imp_VarCyFromUI4"
	case 228:
		return "imp_VarCyFromDec"
	case 229:
		return "imp_VarBstrFromI1"
	case 230:
		return "imp_VarBstrFromUI2"
	case 231:
		return "imp_VarBstrFromUI4"
	case 232:
		return "imp_VarBstrFromDec"
	case 233:
		return "imp_VarBoolFromI1"
	case 234:
		return "imp_VarBoolFromUI2"
	case 235:
		return "imp_VarBoolFromUI4"
	case 236:
		return "imp_VarBoolFromDec"
	case 237:
		return "imp_VarUI1FromI1"
	case 238:
		return "imp_VarUI1FromUI2"
	case 239:
		return "imp_VarUI1FromUI4"
	case 240:
		return "imp_VarUI1FromDec"
	case 241:
		return "imp_VarDecFromI1"
	case 242:
		return "imp_VarDecFromUI2"
	case 243:
		return "imp_VarDecFromUI4"
	case 244:
		return "imp_VarI1FromUI1"
	case 245:
		return "imp_VarI1FromI2"
	case 246:
		return "imp_VarI1FromI4"
	case 247:
		return "imp_VarI1FromR4"
	case 248:
		return "imp_VarI1FromR8"
	case 249:
		return "imp_VarI1FromDate"
	case 250:
		return "imp_VarI1FromCy"
	case 251:
		return "imp_VarI1FromStr"
	case 252:
		return "imp_VarI1FromDisp"
	case 253:
		return "imp_VarI1FromBool"
	case 254:
		return "imp_VarI1FromUI2"
	case 255:
		return "imp_VarI1FromUI4"
	case 256:
		return "imp_VarI1FromDec"
	case 257:
		return "imp_VarUI2FromUI1"
	case 258:
		return "imp_VarUI2FromI2"
	case 259:
		return "imp_VarUI2FromI4"
	case 260:
		return "imp_VarUI2FromR4"
	case 261:
		return "imp_VarUI2FromR8"
	case 262:
		return "imp_VarUI2FromDate"
	case 263:
		return "imp_VarUI2FromCy"
	case 264:
		return "imp_VarUI2FromStr"
	case 265:
		return "imp_VarUI2FromDisp"
	case 266:
		return "imp_VarUI2FromBool"
	case 267:
		return "imp_VarUI2FromI1"
	case 268:
		return "imp_VarUI2FromUI4"
	case 269:
		return "imp_VarUI2FromDec"
	case 270:
		return "imp_VarUI4FromUI1"
	case 271:
		return "imp_VarUI4FromI2"
	case 272:
		return "imp_VarUI4FromI4"
	case 273:
		return "imp_VarUI4FromR4"
	case 274:
		return "imp_VarUI4FromR8"
	case 275:
		return "imp_VarUI4FromDate"
	case 276:
		return "imp_VarUI4FromCy"
	case 277:
		return "imp_VarUI4FromStr"
	case 278:
		return "imp_VarUI4FromDisp"
	case 279:
		return "imp_VarUI4FromBool"
	case 280:
		return "imp_VarUI4FromI1"
	case 281:
		return "imp_VarUI4FromUI2"
	case 282:
		return "imp_VarUI4FromDec"
	case 283:
		return "imp_BSTR_UserSize"
	case 284:
		return "imp_BSTR_UserMarshal"
	case 285:
		return "imp_BSTR_UserUnmarshal"
	case 286:
		return "imp_BSTR_UserFree"
	case 287:
		return "imp_VARIANT_UserSize"
	case 288:
		return "imp_VARIANT_UserMarshal"
	case 289:
		return "imp_VARIANT_UserUnmarshal"
	case 290:
		return "imp_VARIANT_UserFree"
	case 291:
		return "imp_LPSAFEARRAY_UserSize"
	case 292:
		return "imp_LPSAFEARRAY_UserMarshal"
	case 293:
		return "imp_LPSAFEARRAY_UserUnmarshal"
	case 294:
		return "imp_LPSAFEARRAY_UserFree"
	case 295:
		return "imp_LPSAFEARRAY_Size"
	case 296:
		return "imp_LPSAFEARRAY_Marshal"
	case 297:
		return "imp_LPSAFEARRAY_Unmarshal"
	case 298:
		return "imp_VarDecCmpR8"
	case 299:
		return "imp_VarCyAdd"
	case 300:
		return "imp_BSTR_UserUnmarshal64"
	case 301:
		return "imp_DllCanUnloadNow"
	case 302:
		return "imp_DllGetClassObject"
	case 303:
		return "imp_VarCyMul"
	case 304:
		return "imp_VarCyMulI4"
	case 305:
		return "imp_VarCySub"
	case 306:
		return "imp_VarCyAbs"
	case 307:
		return "imp_VarCyFix"
	case 308:
		return "imp_VarCyInt"
	case 309:
		return "imp_VarCyNeg"
	case 310:
		return "imp_VarCyRound"
	case 311:
		return "imp_VarCyCmp"
	case 312:
		return "imp_VarCyCmpR8"
	case 313:
		return "imp_VarBstrCat"
	case 314:
		return "imp_VarBstrCmp"
	case 315:
		return "imp_VarR8Pow"
	case 316:
		return "imp_VarR4CmpR8"
	case 317:
		return "imp_VarR8Round"
	case 318:
		return "imp_VarCat"
	case 319:
		return "imp_VarDateFromUdateEx"
	case 320:
		return "imp_DllRegisterServer"
	case 321:
		return "imp_DllUnregisterServer"
	case 322:
		return "imp_GetRecordInfoFromGuids"
	case 323:
		return "imp_GetRecordInfoFromTypeInfo"
	case 324:
		return "imp_LPSAFEARRAY_UserFree64"
	case 325:
		return "imp_SetVarConversionLocaleSetting"
	case 326:
		return "imp_GetVarConversionLocaleSetting"
	case 327:
		return "imp_SetOaNoCache"
	case 328:
		return "imp_LPSAFEARRAY_UserMarshal64"
	case 329:
		return "imp_VarCyMulI8"
	case 330:
		return "imp_VarDateFromUdate"
	case 331:
		return "imp_VarUdateFromDate"
	case 332:
		return "imp_GetAltMonthNames"
	case 333:
		return "imp_VarI8FromUI1"
	case 334:
		return "imp_VarI8FromI2"
	case 335:
		return "imp_VarI8FromR4"
	case 336:
		return "imp_VarI8FromR8"
	case 337:
		return "imp_VarI8FromCy"
	case 338:
		return "imp_VarI8FromDate"
	case 339:
		return "imp_VarI8FromStr"
	case 340:
		return "imp_VarI8FromDisp"
	case 341:
		return "imp_VarI8FromBool"
	case 342:
		return "imp_VarI8FromI1"
	case 343:
		return "imp_VarI8FromUI2"
	case 344:
		return "imp_VarI8FromUI4"
	case 345:
		return "imp_VarI8FromDec"
	case 346:
		return "imp_VarI2FromI8"
	case 347:
		return "imp_VarI2FromUI8"
	case 348:
		return "imp_VarI4FromI8"
	case 349:
		return "imp_VarI4FromUI8"
	case 350:
		return "imp_LPSAFEARRAY_UserSize64"
	case 351:
		return "imp_LPSAFEARRAY_UserUnmarshal64"
	case 352:
		return "imp_OACreateTypeLib2"
	case 353:
		return "imp_VARIANT_UserFree64"
	case 354:
		return "imp_VARIANT_UserMarshal64"
	case 355:
		return "imp_VARIANT_UserSize64"
	case 356:
		return "imp_VARIANT_UserUnmarshal64"
	case 360:
		return "imp_VarR4FromI8"
	case 361:
		return "imp_VarR4FromUI8"
	case 362:
		return "imp_VarR8FromI8"
	case 363:
		return "imp_VarR8FromUI8"
	case 364:
		return "imp_VarDateFromI8"
	case 365:
		return "imp_VarDateFromUI8"
	case 366:
		return "imp_VarCyFromI8"
	case 367:
		return "imp_VarCyFromUI8"
	case 368:
		return "imp_VarBstrFromI8"
	case 369:
		return "imp_VarBstrFromUI8"
	case 370:
		return "imp_VarBoolFromI8"
	case 371:
		return "imp_VarBoolFromUI8"
	case 372:
		return "imp_VarUI1FromI8"
	case 373:
		return "imp_VarUI1FromUI8"
	case 374:
		return "imp_VarDecFromI8"
	case 375:
		return "imp_VarDecFromUI8"
	case 376:
		return "imp_VarI1FromI8"
	case 377:
		return "imp_VarI1FromUI8"
	case 378:
		return "imp_VarUI2FromI8"
	case 379:
		return "imp_VarUI2FromUI8"
	case 401:
		return "imp_OleLoadPictureEx"
	case 402:
		return "imp_OleLoadPictureFileEx"
	case 411:
		return "imp_SafeArrayCreateVector"
	case 412:
		return "imp_SafeArrayCopyData"
	case 413:
		return "imp_VectorFromBstr"
	case 414:
		return "imp_BstrFromVector"
	case 415:
		return "imp_OleIconToCursor"
	case 416:
		return "imp_OleCreatePropertyFrameIndirect"
	case 417:
		return "imp_OleCreatePropertyFrame"
	case 418:
		return "imp_OleLoadPicture"
	case 419:
		return "imp_OleCreatePictureIndirect"
	case 420:
		return "imp_OleCreateFontIndirect"
	case 421:
		return "imp_OleTranslateColor"
	case 422:
		return "imp_OleLoadPictureFile"
	case 423:
		return "imp_OleSavePictureFile"
	case 424:
		return "imp_OleLoadPicturePath"
	case 425:
		return "imp_VarUI4FromI8"
	case 426:
		return "imp_VarUI4FromUI8"
	case 427:
		return "imp_VarI8FromUI8"
	case 428:
		return "imp_VarUI8FromI8"
	case 429:
		return "imp_VarUI8FromUI1"
	case 430:
		return "imp_VarUI8FromI2"
	case 431:
		return "imp_VarUI8FromR4"
	case 432:
		return "imp_VarUI8FromR8"
	case 433:
		return "imp_VarUI8FromCy"
	case 434:
		return "imp_VarUI8FromDate"
	case 435:
		return "imp_VarUI8FromStr"
	case 436:
		return "imp_VarUI8FromDisp"
	case 437:
		return "imp_VarUI8FromBool"
	case 438:
		return "imp_VarUI8FromI1"
	case 439:
		return "imp_VarUI8FromUI2"
	case 440:
		return "imp_VarUI8FromUI4"
	case 441:
		return "imp_VarUI8FromDec"
	case 442:
		return "imp_RegisterTypeLibForUser"
	case 443:
		return "imp_UnRegisterTypeLibForUser"
	case 444:
		return "imp_OaEnablePerUserTLibRegistration"
	case 500:
		return "imp_OACleanup"
	}
	return ""
}

func translateOrdinalWS32(ord uint16) string {
	switch ord {
	case 1:
		return "imp_accept"
	case 2:
		return "imp_bind"
	case 3:
		return "imp_closesocket"
	case 4:
		return "imp_connect"
	case 5:
		return "imp_getpeername"
	case 6:
		return "imp_getsockname"
	case 7:
		return "imp_getsockopt"
	case 8:
		return "imp_htonl"
	case 9:
		return "imp_htons"
	case 10:
		return "imp_ioctlsocket"
	case 11:
		return "imp_inet_addr"
	case 12:
		return "imp_inet_ntoa"
	case 13:
		return "imp_listen"
	case 14:
		return "imp_ntohl"
	case 15:
		return "imp_ntohs"
	case 16:
		return "imp_recv"
	case 17:
		return "imp_recvfrom"
	case 18:
		return "imp_select"
	case 19:
		return "imp_send"
	case 20:
		return "imp_sendto"
	case 21:
		return "imp_setsockopt"
	case 22:
		return "imp_shutdown"
	case 23:
		return "imp_socket"
	case 24:
		return "imp_GetAddrInfoW"
	case 25:
		return "imp_GetNameInfoW"
	case 26:
		return "imp_WSApSetPostRoutine"
	case 27:
		return "imp_FreeAddrInfoW"
	case 28:
		return "imp_WPUCompleteOverlappedRequest"
	case 29:
		return "imp_WSAAccept"
	case 30:
		return "imp_WSAAddressToStringA"
	case 31:
		return "imp_WSAAddressToStringW"
	case 32:
		return "imp_WSACloseEvent"
	case 33:
		return "imp_WSAConnect"
	case 34:
		return "imp_WSACreateEvent"
	case 35:
		return "imp_WSADuplicateSocketA"
	case 36:
		return "imp_WSADuplicateSocketW"
	case 37:
		return "imp_WSAEnumNameSpaceProvidersA"
	case 38:
		return "imp_WSAEnumNameSpaceProvidersW"
	case 39:
		return "imp_WSAEnumNetworkEvents"
	case 40:
		return "imp_WSAEnumProtocolsA"
	case 41:
		return "imp_WSAEnumProtocolsW"
	case 42:
		return "imp_WSAEventSelect"
	case 43:
		return "imp_WSAGetOverlappedResult"
	case 44:
		return "imp_WSAGetQOSByName"
	case 45:
		return "imp_WSAGetServiceClassInfoA"
	case 46:
		return "imp_WSAGetServiceClassInfoW"
	case 47:
		return "imp_WSAGetServiceClassNameByClassIdA"
	case 48:
		return "imp_WSAGetServiceClassNameByClassIdW"
	case 49:
		return "imp_WSAHtonl"
	case 50:
		return "imp_WSAHtons"
	case 51:
		return "imp_gethostbyaddr"
	case 52:
		return "imp_gethostbyname"
	case 53:
		return "imp_getprotobyname"
	case 54:
		return "imp_getprotobynumber"
	case 55:
		return "imp_getservbyname"
	case 56:
		return "imp_getservbyport"
	case 57:
		return "imp_gethostname"
	case 58:
		return "imp_WSAInstallServiceClassA"
	case 59:
		return "imp_WSAInstallServiceClassW"
	case 60:
		return "imp_WSAIoctl"
	case 61:
		return "imp_WSAJoinLeaf"
	case 62:
		return "imp_WSALookupServiceBeginA"
	case 63:
		return "imp_WSALookupServiceBeginW"
	case 64:
		return "imp_WSALookupServiceEnd"
	case 65:
		return "imp_WSALookupServiceNextA"
	case 66:
		return "imp_WSALookupServiceNextW"
	case 67:
		return "imp_WSANSPIoctl"
	case 68:
		return "imp_WSANtohl"
	case 69:
		return "imp_WSANtohs"
	case 70:
		return "imp_WSAProviderConfigChange"
	case 71:
		return "imp_WSARecv"
	case 72:
		return "imp_WSARecvDisconnect"
	case 73:
		return "imp_WSARecvFrom"
	case 74:
		return "imp_WSARemoveServiceClass"
	case 75:
		return "imp_WSAResetEvent"
	case 76:
		return "imp_WSASend"
	case 77:
		return "imp_WSASendDisconnect"
	case 78:
		return "imp_WSASendTo"
	case 79:
		return "imp_WSASetEvent"
	case 80:
		return "imp_WSASetServiceA"
	case 81:
		return "imp_WSASetServiceW"
	case 82:
		return "imp_WSASocketA"
	case 83:
		return "imp_WSASocketW"
	case 84:
		return "imp_WSAStringToAddressA"
	case 85:
		return "imp_WSAStringToAddressW"
	case 86:
		return "imp_WSAWaitForMultipleEvents"
	case 87:
		return "imp_WSCDeinstallProvider"
	case 88:
		return "imp_WSCEnableNSProvider"
	case 89:
		return "imp_WSCEnumProtocols"
	case 90:
		return "imp_WSCGetProviderPath"
	case 91:
		return "imp_WSCInstallNameSpace"
	case 92:
		return "imp_WSCInstallProvider"
	case 93:
		return "imp_WSCUnInstallNameSpace"
	case 94:
		return "imp_WSCUpdateProvider"
	case 95:
		return "imp_WSCWriteNameSpaceOrder"
	case 96:
		return "imp_WSCWriteProviderOrder"
	case 97:
		return "imp_freeaddrinfo"
	case 98:
		return "imp_getaddrinfo"
	case 99:
		return "imp_getnameinfo"
	case 101:
		return "imp_WSAAsyncSelect"
	case 102:
		return "imp_WSAAsyncGetHostByAddr"
	case 103:
		return "imp_WSAAsyncGetHostByName"
	case 104:
		return "imp_WSAAsyncGetProtoByNumber"
	case 105:
		return "imp_WSAAsyncGetProtoByName"
	case 106:
		return "imp_WSAAsyncGetServByPort"
	case 107:
		return "imp_WSAAsyncGetServByName"
	case 108:
		return "imp_WSACancelAsyncRequest"
	case 109:
		return "imp_WSASetBlockingHook"
	case 110:
		return "imp_WSAUnhookBlockingHook"
	case 111:
		return "imp_WSAGetLastError"
	case 112:
		return "imp_WSASetLastError"
	case 113:
		return "imp_WSACancelBlockingCall"
	case 114:
		return "imp_WSAIsBlocking"
	case 115:
		return "imp_WSAStartup"
	case 116:
		return "imp_WSACleanup"
	case 151:
		return "imp___WSAFDIsSet"
	case 500:
		return "imp_WEP"
	}

	return ""
}
