#!/usr/bin/env python3

import sys
import os
import os.path
import re

#------------------------------------------------------------------------#
# Process a *.crabir file generated by the Clam option --ocrab and print
# information about how many assertions were proven by Clam.
#------------------------------------------------------------------------#
# The option --ocrab prints the CrabIR together with comments next to
# each assertion. These comments state whether the assertion was
# proved for all possible calling contexts or otherwise, it says how
# many times the assertion was not proven. Note that the
# inter-procedural analysis might re-analyze a function multiple times
# (one per different calling context). Because of this, the same
# assertion can be re-checked multiple times. This script reports that an
# assertion is proven if the assertion was proven for all calling
# contexts.
#
# Usage:
#  clam.py prog.c --crab-check=assert --ocrab=prog.crabir MORE_OPTIONS
#  read_results.py prog.crabir
#------------------------------------------------------------------------#

def processLog(f, verbosity):
    #print("Verbosity level=" + str(verbosity))
    fd = None
    try:
        fd = open(f, 'r')
    except Exception as e:
        sys.exit('ERROR: could not open {}'.format(f))
        print(e)

    funmap = {}
    curr_func = None
    curr_num_of_asserts, curr_num_of_ok_asserts, curr_num_of_fail_asserts = 0, 0, 0
    curr_ok_assert_lines, curr_fail_assert_lines = "", ""
    for line in fd:
        if "-- Inserted" in line:
            s1,_,_ = line.split('.')
            print(s1)
        elif "declare" in line:
            _,s = line.split('declare')
            # remove the first character that it's a whitespace
            s = s[1:]
            # match any character until next (
            pattern = re.compile(r"^.*?(?=\()")
            func = re.search(pattern, s).group(0)
            if curr_func is not None:
                # store results
                funmap[curr_func] = (curr_num_of_asserts, curr_num_of_ok_asserts, curr_num_of_fail_asserts, \
                                     curr_ok_assert_lines, curr_fail_assert_lines)
                # reset counters
                curr_num_of_asserts, curr_num_of_ok_asserts, curr_num_of_fail_asserts = 0, 0, 0
                curr_ok_assert_lines, curr_fail_assert_lines = "", ""
                
            curr_func = func
        elif re.search("Result:  OK", line):
            curr_num_of_ok_asserts += 1
            curr_num_of_asserts += 1
            curr_ok_assert_lines += line 
        elif re.search("Result:  FAIL", line):
            curr_num_of_fail_asserts += 1
            curr_num_of_asserts += 1
            curr_fail_assert_lines += line
            
    total_asserts, total_ok_asserts, total_fail_asserts, total_skipped_asserts = 0,0,0,0
    for key, value in funmap.items():
        (total, ok, fail, ok_lines, fail_lines) = value
        if total > 0 and ok == 0 and fail == 0:
            print("Function " + str(key) + " has " + str(total) + \
                  " assertions but it was not analyzed")
            total_skipped_asserts += total
            continue
        total_asserts += total
        total_ok_asserts += ok
        total_fail_asserts += fail
        if (verbosity == 1 and (ok < total)) or verbosity > 1:
            print("Function " + str(key) + ":")
            print("\tTotal number of assertions:" + str(total))
            print("\tTotal number of proven assertions:" + str(ok))
            print("\tTotal number of fail assertions:" + str(fail))                
        if verbosity > 2:
            print("Proven assertions:")
            print(ok_lines)
            print("Fail assertions:")            
            print(fail_lines)
        
    print("-- Checked assertions  : " + str(total_asserts))
    print("-- Unchecked assertions: " + str(total_skipped_asserts) + \
          " (These assertions are in functions that have not been analyzed)")
    print("-- Proven assertions   : " + str(total_ok_asserts))    
    print("-- Failed assertions   : " + str(total_fail_asserts))
    warning_ratio = 0
    if total_asserts > 0:
        ok_ratio = ((total_ok_asserts / total_asserts) * 100.0)
        print("   Ratio of proven assertions: " + str(ok_ratio) + "%")
    if total_fail_asserts + total_ok_asserts < total_asserts:
        print("-- Missed assertions   : " + str(total_asserts - (total_ok_asserts + total_fail_asserts)) + \
              "   <-- This should be 0")
        
def parseOpt (argv):
    import argparse as a
    p = a.ArgumentParser(description='Process a .crabir file produced by Clam option --ocrab')
    p.add_argument(metavar='FILE', dest="infile", nargs=1, help=".crabir file")
    p.add_argument('-verbose','--verbose', metavar='NUM', required=False,
                   type=int,  dest="verbose",  default=0,
                   help="Verbosity level")
    args = p.parse_args(argv)
    return args

def main (argv):
    args = parseOpt(argv[1:])
    processLog(args.infile[0], args.verbose)
    return 0
    
if __name__ == '__main__':
    res = None
    try:
        res = main(sys.argv)
    except Exception as e:
        print(e)
    except KeyboardInterrupt:
        pass
    finally:
        sys.exit(res)
