import { cn } from "@/lib/utils";
import { CircleSlash, TerminalSquare } from "lucide-react";

type IconProps = {
  className?: string;
};

function NextIcon({ className }: IconProps) {
  return (
    <svg
      height="1024pt"
      viewBox=".5 -.2 1023 1024.1"
      width="1024pt"
      xmlns="http://www.w3.org/2000/svg"
      className={cn("dark:fill-white", className)}
    >
      <path d="m478.5.6c-2.2.2-9.2.9-15.5 1.4-145.3 13.1-281.4 91.5-367.6 212-48 67-78.7 143-90.3 223.5-4.1 28.1-4.6 36.4-4.6 74.5s.5 46.4 4.6 74.5c27.8 192.1 164.5 353.5 349.9 413.3 33.2 10.7 68.2 18 108 22.4 15.5 1.7 82.5 1.7 98 0 68.7-7.6 126.9-24.6 184.3-53.9 8.8-4.5 10.5-5.7 9.3-6.7-.8-.6-38.3-50.9-83.3-111.7l-81.8-110.5-102.5-151.7c-56.4-83.4-102.8-151.6-103.2-151.6-.4-.1-.8 67.3-1 149.6-.3 144.1-.4 149.9-2.2 153.3-2.6 4.9-4.6 6.9-8.8 9.1-3.2 1.6-6 1.9-21.1 1.9h-17.3l-4.6-2.9c-3-1.9-5.2-4.4-6.7-7.3l-2.1-4.5.2-200.5.3-200.6 3.1-3.9c1.6-2.1 5-4.8 7.4-6.1 4.1-2 5.7-2.2 23-2.2 20.4 0 23.8.8 29.1 6.6 1.5 1.6 57 85.2 123.4 185.9s157.2 238.2 201.8 305.7l81 122.7 4.1-2.7c36.3-23.6 74.7-57.2 105.1-92.2 64.7-74.3 106.4-164.9 120.4-261.5 4.1-28.1 4.6-36.4 4.6-74.5s-.5-46.4-4.6-74.5c-27.8-192.1-164.5-353.5-349.9-413.3-32.7-10.6-67.5-17.9-106.5-22.3-9.6-1-75.7-2.1-84-1.3zm209.4 309.4c4.8 2.4 8.7 7 10.1 11.8.8 2.6 1 58.2.8 183.5l-.3 179.8-31.7-48.6-31.8-48.6v-130.7c0-84.5.4-132 1-134.3 1.6-5.6 5.1-10 9.9-12.6 4.1-2.1 5.6-2.3 21.3-2.3 14.8 0 17.4.2 20.7 2z" />
      <path d="m784.3 945.1c-3.5 2.2-4.6 3.7-1.5 2 2.2-1.3 5.8-4 5.2-4.1-.3 0-2 1-3.7 2.1zm-6.9 4.5c-1.8 1.4-1.8 1.5.4.4 1.2-.6 2.2-1.3 2.2-1.5 0-.8-.5-.6-2.6 1.1zm-5 3c-1.8 1.4-1.8 1.5.4.4 1.2-.6 2.2-1.3 2.2-1.5 0-.8-.5-.6-2.6 1.1zm-5 3c-1.8 1.4-1.8 1.5.4.4 1.2-.6 2.2-1.3 2.2-1.5 0-.8-.5-.6-2.6 1.1zm-7.6 4c-3.8 2-3.6 2.8.2.9 1.7-.9 3-1.8 3-2 0-.7-.1-.6-3.2 1.1z" />
    </svg>
  );
}

function RemixIcon({ className }: IconProps) {
  return (
    <svg
      width="256px"
      height="297px"
      viewBox="0 0 256 297"
      xmlns="http://www.w3.org/2000/svg"
      className={cn("dark:fill-white", className)}
    >
      <g stroke="none" strokeWidth="1" fillRule="evenodd">
        <path
          d="M141.674538,-7.10542736e-15 C218.04743,-7.10542736e-15 256,36.3493031 256,94.4136694 C256,137.843796 229.292875,166.16709 193.214546,170.888177 C223.670152,177.025429 241.473826,194.491998 244.754554,228.952544 L245.229325,235.289856 L245.643706,241.214203 L246.00181,246.756109 L246.250531,250.934795 L246.517683,255.865245 L246.656217,258.679019 L246.853693,263.148984 L247.012965,267.370833 L247.091895,269.797544 L247.198581,273.685906 L247.290626,278.131883 L247.324005,280.280236 L247.384197,286.505871 L247.403543,293.002292 L247.40462,296.886512 L168.646185,296.886512 L168.650135,295.266478 L168.650135,295.266478 L168.678181,292.120279 L168.678181,292.120279 L168.725186,289.055223 L168.861417,281.631321 L168.895871,279.142491 L168.922852,275.239899 L168.922852,275.239899 L168.919162,272.744266 L168.896218,270.127045 L168.864335,268.072886 L168.799537,265.197081 L168.706158,262.147348 L168.580806,258.904651 L168.42009,255.449957 L168.325411,253.637163 L168.164297,250.804473 L167.978998,247.828446 L167.691838,243.623566 L167.444542,240.281862 C167.373519,239.25114 167.291291,238.24473 167.19786,237.262104 L166.996058,235.328408 C164.395177,212.50087 155.340815,203.170989 139.832974,200.059114 L138.525715,199.814028 C137.64425,199.660026 136.742867,199.52459 135.821566,199.406474 L134.424675,199.242133 C134.189371,199.216855 133.952821,199.192621 133.715026,199.169411 L132.27332,199.042283 L132.27332,199.042283 L130.801736,198.938792 L130.801736,198.938792 L129.300276,198.858003 L129.300276,198.858003 L127.785563,198.799503 L126.241612,198.761396 L124.668422,198.742777 L124.668422,198.742777 L0,198.740492 L0,136.900224 L127.619345,136.900224 C129.706029,136.900224 131.728173,136.860653 133.685777,136.779928 L135.621869,136.685425 L135.621869,136.685425 L137.514935,136.563134 L137.514935,136.563134 L139.364974,136.412701 C139.669729,136.385264 139.97269,136.35664 140.273859,136.326822 L142.05936,136.133518 C143.235352,135.995014 144.382659,135.837162 145.501284,135.659493 L147.157707,135.378069 C167.866574,131.62361 178.22062,120.630459 178.22062,99.1783057 C178.22062,75.1035054 161.354066,60.5128152 127.619345,60.5128152 L0,60.5128152 L0,-7.10542736e-15 L141.674538,-7.10542736e-15 Z M83.2762921,250.785352 C93.6094556,250.785352 97.9327877,256.522818 99.4729615,262.01452 L99.6761617,262.804225 L99.6761617,262.804225 L99.8429155,263.58653 L99.9515227,264.204367 L99.9979397,264.509915 L100.075689,265.112992 L100.134243,265.703672 L100.156667,265.993728 L100.188494,266.561991 L100.198173,266.839685 L100.205751,267.380932 L100.205751,296.886512 L0,296.886512 L0,250.785352 L83.2762921,250.785352 Z"
          fillRule="nonzero"
        ></path>
      </g>
    </svg>
  );
}

function ViteIcon({ className }: IconProps) {
  return (
    <svg
      viewBox="0 0 32 32"
      xmlns="http://www.w3.org/2000/svg"
      fill="none"
      className={className}
    >
      <g id="SVGRepo_bgCarrier" strokeWidth="0"></g>
      <g
        id="SVGRepo_tracerCarrier"
        strokeLinecap="round"
        strokeLinejoin="round"
      ></g>
      <g id="SVGRepo_iconCarrier">
        <path
          d="M29.8836 6.146L16.7418 29.6457c-.2714.4851-.9684.488-1.2439.0052L2.0956 6.1482c-.3-.5262.1498-1.1635.746-1.057l13.156 2.3516a.7144.7144 0 00.2537-.0004l12.8808-2.3478c.5942-.1083 1.0463.5241.7515 1.0513z"
          fill="url(#paint0_linear)"
        ></path>
        <path
          d="M22.2644 2.0069l-9.7253 1.9056a.3571.3571 0 00-.2879.3294l-.5982 10.1038c-.014.238.2045.4227.4367.3691l2.7077-.6248c.2534-.0585.4823.1647.4302.4194l-.8044 3.9393c-.0542.265.1947.4918.4536.4132l1.6724-.5082c.2593-.0787.5084.1487.4536.414l-1.2784 6.1877c-.08.387.4348.598.6495.2662L16.5173 25 24.442 9.1848c.1327-.2648-.096-.5667-.387-.5106l-2.787.5379c-.262.0505-.4848-.1934-.4109-.4497l1.8191-6.306c.074-.2568-.1496-.5009-.4118-.4495z"
          fill="url(#paint1_linear)"
        ></path>
        <defs id="defs50">
          <linearGradient
            id="paint0_linear"
            x1="6.0002"
            y1="32.9999"
            x2="235"
            y2="344"
            gradientUnits="userSpaceOnUse"
            gradientTransform="matrix(.07142 0 0 .07142 1.3398 1.8944)"
          >
            <stop stopColor="#41D1FF" id="stop38"></stop>
            <stop offset="1" stopColor="#BD34FE" id="stop40"></stop>
          </linearGradient>
          <linearGradient
            id="paint1_linear"
            x1="194.651"
            y1="8.8182"
            x2="236.076"
            y2="292.989"
            gradientUnits="userSpaceOnUse"
            gradientTransform="matrix(.07142 0 0 .07142 1.3398 1.8944)"
          >
            <stop stopColor="#FFEA83" id="stop43"></stop>
            <stop offset=".0833" stopColor="#FFDD35" id="stop45"></stop>
            <stop offset="1" stopColor="#FFA800" id="stop47"></stop>
          </linearGradient>
        </defs>
      </g>
    </svg>
  );
}

function SvelteIcon({ className }: IconProps) {
  return (
    <svg
      height="2500"
      viewBox="-23.04085003 -23.7 545.4320132 647"
      width="2079"
      xmlns="http://www.w3.org/2000/svg"
      className={className}
    >
      <path
        d="m466.95 79.52c-55.66-79.62-165.6-103.22-245.08-52.6l-139.58 88.93c-9.39 5.9-18.15 12.76-26.12 20.47-7.98 7.71-15.13 16.23-21.34 25.42s-11.45 19-15.64 29.27a160.478 160.478 0 0 0 -9.26 31.87c-1.65 9.15-2.55 18.43-2.67 27.73-.13 9.31.52 18.61 1.93 27.8 1.41 9.2 3.58 18.27 6.48 27.11s6.53 17.42 10.85 25.66a161.68 161.68 0 0 0 -8.22 13.97c-2.51 4.79-4.77 9.71-6.78 14.73s-3.76 10.14-5.25 15.34-2.71 10.47-3.67 15.79a170.365 170.365 0 0 0 1.55 67.48c2.5 11.05 6.09 21.83 10.73 32.17s10.29 20.2 16.89 29.42c55.66 79.62 165.59 103.22 245.07 52.6l139.58-88.56c9.39-5.91 18.13-12.78 26.1-20.5a160.58 160.58 0 0 0 21.33-25.42c6.21-9.18 11.45-18.99 15.64-29.26 4.19-10.26 7.3-20.94 9.29-31.85 1.65-9.15 2.54-18.42 2.66-27.72s-.53-18.6-1.95-27.79c-1.41-9.19-3.58-18.25-6.49-27.09-2.91-8.83-6.54-17.41-10.86-25.65 2.97-4.51 5.72-9.18 8.23-13.97 2.5-4.79 4.77-9.71 6.78-14.73s3.77-10.14 5.27-15.34c1.49-5.19 2.73-10.46 3.7-15.78 1.98-11.16 2.84-22.49 2.58-33.82s-1.65-22.6-4.15-33.66c-2.5-11.05-6.09-21.83-10.73-32.17a170.906 170.906 0 0 0 -16.87-29.42"
        fill="#ff3e00"
      />
      <path
        d="m208.23 527.78a110.876 110.876 0 0 1 -33.49 3.42c-11.27-.58-22.39-2.86-32.97-6.79a111.06 111.06 0 0 1 -29.42-16.35 111.108 111.108 0 0 1 -23.15-24.42c-3.97-5.55-7.37-11.47-10.15-17.69a102.38 102.38 0 0 1 -6.45-19.34c-1.49-6.65-2.33-13.43-2.48-20.24s.38-13.62 1.58-20.33c.19-1.09.41-2.18.65-3.26.23-1.09.49-2.17.77-3.24.27-1.08.57-2.15.89-3.22.31-1.06.65-2.12 1-3.17l2.63-8.03 7.17 5.35c4.11 3 8.35 5.83 12.7 8.47 4.35 2.65 8.81 5.11 13.37 7.37 4.55 2.27 9.21 4.35 13.94 6.22 4.73 1.88 9.54 3.55 14.42 5.02l5.35 1.55-.48 5.35a31.395 31.395 0 0 0 1.12 10.81c.49 1.76 1.14 3.46 1.93 5.1s1.72 3.21 2.78 4.69a33.4 33.4 0 0 0 6.99 7.35c2.68 2.08 5.67 3.74 8.86 4.92s6.53 1.86 9.93 2.03c3.39.18 6.79-.17 10.08-1.03.76-.2 1.5-.43 2.24-.69s1.47-.54 2.18-.86c.72-.31 1.42-.65 2.12-1.02.69-.36 1.36-.75 2.02-1.17l139.37-88.94a28.96 28.96 0 0 0 4.75-3.72c1.45-1.41 2.74-2.96 3.87-4.63s2.07-3.46 2.83-5.33c.75-1.87 1.31-3.81 1.67-5.79.35-2.03.5-4.08.45-6.14-.05-2.05-.31-4.09-.77-6.1-.45-2-1.11-3.95-1.96-5.83-.84-1.87-1.88-3.65-3.08-5.32-1.94-2.79-4.29-5.26-6.98-7.34s-5.68-3.74-8.86-4.92a33.464 33.464 0 0 0 -9.93-2.04c-3.4-.17-6.8.18-10.09 1.03-.75.2-1.5.43-2.24.69s-1.46.54-2.18.85c-.72.32-1.42.66-2.11 1.03-.69.36-1.37.76-2.03 1.18l-53.52 33.98c-2.18 1.38-4.42 2.68-6.7 3.9-2.29 1.21-4.61 2.34-6.98 3.38s-4.78 1.99-7.22 2.84c-2.44.86-4.91 1.62-7.41 2.29-10.91 2.82-22.18 3.96-33.43 3.38s-22.34-2.87-32.9-6.78c-10.56-3.92-20.46-9.43-29.36-16.33s-16.7-15.11-23.13-24.36c-3.95-5.55-7.34-11.48-10.11-17.7-2.78-6.22-4.93-12.7-6.42-19.34-1.49-6.65-2.31-13.43-2.45-20.24-.15-6.8.38-13.61 1.59-20.31a96.419 96.419 0 0 1 14.94-36.86 96.283 96.283 0 0 1 28.57-27.68l139.8-88.93c2.17-1.38 4.39-2.68 6.66-3.9 2.27-1.21 4.59-2.34 6.94-3.38a98.21 98.21 0 0 1 7.18-2.84c2.42-.86 4.88-1.63 7.37-2.3 10.92-2.83 22.21-3.99 33.47-3.42 11.27.58 22.38 2.86 32.96 6.79 10.58 3.92 20.49 9.44 29.41 16.35a111.11 111.11 0 0 1 23.14 24.43c3.96 5.54 7.37 11.46 10.16 17.68s4.95 12.69 6.46 19.34c1.5 6.65 2.34 13.43 2.49 20.24.16 6.81-.36 13.62-1.56 20.33-.21 1.1-.43 2.2-.68 3.29-.24 1.09-.5 2.18-.78 3.26-.27 1.09-.57 2.17-.88 3.24-.31 1.08-.63 2.15-.98 3.21l-2.67 8.03-7.12-5.35c-4.12-3.03-8.37-5.87-12.73-8.54-4.36-2.66-8.84-5.14-13.41-7.43a182.39 182.39 0 0 0 -28.45-11.32l-5.36-1.55.49-5.35c.15-1.83.14-3.67-.03-5.49-.16-1.82-.49-3.63-.97-5.4-.49-1.76-1.12-3.49-1.91-5.14-.78-1.66-1.71-3.24-2.77-4.74a33.153 33.153 0 0 0 -6.99-7.2 32.991 32.991 0 0 0 -8.82-4.8 33.244 33.244 0 0 0 -19.83-.89c-.76.2-1.51.43-2.24.68-.74.26-1.47.55-2.19.86-.71.31-1.42.66-2.11 1.02-.69.37-1.37.76-2.03 1.18l-139.63 88.78c-1.7 1.07-3.29 2.32-4.73 3.72s-2.74 2.95-3.87 4.61a29.724 29.724 0 0 0 -2.83 5.31c-.76 1.87-1.32 3.8-1.68 5.78-.35 2.03-.5 4.09-.45 6.15a31.547 31.547 0 0 0 2.73 11.95 31.84 31.84 0 0 0 3.07 5.34c1.93 2.76 4.27 5.22 6.94 7.28a33.26 33.26 0 0 0 8.79 4.9 33.533 33.533 0 0 0 19.86 1.09c.75-.21 1.5-.44 2.24-.7.73-.26 1.46-.55 2.18-.86a29.2 29.2 0 0 0 2.11-1.02c.69-.36 1.37-.75 2.03-1.17l53.52-33.92c2.19-1.4 4.42-2.72 6.71-3.94 2.28-1.23 4.61-2.36 6.99-3.41a99.39 99.39 0 0 1 7.23-2.84c2.45-.86 4.93-1.62 7.44-2.28 10.92-2.84 22.2-4 33.47-3.44 11.27.57 22.38 2.85 32.96 6.77 10.57 3.92 20.49 9.43 29.4 16.35 8.92 6.91 16.72 15.14 23.15 24.41 3.96 5.55 7.36 11.47 10.15 17.69a102.65 102.65 0 0 1 6.46 19.34c1.5 6.64 2.34 13.42 2.5 20.23.16 6.82-.37 13.63-1.56 20.33a96.419 96.419 0 0 1 -5.55 19.21 95.753 95.753 0 0 1 -9.4 17.65c-3.73 5.54-8.03 10.68-12.83 15.33s-10.07 8.79-15.73 12.35l-139.64 88.93c-2.19 1.39-4.43 2.7-6.71 3.92-2.29 1.22-4.62 2.35-7 3.39-2.37 1.05-4.78 2-7.23 2.86-2.44.86-4.92 1.63-7.42 2.3"
        fill="#fff"
      />
    </svg>
  );
}

function SolidIcon({ className }: IconProps) {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      viewBox="0 0 166 155.3"
      className={className}
    >
      <path
        d="M163 35S110-4 69 5l-3 1c-6 2-11 5-14 9l-2 3-15 26 26 5c11 7 25 10 38 7l46 9 18-30z"
        fill="#76b3e1"
      />
      <linearGradient
        id="a"
        gradientUnits="userSpaceOnUse"
        x1="27.5"
        y1="3"
        x2="152"
        y2="63.5"
      >
        <stop offset=".1" stopColor="#76b3e1" />
        <stop offset=".3" stopColor="#dcf2fd" />
        <stop offset="1" stopColor="#76b3e1" />
      </linearGradient>
      <path
        d="M163 35S110-4 69 5l-3 1c-6 2-11 5-14 9l-2 3-15 26 26 5c11 7 25 10 38 7l46 9 18-30z"
        opacity=".3"
        fill="url(#a)"
      />
      <path
        d="M52 35l-4 1c-17 5-22 21-13 35 10 13 31 20 48 15l62-21S92 26 52 35z"
        fill="#518ac8"
      />
      <linearGradient
        id="b"
        gradientUnits="userSpaceOnUse"
        x1="95.8"
        y1="32.6"
        x2="74"
        y2="105.2"
      >
        <stop offset="0" stopColor="#76b3e1" />
        <stop offset=".5" stopColor="#4377bb" />
        <stop offset="1" stopColor="#1f3b77" />
      </linearGradient>
      <path
        d="M52 35l-4 1c-17 5-22 21-13 35 10 13 31 20 48 15l62-21S92 26 52 35z"
        opacity=".3"
        fill="url(#b)"
      />
      <linearGradient
        id="c"
        gradientUnits="userSpaceOnUse"
        x1="18.4"
        y1="64.2"
        x2="144.3"
        y2="149.8"
      >
        <stop offset="0" stopColor="#315aa9" />
        <stop offset=".5" stopColor="#518ac8" />
        <stop offset="1" stopColor="#315aa9" />
      </linearGradient>
      <path
        d="M134 80a45 45 0 00-48-15L24 85 4 120l112 19 20-36c4-7 3-15-2-23z"
        fill="url(#c)"
      />
      <linearGradient
        id="d"
        gradientUnits="userSpaceOnUse"
        x1="75.2"
        y1="74.5"
        x2="24.4"
        y2="260.8"
      >
        <stop offset="0" stopColor="#4377bb" />
        <stop offset=".5" stopColor="#1a336b" />
        <stop offset="1" stopColor="#1a336b" />
      </linearGradient>
      <path
        d="M114 115a45 45 0 00-48-15L4 120s53 40 94 30l3-1c17-5 23-21 13-34z"
        fill="url(#d)"
      />
    </svg>
  );
}

function QwikIcon({ className }: IconProps) {
  return (
    <svg
      height="2500"
      preserveAspectRatio="xMidYMid"
      viewBox="-3.523 -.15 262.378 275.068"
      width="2336"
      xmlns="http://www.w3.org/2000/svg"
      xmlnsXlink="http://www.w3.org/1999/xlink"
      className={className}
    >
      <linearGradient id="a">
        <stop offset="0" stopColor="#4340c4" />
        <stop offset=".12" stopColor="#4642c8" />
        <stop offset="1" stopColor="#594ee4" />
      </linearGradient>
      <linearGradient
        id="b"
        x1="22.347%"
        x2="77.517%"
        xlinkHref="#a"
        y1="49.545%"
        y2="50.388%"
      />
      <linearGradient
        id="c"
        x1="38.874%"
        x2="60.879%"
        y1="49.845%"
        y2="50.385%"
      >
        <stop offset="0" stopColor="#4340c4" />
        <stop offset=".74" stopColor="#534adb" />
        <stop offset="1" stopColor="#594ee4" />
      </linearGradient>
      <linearGradient
        id="d"
        x1="-.004%"
        x2="100.123%"
        y1="49.529%"
        y2="50.223%"
      >
        <stop offset="0" stopColor="#4340c4" />
        <stop offset=".23" stopColor="#4340c4" />
        <stop offset=".6" stopColor="#4f48d5" />
        <stop offset="1" stopColor="#594ee4" />
      </linearGradient>
      <linearGradient id="e" x1="35.4%" x2="64.895%" y1="49.459%" y2="50.085%">
        <stop offset="0" stopColor="#0080ff" />
        <stop offset="1" stopColor="#00b9ff" />
      </linearGradient>
      <linearGradient
        id="f"
        x1="-.243%"
        x2="100.411%"
        y1="49.366%"
        y2="50.467%"
      >
        <stop offset="0" stopColor="#0080ff" />
        <stop offset=".17" stopColor="#008bff" />
        <stop offset=".47" stopColor="#00a7ff" />
        <stop offset=".63" stopColor="#00b9ff" />
        <stop offset="1" stopColor="#00b9ff" />
      </linearGradient>
      <linearGradient
        id="g"
        x1="-.125%"
        x2="100.225%"
        y1="49.627%"
        y2="50.101%"
      >
        <stop offset="0" stopColor="#00b9ff" />
        <stop offset=".3" stopColor="#0080ff" />
        <stop offset=".6" stopColor="#2d67f1" />
        <stop offset=".86" stopColor="#4d55e8" />
        <stop offset="1" stopColor="#594ee4" />
      </linearGradient>
      <linearGradient
        id="h"
        x1="4.557%"
        x2="99.354%"
        xlinkHref="#a"
        y1="50.184%"
        y2="51.298%"
      />
      <path
        d="m175.05 236.86 25.163-15.072 49.298-86.93-76.287 89.098z"
        fill="url(#b)"
      />
      <path
        d="m242.337 80.408-4.926-9.4-1.932-3.663-.2.196-25.818-47.015c-6.477-11.876-18.83-19.295-32.451-19.452l-25.074.206 36.214 113.52-23.958 23.331 8.924 86.245 73.769-84.021c10.005-11.587 11.97-28.09 4.92-41.646l-9.466-18.302z"
        fill="url(#c)"
      />
      <path
        d="m201.113 72.256-43.18-70.907-74.523-1.346c-13.245-.153-25.58 6.57-32.53 17.867l-43.87 83.877 34.443-33.334 43.248-60.057 97.894 112.153 18.3-18.626c8.397-8.142 5.54-19.558.22-29.625z"
        fill="url(#d)"
      />
      <path
        d="m97.784 95.26-13.262-86.464-73.148 88.03c-12.328 11.935-14.897 30.662-6.419 45.49l42.98 74.727c6.553 11.464 18.755 18.577 32.024 18.729l42.945.49-51.444-116.655z"
        fill="url(#e)"
      />
      <path
        d="m173.227 223.9-101.847-104.878-13.196 12.59c-10.812 10.248-11.106 27.332-.728 37.921l43.99 66.384 70.65.907 1.127-12.926z"
        fill="url(#f)"
      />
      <path d="m101.584 235.903 72.292-11.6 47.704 49.465z" fill="url(#g)" />
      <path
        d="m173.111 224.483 27.168-3.457 24.096 49.915c1.06 2.06-1.719 3.977-3.373 2.302z"
        fill="url(#h)"
      />
      <path
        d="m182.708 120.058-98.028-111.458 12.502 85.958-26.022 24.222 101.772 105.372-7.595-85.905 17.37-18.192z"
        fill="#fff"
      />
    </svg>
  );
}

const icons = {
  next: NextIcon,
  remix: RemixIcon,
  vite: ViteIcon,
  svelte: SvelteIcon,
  solid: SolidIcon,
  qwik: QwikIcon,
  cli: TerminalSquare,
  unknown: CircleSlash,
};

type BrandIconKey = keyof typeof icons;
type BrandIcon = (typeof icons)[BrandIconKey];

function isBrandIcon(icon: string): icon is BrandIconKey {
  return icon in icons;
}

type Props = {
  icon: BrandIconKey | (string & {});
  className?: string;
};

export function BrandIcon({ icon, className }: Props) {
  let Icon: BrandIcon = icons.unknown;
  if (isBrandIcon(icon)) {
    Icon = icons[icon];
  }
  return <Icon className={className} />;
}
