{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_ObjectFromPoolAbstract;

interface

type
  TObjectFromPoolAbstract = class(TObject, IInterface)
  private
    FRefCount: Integer;
    FNextFree: TObjectFromPoolAbstract;
  protected
    function CheckNeedDestroyObject: Boolean; virtual; abstract;
    procedure InternalCleanup; virtual;
  protected
    function QueryInterface(
      const IID: TGUID;
      out Obj
    ): HResult; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  public
    property NextFree: TObjectFromPoolAbstract read FNextFree write FNextFree;
  end;

implementation

uses
  Windows;

{ TObjectFromPoolAbstract }

procedure TObjectFromPoolAbstract.InternalCleanup;
begin
  // Do nothing by default
end;

function TObjectFromPoolAbstract.QueryInterface(
  const IID: TGUID;
  out Obj
): HResult;
begin
  if GetInterface(IID, Obj) then begin
    Result := 0;
  end else begin
    Result := E_NOINTERFACE;
  end;
end;

function TObjectFromPoolAbstract._AddRef: Integer;
begin
  Assert(FNextFree = nil);
  Result := InterlockedIncrement(FRefCount);
end;

function TObjectFromPoolAbstract._Release: Integer;
begin
  Result := InterlockedDecrement(FRefCount);
  if Result = 0 then begin
    InternalCleanup;
    if CheckNeedDestroyObject then begin
      Destroy;
    end;
  end;
end;

end.
