{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_MapTypeChangeableByConfig;

interface

uses
  SysUtils,
  i_Notifier,
  i_Listener,
  i_MapType,
  i_MapTypeSet,
  i_ActiveMapsConfig,
  u_ChangeableBase;

type
  TMapTypeChangeableByConfig = class(TChangeableWithSimpleLockBase, IMapTypeChangeable)
  private
    FConfig: IActiveMapConfig;
    FMapsSet: IMapTypeSet;

    FListener: IListener;

    FStatic: IMapType;
    procedure OnConfigChange;
  private
    function GetStatic: IMapType;
  public
    constructor Create(
      const AConfig: IActiveMapConfig;
      const AMapsSet: IMapTypeSet
    );
    destructor Destroy; override;
  end;

implementation

uses
  c_ZeroGUID,
  u_ListenerByEvent;

{ TMapTypeChangeableByConfig }

constructor TMapTypeChangeableByConfig.Create(
  const AConfig: IActiveMapConfig;
  const AMapsSet: IMapTypeSet
);
begin
  Assert(Assigned(AConfig));
  Assert(Assigned(AMapsSet));
  inherited Create;
  FConfig := AConfig;
  FMapsSet := AMapsSet;

  FListener := TNotifyNoMmgEventListener.Create(Self.OnConfigChange);
  FConfig.ChangeNotifier.Add(FListener);
  OnConfigChange;
end;

destructor TMapTypeChangeableByConfig.Destroy;
begin
  if Assigned(FConfig) and Assigned(FListener) then begin
    FConfig.ChangeNotifier.Remove(FListener);
    FListener := nil;
    FConfig := nil;
  end;
  inherited;
end;

function TMapTypeChangeableByConfig.GetStatic: IMapType;
begin
  CS.BeginRead;
  try
    Result := FStatic;
  finally
    CS.EndRead;
  end;
end;

procedure TMapTypeChangeableByConfig.OnConfigChange;
var
  VGUID: TGUID;
  VMapType: IMapType;
  VChanged: Boolean;
begin
  VGUID := FConfig.MainMapGUID;
  Assert(not IsEqualGUID(VGUID, CGUID_Zero));
  VChanged := False;
  CS.BeginWrite;
  try
    if not Assigned(FStatic) or not IsEqualGUID(VGUID, FStatic.GUID) then begin
      VMapType := FMapsSet.GetMapTypeByGUID(VGUID);
      Assert(Assigned(VMapType));
      FStatic := VMapType;
      VChanged := True;
    end;
  finally
    CS.EndWrite;
  end;
  if VChanged then begin
    DoChangeNotify;
  end;
end;

end.
