{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_SunCalcDataProviderChangeable;

interface

uses
  SysUtils,
  i_Listener,
  i_SunCalcConfig,
  i_SunCalcDataProvider,
  u_ConfigDataElementBase;

type
  TSunCalcDataProviderChangeable = class(
    TConfigDataElementWithStaticBaseEmptySaveLoad,
    ISunCalcDataProviderChangeable
  )
  private
    FConfig: ISunCalcConfig;
    FListener: IListener;
    FProviderType: TSunCalcDataProviderType;
    procedure OnConfigChange;
  protected
    function CreateStatic: IInterface; override;
  private
    { ISunCalcDataProviderChangeable }
    function GetStatic: ISunCalcDataProvider;
  public
    constructor Create(const AConfig: ISunCalcConfig);
    destructor Destroy; override;
  end;

implementation

uses
  u_ListenerByEvent,
  u_SunCalcDataProviderSun,
  u_SunCalcDataProviderMoon;

{ TSunCalcDataProviderChangeable }

constructor TSunCalcDataProviderChangeable.Create(const AConfig: ISunCalcConfig);
begin
  Assert(AConfig <> nil);

  inherited Create;

  FConfig := AConfig;
  FProviderType := FConfig.DataProviderType;

  FListener := TNotifyNoMmgEventListener.Create(Self.OnConfigChange);
  FConfig.ChangeNotifier.Add(FListener);
end;

destructor TSunCalcDataProviderChangeable.Destroy;
begin
  if (FConfig <> nil) and (FListener <> nil) then begin
    FConfig.ChangeNotifier.Remove(FListener);
    FConfig := nil;
  end;
  inherited Destroy;
end;

function TSunCalcDataProviderChangeable.CreateStatic: IInterface;
begin
  case FProviderType of
    scdpSun  : Result := TSunCalcDataProviderSun.Create;
    scdpMoon : Result := TSunCalcDataProviderMoon.Create;
  else
    Assert(False);
    Result := nil;
  end;
end;

function TSunCalcDataProviderChangeable.GetStatic: ISunCalcDataProvider;
begin
  Result := GetStaticInternal as ISunCalcDataProvider;
end;

procedure TSunCalcDataProviderChangeable.OnConfigChange;
var
  VType: TSunCalcDataProviderType;
begin
  LockWrite;
  try
    VType := FConfig.DataProviderType;
    if VType <> FProviderType then begin
      FProviderType := VType;
      SetChanged;
    end;
  finally
    UnlockWrite;
  end;
end;

end.
