{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_MergePolygonsPresenterOnPanel;

interface

uses
  Classes,
  Controls,
  i_LanguageManager,
  i_NotifierOperation,
  i_MapViewGoto,
  i_RegionProcess,
  i_VectorItemSubset,
  i_VectorDataFactory,
  i_GeometryLonLatFactory,
  i_MergePolygonsPresenter,
  i_MergePolygonsResult,
  u_MarkDbGUIHelper,
  u_BaseInterfacedObject,
  fr_MergePolygons;

type
  TMergePolygonsPresenterOnPanel = class(TBaseInterfacedObject, IMergePolygonsPresenter)
  private
    FDrawParent: TWinControl;
    FOnAddItems: TNotifyEvent;
    FLanguageManager: ILanguageManager;
    FAppClosingNotifier: INotifierOneOperation;
    FVectorDataFactory: IVectorDataFactory;
    FVectorGeometryLonLatFactory: IGeometryLonLatFactory;
    FMergePolygonsResult: IMergePolygonsResult;
    FMapGoto: IMapViewGoto;
    FRegionProcess: IRegionProcess;
    FMarkDBGUI: TMarkDbGUIHelper;
    FfrMergePolygons: TfrMergePolygons;
  private
    { IMergePolygonsPresenter }
    procedure AddVectorItems(const AItems: IVectorItemSubset);
    procedure ClearAll;
  public
    constructor Create(
      ADrawParent: TWinControl;
      AOnAddItems: TNotifyEvent;
      const ALanguageManager: ILanguageManager;
      const AAppClosingNotifier: INotifierOneOperation;
      const AVectorDataFactory: IVectorDataFactory;
      const AVectorGeometryLonLatFactory: IGeometryLonLatFactory;
      const AMergePolygonsResult: IMergePolygonsResult;
      const AMapGoto: IMapViewGoto;
      const ARegionProcess: IRegionProcess;
      const AMarkDBGUI: TMarkDbGUIHelper
    );
    destructor Destroy; override;
  end;

implementation

uses
  SysUtils;

{ TMergePolygonsPresenterOnPanel }

constructor TMergePolygonsPresenterOnPanel.Create(
  ADrawParent: TWinControl;
  AOnAddItems: TNotifyEvent;
  const ALanguageManager: ILanguageManager;
  const AAppClosingNotifier: INotifierOneOperation;
  const AVectorDataFactory: IVectorDataFactory;
  const AVectorGeometryLonLatFactory: IGeometryLonLatFactory;
  const AMergePolygonsResult: IMergePolygonsResult;
  const AMapGoto: IMapViewGoto;
  const ARegionProcess: IRegionProcess;
  const AMarkDBGUI: TMarkDbGUIHelper
);
begin
  inherited Create;

  FDrawParent := ADrawParent;
  FOnAddItems := AOnAddItems;
  FLanguageManager := ALanguageManager;
  FAppClosingNotifier := AAppClosingNotifier;
  FVectorDataFactory := AVectorDataFactory;
  FVectorGeometryLonLatFactory := AVectorGeometryLonLatFactory;
  FMergePolygonsResult := AMergePolygonsResult;
  FMapGoto := AMapGoto;
  FRegionProcess := ARegionProcess;
  FMarkDBGUI := AMarkDBGUI;
  FfrMergePolygons := nil;
end;

destructor TMergePolygonsPresenterOnPanel.Destroy;
begin
  FMarkDBGUI := nil;
  FreeAndNil(FfrMergePolygons);
  inherited Destroy;
end;

procedure TMergePolygonsPresenterOnPanel.AddVectorItems(
  const AItems: IVectorItemSubset
);
begin
  Assert(Assigned(AItems));

  if not Assigned(FfrMergePolygons) then begin
    FfrMergePolygons :=
      TfrMergePolygons.Create(
        FDrawParent,
        FLanguageManager,
        FAppClosingNotifier,
        FVectorDataFactory,
        FVectorGeometryLonLatFactory,
        FMergePolygonsResult,
        FMapGoTo,
        FRegionProcess,
        FMarkDBGUI
      );
  end;

  FfrMergePolygons.Visible := True;
  FfrMergePolygons.AddItems(AItems);

  if Assigned(FOnAddItems) then begin
    FOnAddItems(Self);
  end;
end;

procedure TMergePolygonsPresenterOnPanel.ClearAll;
begin
  if Assigned(FfrMergePolygons) then begin
    FfrMergePolygons.Clear;
    FfrMergePolygons.Visible := False;
  end;
end;

end.
