{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_GeoCoderBy2GIS;

interface

uses
  Classes,
  i_InterfaceListSimple,
  i_NotifierOperation,
  i_LocalCoordConverter,
  i_DownloadRequest,
  i_DownloadResult,
  u_GeoCoderBasic;

type
  TGeoCoderBy2GIS = class(TGeoCoderBasic)
  protected
    function PrepareRequest(
      const ASearch: string;
      const ALocalConverter: ILocalCoordConverter
    ): IDownloadRequest; override;
    function ParseResultToPlacemarksList(
      const ACancelNotifier: INotifierOperation;
      AOperationID: Integer;
      const AResult: IDownloadResultOk;
      const ASearch: string;
      const ALocalConverter: ILocalCoordConverter
    ): IInterfaceListSimple; override;
  public
  end;

implementation

uses
  XMLIntf,
  XMLDoc,
  SysUtils,
  t_GeoTypes,
  i_Projection,
  i_GeoCoder,
  i_VectorDataItemSimple,
  u_InterfaceListSimple,
  u_AnsiStr,
  u_ResStrings,
  u_GeoToStrFunc;

{ TGeoCoderBy2GIS }

function TGeoCoderBy2GIS.ParseResultToPlacemarksList(
  const ACancelNotifier: INotifierOperation;
  AOperationID: Integer;
  const AResult: IDownloadResultOk;
  const ASearch: string;
  const ALocalConverter: ILocalCoordConverter
): IInterfaceListSimple;
var
  Stream: TMemoryStream;
  Node: IXMLNode;
  PlacemarkNode, AddressNode: IXMLNode;
  i: Integer;
  VPoint: TDoublePoint;
  VDesc: string;
  VPlace: IVectorDataItem;
  VList: IInterfaceListSimple;
  VFormatSettings: TFormatSettings;
  XMLDocument: IXMLDocument;
begin
  if AResult.Data.Size <= 0 then begin
    raise EParserError.Create(SAS_ERR_EmptyServerResponse);
  end;
  VFormatSettings.DecimalSeparator := '.';
  VList := TInterfaceListSimple.Create;
  XMLDocument := TXMLDocument.Create(nil);
  Stream := TMemoryStream.Create;
  try
    Stream.Write(AResult.Data.Buffer^, AResult.Data.Size);
    XMLDocument.LoadFromStream(Stream);
    Node := XMLDocument.DocumentElement;
    Node := Node.ChildNodes.FindNode('result');
    if (Node <> nil) and (Node.ChildNodes.Count > 0) then begin
      for i := 0 to Node.ChildNodes.Count - 1 do begin
        if Node.ChildNodes[i].NodeName = 'filial' then begin
          try
            PlacemarkNode := Node.ChildNodes[i];
            AddressNode := PlacemarkNode.ChildNodes.FindNode('name');
            VPoint.X := StrToFloat(PlacemarkNode.ChildNodes.FindNode('lon').Text, VFormatSettings);
            VPoint.Y := StrToFloat(PlacemarkNode.ChildNodes.FindNode('lat').Text, VFormatSettings);
            VDesc := PlacemarkNode.ChildNodes.FindNode('city_name').Text + ', ' +
              PlacemarkNode.ChildNodes.FindNode('address').text;
            if (AddressNode <> nil) then begin
              VPlace := PlacemarkFactory.Build(VPoint, AddressNode.Text, VDesc, '', 4);
              VList.Add(VPlace);
            end;
          except

          end;
        end;
      end;
    end;
    Result := VList;
  finally
    Stream.Free;
  end;
end;

function TGeoCoderBy2GIS.PrepareRequest(
  const ASearch: string;
  const ALocalConverter: ILocalCoordConverter
): IDownloadRequest;
var
  VSearch: String;
  VProjection: IProjection;
  VMapRect: TDoubleRect;
  VLonLatRect: TDoubleRect;
  VRadius: integer;
begin
  VSearch := ASearch;
  VProjection := ALocalConverter.Projection;
  VMapRect := ALocalConverter.GetRectInMapPixelFloat;
  VProjection.ValidatePixelRectFloat(VMapRect);
  VLonLatRect := VProjection.PixelRectFloat2LonLatRect(VMapRect);

  VRadius := Round(VProjection.ProjectionType.Datum.CalcDist(VLonLatRect.TopLeft, VLonLatRect.BottomRight));
  if VRadius > 40000 then begin
    VRadius := 40000;
  end;
  //point='+R2StrPoint(FCurrentPos.x)+','+R2StrPoint(FCurrentPos.y)+'&radius=40000&where='
  Result :=
    PrepareRequestByURL(
      'http://catalog.api.2gis.ru/search?what=' + URLEncode(AnsiToUtf8(VSearch)) +
      '&point=' + R2AnsiStrPoint(ALocalConverter.GetCenterLonLat.x) + ',' + R2AnsiStrPoint(ALocalConverter.GetCenterLonLat.y) +
      '&radius=' + IntToStrA(VRadius) +
      '&page=1&pagesize=50&key=ruihvk0699&version=1.3&sort=relevance&output=xml'
    );
end;

end.
