{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_Bitmap32SurfaceByGr32;

interface

uses
  Types,
  GR32,
  t_Bitmap32,
  i_Bitmap32Static,
  i_Bitmap32Surface,
  u_BaseInterfacedObject;

type
  TBitmap32SurfaceByGr32 = class(TBaseInterfacedObject, IBitmap32Surface)
  private
    FBitmap: TCustomBitmap32;
  private
    function GetSize: TPoint;
    function GetData: PColor32Array;
    function GetIsInited: Boolean;

    procedure Clear;
    procedure FullFill(const AFillColor: TColor32);

    procedure FillRect(
      const ARect: TRect;
      const AValue: TColor32
    );
    procedure FrameRect(
      const ARect: TRect;
      const AValue: TColor32
    );
    procedure Line(
      const APoint1, APoint2: TPoint;
      const AValue: TColor32
    );
    procedure SetPixel(
      const APoint: TPoint;
      const AValue: TColor32
    );

    procedure DrawBitmapStatic(const ASource: IBitmap32Static);
    procedure DrawBitmapStaticAt(
      const APosition: TPoint;
      const ASource: IBitmap32Static
    );
    procedure DrawBitmapData(
      const ASize: TPoint;
      const AData: PColor32Array
    );
    procedure DrawBitmapDataAt(
      const APosition: TPoint;
      const ASize: TPoint;
      const AData: PColor32Array
    );
  public
    constructor Create(ABitmap: TCustomBitmap32);
  end;

implementation

uses
  u_BitmapFunc;

{ TBitmap32SurfaceByGr32 }

constructor TBitmap32SurfaceByGr32.Create(ABitmap: TCustomBitmap32);
begin
  Assert(Assigned(ABitmap));
  inherited Create;
  FBitmap := ABitmap;
end;

procedure TBitmap32SurfaceByGr32.Clear;
begin
  FBitmap.Clear(0);
end;

procedure TBitmap32SurfaceByGr32.DrawBitmapData(
  const ASize: TPoint;
  const AData: PColor32Array
);
begin
  BlockTransferFull(
    FBitmap,
    0, 0,
    ASize,
    AData,
    dmBlend
  );
end;

procedure TBitmap32SurfaceByGr32.DrawBitmapDataAt(
  const APosition, ASize: TPoint;
  const AData: PColor32Array
);
begin
  BlockTransferFull(
    FBitmap,
    APosition.X,
    APosition.Y,
    ASize,
    AData,
    dmBlend
  );
end;

procedure TBitmap32SurfaceByGr32.DrawBitmapStatic(
  const ASource: IBitmap32Static
);
begin
  BlockTransferFull(
    FBitmap,
    0, 0,
    ASource,
    dmBlend
  );
end;

procedure TBitmap32SurfaceByGr32.DrawBitmapStaticAt(
  const APosition: TPoint;
  const ASource: IBitmap32Static
);
begin
  BlockTransferFull(
    FBitmap,
    APosition.X,
    APosition.Y,
    ASource,
    dmBlend
  );
end;

procedure TBitmap32SurfaceByGr32.FillRect(
  const ARect: TRect;
  const AValue: TColor32
);
begin
  FBitmap.FillRectTS(ARect, AValue);
end;

procedure TBitmap32SurfaceByGr32.FrameRect(
  const ARect: TRect;
  const AValue: TColor32
);
begin
  FBitmap.FrameRectTS(ARect, AValue);
end;

procedure TBitmap32SurfaceByGr32.FullFill(const AFillColor: TColor32);
begin
  FBitmap.Clear(AFillColor);
end;

function TBitmap32SurfaceByGr32.GetData: PColor32Array;
begin
  Result := FBitmap.Bits;
end;

function TBitmap32SurfaceByGr32.GetIsInited: Boolean;
begin
  Result := True;
end;

function TBitmap32SurfaceByGr32.GetSize: TPoint;
begin
  Result := Types.Point(FBitmap.Width, FBitmap.Height);
end;

procedure TBitmap32SurfaceByGr32.Line(
  const APoint1, APoint2: TPoint;
  const AValue: TColor32
);
begin
  FBitmap.LineTS(APoint1.X, APoint1.Y, APoint2.X, APoint2.Y, AValue);
end;

procedure TBitmap32SurfaceByGr32.SetPixel(
  const APoint: TPoint;
  const AValue: TColor32
);
begin
  FBitmap.SetPixelTS(APoint.X, APoint.Y, AValue);
end;

end.
