{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_Bitmap32BufferFactory;

interface

uses
  SysUtils,
  Types,
  t_Bitmap32,
  i_NotifierTime,
  i_Bitmap32Static,
  i_Bitmap32BufferFactory,
  u_ObjectPoolBitmap32Standart,
  u_BaseInterfacedObject;

type
  TBitmap32BufferFactory = class(TBaseInterfacedObject, IBitmap32BufferFactory)
  private
    FStandartSizePool: IObjectPoolBitmap32Standart;
    FFactorySimple: IBitmap32BufferFactory;
  private
    function Build(
      const ASize: TPoint;
      const AData: PColor32Array
    ): IBitmap32Buffer;
    function BuildEmpty(const ASize: TPoint): IBitmap32Buffer;
    function BuildEmptyClear(
      const ASize: TPoint;
      const AColor: TColor32
    ): IBitmap32Buffer;
  public
    constructor Create(
      const ATTLNotifier: INotifierTime;
      const ASync: IReadWriteSync
    );
  end;

implementation

uses
  GR32_LowLevel,
  u_Bitmap32BufferFactorySimple;

{ TBitmap32StaticFactory }

constructor TBitmap32BufferFactory.Create(
  const ATTLNotifier: INotifierTime;
  const ASync: IReadWriteSync
);
begin
  inherited Create;
  FStandartSizePool :=
    TObjectPoolBitmap32Standart.Create(
      ATTLNotifier,
      ASync,
      100,
      200
    );
  FFactorySimple := TBitmap32BufferFactorySimple.Create;
end;

function TBitmap32BufferFactory.Build(
  const ASize: TPoint;
  const AData: PColor32Array
): IBitmap32Buffer;
begin
  Assert(ASize.X > 0);
  Assert(ASize.Y > 0);
  Assert(ASize.X < 1 shl 16);
  Assert(ASize.Y < 1 shl 16);
  Assert(ASize.X * ASize.Y < 1 shl 28);
  Assert(AData <> nil);

  Result := BuildEmpty(ASize);
  if (Result <> nil) and (AData <> nil) then begin
    if AData <> nil then begin
      MoveLongWord(AData^, Result.Data^, ASize.X * ASize.Y);
    end;
  end;
end;

function TBitmap32BufferFactory.BuildEmpty(
  const ASize: TPoint
): IBitmap32Buffer;
var
  VStandartSize: TPoint;
begin
  VStandartSize := FStandartSizePool.Size;
  if (ASize.X = VStandartSize.X) and (ASize.Y = VStandartSize.Y) then begin
    Result := FStandartSizePool.Build;
  end else begin
    Result := FFactorySimple.BuildEmpty(ASize);
  end;
end;

function TBitmap32BufferFactory.BuildEmptyClear(
  const ASize: TPoint;
  const AColor: TColor32
): IBitmap32Buffer;
begin
  Result := BuildEmpty(ASize);
  if Result <> nil then begin
    FillLongword(Result.Data^, ASize.X * ASize.Y, AColor);
  end;
end;

end.
