{******************************************************************************}
{* This file is part of SAS.Planet project.                                   *}
{*                                                                            *}
{* Copyright (C) 2007-2022, SAS.Planet development team.                      *}
{*                                                                            *}
{* SAS.Planet is free software: you can redistribute it and/or modify         *}
{* it under the terms of the GNU General Public License as published by       *}
{* the Free Software Foundation, either version 3 of the License, or          *}
{* (at your option) any later version.                                        *}
{*                                                                            *}
{* SAS.Planet is distributed in the hope that it will be useful,              *}
{* but WITHOUT ANY WARRANTY; without even the implied warranty of             *}
{* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *}
{* GNU General Public License for more details.                               *}
{*                                                                            *}
{* You should have received a copy of the GNU General Public License          *}
{* along with SAS.Planet. If not, see <http://www.gnu.org/licenses/>.         *}
{*                                                                            *}
{* https://github.com/sasgis/sas.planet.src                                   *}
{******************************************************************************}

unit u_StringListStatic;

interface

uses
  Classes,
  i_StringListStatic,
  u_BaseInterfacedObject;

type
  TStringListStatic = class(TBaseInterfacedObject, IStringListStatic)
  private
    FList: TStringList;
  private
    function GetCount: Integer;
    function GetItem(AIndex: Integer): string;
    function IndexOf(const S: string): Integer;
  public
    constructor CreateByStrings(const AList: TStrings);
    constructor CreateWithOwn(var AList: TStringList);
    constructor CreateFromFile(const AFileName: string);
    constructor CreateByPlainText(const APlainText: string);
    destructor Destroy; override;
  end;

implementation

uses
  SysUtils;

{ TStringListStatic }

constructor TStringListStatic.CreateByStrings(const AList: TStrings);
var
  VList: TStringList;
begin
  VList := TStringList.Create;
  try
    VList.Assign(AList);
    CreateWithOwn(VList);
    VList := nil;
  finally
    VList.Free;
  end;
end;

constructor TStringListStatic.CreateWithOwn(var AList: TStringList);
begin
  inherited Create;
  FList := AList;
  AList := nil;
end;

constructor TStringListStatic.CreateFromFile(const AFileName: string);
var
  VStream: TFileStream;
begin
  inherited Create;
  FList := TStringList.Create;
  try
    VStream := TFileStream.Create(AFileName, fmOpenRead or fmShareDenyNone);
    try
      FList.LoadFromStream(VStream);
    finally
      VStream.Free;
    end;
  except
    //
  end;
end;

constructor TStringListStatic.CreateByPlainText(const APlainText: string);
begin
  inherited Create;
  FList := TStringList.Create;
  FList.Text := APlainText;
end;

destructor TStringListStatic.Destroy;
begin
  FreeAndNil(FList);
  inherited;
end;

function TStringListStatic.GetCount: Integer;
begin
  Result := FList.Count;
end;

function TStringListStatic.GetItem(AIndex: Integer): string;
begin
  Result := FList.Strings[AIndex];
end;

function TStringListStatic.IndexOf(const S: string): Integer;
begin
  Result := FList.IndexOf(S);
end;

end.
