"""
Apply operation to bitmap font
(c) 2019--2024 Rob Hagemans, licence: https://opensource.org/licenses/MIT
"""

import sys
import logging
from types import SimpleNamespace as Namespace
from pathlib import Path

import monobit
from monobit.plumbing import (
    wrap_main, parse_subcommands, argrecord, GLOBAL_ARG_PREFIX
)
from monobit.plumbing.help import print_help

script_name = Path(sys.argv[0]).name

operations = {
    'to': monobit.save,
    **monobit.operations
}

global_options = {
    'help': (bool, 'Print a help message and exit.'),
    'version': (bool, 'Show monobit version and exit.'),
    'debug': (bool, 'Enable debugging output.'),
}

usage = (
    f'usage: {script_name} '
    + '[INFILE] [LOAD-OPTIONS] '
    + ' '.join(f'[{GLOBAL_ARG_PREFIX}{_op}]' for _op in global_options)
    + ' [COMMAND [OPTION...]] ...'
    + ' [to [OUTFILE] [SAVE-OPTIONS]]'
)


def help(command_args):
    """Print the usage help message."""
    print_help(command_args, usage, operations, global_options)


def version():
    """Print the version string."""
    print(f'monobit v{monobit.__version__}')


def main():
    command_args, global_args = parse_subcommands(operations, global_options=global_options)
    debug = 'debug' in global_args.kwargs
    with wrap_main(debug):
        if 'help' in global_args.kwargs:
            help(command_args)

        elif 'version' in global_args.kwargs:
            version()

        else:
            # ensure first command is load
            if not command_args[0].command and (
                    command_args[0].args or command_args[0].kwargs
                    or len(command_args) == 1 or command_args[1].command != 'load'
                ):
                command_args[0].command = 'load'
                command_args[0].func = operations['load']
                # special case `convert infile outfile` for convenience
                if len(command_args[0].args) == 2:
                    command_args.append(argrecord(
                        command='save', func=operations['save'],
                        args=[command_args[0].args.pop()])
                    )
            # ensure last command is save
            if command_args[-1].command not in ('to', 'save'):
                command_args.append(argrecord(command='save', func=operations['save']))

            fonts = []
            for args in command_args:
                if not args.command:
                    continue
                logging.debug('Executing command `%s`', args.command)
                operation = operations[args.command]
                if operation == monobit.load:
                    # no font/pack arg, pack return
                    fonts += operation(*args.args, **args.kwargs)
                elif operation.pack_operation:
                    # pack arg, pack return
                    fonts = operation(fonts, *args.args, **args.kwargs)
                else:
                    # font arg, font return
                    fonts = tuple(
                        operation(_font, *args.args, **args.kwargs)
                        for _font in fonts
                    )

if __name__ == '__main__':
    main()
