# Contributing to PuPHPeteer

Please, be sure to read this document _before_ creating an issue or a pull-request.

## Where to ask a question or report a bug

First, be sure to differentiate the original project [Puppeteer ![#40b5a4](https://placehold.it/10/40b5a4/000000?text=+)](https://github.com/GoogleChrome/puppeteer), developed by Google to interact with Chromium, from [PuPHPeteer ![#8846F0](https://placehold.it/10/8846F0/000000?text=+)](https://github.com/nesk/puphpeteer), an unofficial PHP bridge to call Puppeteer methods in PHP.

Here are some cases to help you choosing where to ask a question or report a bug. The first list item matching your case should be used:

1. You don't know how to do _[something]_ with **Puppeteer ![#40b5a4](https://placehold.it/10/40b5a4/000000?text=+)**: ask your question [on StackOverflow with the _puppeteer_ tag](https://stackoverflow.com/questions/tagged/puppeteer?sort=newest).
2. You don't know how to do _[something]_ with **PuPHPeteer ![#8846F0](https://placehold.it/10/8846F0/000000?text=+)**: ask your question [in PuPHPeteer's issues](https://github.com/nesk/puphpeteer/issues).
3. You encountered a bug:
    1. It is reproducible with **Puppeteer ![#40b5a4](https://placehold.it/10/40b5a4/000000?text=+)**: report the bug [in Puppeteer's issues](https://github.com/GoogleChrome/puppeteer/issues).
    2. it is reproducible only with **PuPHPeteer ![#8846F0](https://placehold.it/10/8846F0/000000?text=+)**: report the bug [in PuPHPeteer's issues](https://github.com/nesk/puphpeteer/issues).

## Provide logs with your bug report

Bug reports should contain logs generated by your reproducible example. To get them you must provide a logger to your PuPHPeteer instance. Say you have the following code in your bug report:

```php
use Nesk\Puphpeteer\Puppeteer;

$puppeteer = new Puppeteer;
$browser = $puppeteer->launch();
$browser->newPage()->goto('https://example.com');
$browser->close();
```

Require Monolog with Composer:

```shell
composer require monolog/monolog
```

And provide a Monolog instance to the `Puppeteer` constructor:

```diff
  use Nesk\Puphpeteer\Puppeteer;

- $puppeteer = new Puppeteer;
+ $logPath = 'path/to/your.log';
+
+ $logger = new \Monolog\Logger('PuPHPeteer');
+ $logger->pushHandler(new \Monolog\Handler\StreamHandler($logPath, \Monolog\Logger::DEBUG));
+
+ $puppeteer = new Puppeteer([
+     'logger' => $logger,
+     'log_node_console' => true,
+     'log_browser_console' => true,
+ ]);
+
  $browser = $puppeteer->launch();
  $browser->newPage()->goto('https://example.com');
  $browser->close();
```

Execute your code and `path/to/your.log` will contain the generated logs, here's an example of what you can get and provide in your bug report:

```
[2018-08-17 10:26:01] PuPHPeteer.INFO: Applying options... {"options":{"read_timeout":30,"log_browser_console":true,"logger":"[object] (Monolog\\Logger: {})","log_node_console":true}} []
[2018-08-17 10:26:01] PuPHPeteer.DEBUG: Options applied and merged with defaults {"options":{"executable_path":"node","idle_timeout":60,"read_timeout":30,"stop_timeout":3,"logger":"[object] (Monolog\\Logger: {})","log_node_console":true,"debug":false,"log_browser_console":true}} []
[2018-08-17 10:26:01] PuPHPeteer.INFO: Starting process with command line: 'node' '/Users/johann/Development/puphpeteer/node_modules/@nesk/rialto/src/node-process/serve.js' '/Users/johann/Development/puphpeteer/src/PuppeteerConnectionDelegate.js' '{"idle_timeout":60,"log_node_console":true,"log_browser_console":true}' {"commandline":"'node' '/Users/johann/Development/puphpeteer/node_modules/@nesk/rialto/src/node-process/serve.js' '/Users/johann/Development/puphpeteer/src/PuppeteerConnectionDelegate.js' '{\"idle_timeout\":60,\"log_node_console\":true,\"log_browser_console\":true}'"} []
[2018-08-17 10:26:01] PuPHPeteer.INFO: Process started with PID 18153 {"pid":18153} []
[2018-08-17 10:26:01] PuPHPeteer.DEBUG: Sending an instruction to the port 59621... {"pid":18153,"port":59621,"instruction":{"type":"call","name":"launch","catched":false,"value":[]}} []
[2018-08-17 10:26:01] PuPHPeteer.DEBUG: Received data from the port 59621... {"pid":18153,"port":59621,"data":"[object] (Nesk\\Puphpeteer\\Resources\\Browser: {\"__rialto_resource__\":true,\"class_name\":\"Browser\",\"id\":\"1534501561533.8093\"})"} []
```
