# flake8: noqa
import pytest

from chatlab import FunctionRegistry
from chatlab.builtins import os_functions
from chatlab.tools.files import (
    get_file_size,
    is_directory,
    is_file,
    list_files,
    read_file,
    write_file,
)
from chatlab.tools.shell import run_shell_command


def test_chat_function_adherence():
    assert len(os_functions) > 0

    for function in os_functions:
        assert function.__name__ is not None and function.__name__ != ""
        assert function.__doc__ is not None and function.__doc__ != ""

        fr = FunctionRegistry()
        schema = fr.register(function)
        assert schema is not None


# TODO: Determine if this is part of the testing suite on Windows
async def test_run_shell_command():
    command = "echo Hello, ChatLab!"
    result = await run_shell_command(command)
    assert "Hello, ChatLab!" in result

    command = "adsflkajsdg"
    result = await run_shell_command(command)
    assert "adsflkajsdg: command not found" in result


@pytest.mark.asyncio
async def test_list_files():
    directory = "chatlab/tools"
    files = await list_files(directory)
    assert isinstance(files, list)
    assert len(files) > 0


@pytest.mark.asyncio
async def test_get_file_size():
    file_path = "chatlab/tools/files.py"
    size = await get_file_size(file_path)
    assert isinstance(size, int)
    assert size > 0


@pytest.mark.asyncio
async def test_is_file():
    file_path = "chatlab/tools/files.py"
    assert await is_file(file_path)


@pytest.mark.asyncio
async def test_is_directory():
    directory = "chatlab/tools"
    assert await is_directory(directory)


@pytest.mark.asyncio
async def test_write_and_read_file(tmp_path):
    file_path = tmp_path / "test_file.txt"
    content = "Hello, ChatLab!"

    # Write content to file
    await write_file(str(file_path), content)

    # Read content from file
    result_content = await read_file(str(file_path))
    assert result_content == content
