<?php
/**
 * Class View
 *
 * A View of MVC design pattern manages the presentation.
 * Technically speaking it has those main roles:
 *
 * - Loads and manages HTML templates file. A template files implements the GUI design or
 *   or, generally, the output of the application.
 * - Renders/Parses template's variables (also known as placeholders) with data.
 *   A placeholder is something into braces like {PLACEHOLDER} located into template file.
 * - Parses and renders template's block with data. A block is a text between a comment in
 *   the format:  <!-- BEGIN Block Name -->Text<!-- END Block Name -->, located into
 *   template file.
 * - Manages repetitions of data that must be rendered into a block. For example each
 *   row of a database table's rows can be rendered by using design from a block
 * - Outputs the template with all its blocks and variables dynamically rendered with data.
 * - Provides functions for hiding or showing blocks as well as the global content.
 * - Parses and renders for error messages into an appropriate block. Errors are
 *   application messages previously stored into an array.
 *
 * @package framework
 * @filesource framework/View.php
 * @author Rosario Carvello <rosario.carvello@gmail.com>
 * @version GIT:v1.0.1
 * @note none
 * @copyright (c) 2016 Rosario Carvello <rosario.carvello@gmail.com> - All rights reserved. See License.txt file
 * @license BSD
 * @license https://opensource.org/licenses/BSD-3-Clause This software is distributed under BSD Public License.
 */

namespace framework;

use framework\exceptions\VariableNotFoundException;
use framework\exceptions\TemplateNotFoundException;
use framework\exceptions\NotInitializedViewException;
use framework\exceptions\BlockNotFoundException;

/** TODO
 * use HTMLPurifier;
 * use HTMLPurifier_Config;
 */

class View
{
    /**
     * @var string Stores template. Step by step it replaces static content of the template
     *             with some dynamic content generated by its parsing functions.
     */
    protected $tpl;

    /**
     * @var array Stores each opened blocks contained into the tpl variable.
     *            Each elements of blocks contain four section:
     *            - structure       Original content including BEGIN END pseudo tags
     *            - template        Content to process
     *            - last_parsed     Last parsed content  (after each call of setVar() or
     *                              parseCurrentBlock() method)
     *            - parsed          Final parsed content (after invoking serBlock() method)
     */
    protected $blocks;

    /**
     * @var string Stores the name of current opened block
     *
     */
    protected $currentBlock;

    /**
     * @var string Store the path depth of the template (to use for JS/CSS fieles inclusion)
     */
    protected $tplPathDepth;

    /**
     * View constructor.
     * Creates View object to manage a static html template.
     * The static template name must exist into APP_TEMPLATES_PATH folder and must have
     * an extension ".html.tpl".
     *
     * @param string|null $tplName The template name without extension. If null it assumes
     *                             the View class name
     * @throws TemplateNotFoundException. If static template file was not found
     */

    protected $tplFileName = "";

    function __construct($tplName = null)
    {
        $this->blocks = array();
        if (!empty($tplName))
            $this->loadTemplate($tplName);
        $this->tplFileName = $tplName;
    }

    /**
     * Loads a template file.
     *
     * @param string $tplName The template name without extension.
     * @throws TemplateNotFoundException. If static template file was not found
     */
    protected function loadTemplate($tplName)
    {
        $this->setTplPathDept($tplName);
        $tpl = APP_TEMPLATES_PATH . "/" . $tplName . ".html.tpl";
        $this->tpl = @file_get_contents($tpl);
        if ($this->tpl == false)
            throw new TemplateNotFoundException("Template $tpl non trovato", 101);
        $this->blocks = array();
    }

    /**
     * Loads a user custom template
     *
     * @param string $tplName The template name with its path but without .html.tpl extension
     * @throws TemplateNotFoundException If static template file not found
     */
    public function loadCustomTemplate($tplName)
    {
        $this->setTplPathDept($tplName);
        $tpl = RELATIVE_PATH . $tplName . ".html.tpl";
        $this->tpl = @file_get_contents($tpl);
        if ($this->tpl == false)
            throw new TemplateNotFoundException("Template $tpl non trovato", 101);
        $this->blocks = array();
    }

    /**
     * Sets path depth of given teplate reference
     * @param string $tplName
     */
    private function setTplPathDept($tplName)
    {
        $slashesOccurence = substr_count($tplName, '/');
        $this->tplPathDepth = str_repeat("../", $slashesOccurence);
    }

    /**
     * Get path depth of teplate reference
     *
     */
    public function getTplPathDepth()
    {
        return $this->tplPathDepth;
    }

    /**
     * Set TEMPLATE_PATH placeholder.
     *
     * @param bool $useSiteUrl If true (default) sse site url path. False use path depth
     */
    public function setVarTemplatePath($useSiteUrl = true)
    {
        if ($useSiteUrl) {
            $this->setVar("TEMPLATE_PATH", SITEURL . "/");
        } else {
            $this->setVar("TEMPLATE_PATH", $this->getTplPathDepth());
        }
    }

    /**
     * Checks if a template is loaded.
     *
     * @throws NotInitializedViewException. If template was not loaded
     */
    private function checkLoadedTpl()
    {
        if (!$this->tpl)
            throw new NotInitializedViewException("La vista non è stata inizializzata", 102);
    }

    /**
     * Checks if the given variable exist into the template or, previously opened, active
     * block.
     *
     * @param string $var The variable/placeholder name to check
     * @param bool|true $useBraces If true (default) the variable name contain braces
     * @return bool
     * @throws NotInitializedViewException If template was not loaded
     */
    public function checkVar($var, $useBraces = true)
    {
        $this->checkLoadedTpl();
        if ($this->currentBlock != "" && $useBraces) {
            $found = strpos($this->blocks[$this->currentBlock]["last_parsed"], "{" . $var . "}");
            if (!$found) {
                return false;
            } else {
                return true;
            }
        } else if ($useBraces) {
            $found = strpos($this->tpl, "{" . $var . "}");
            if (!$found) {
                return false;
            } else {
                return true;
            }
        } else if (!$useBraces) {
            $found = strpos($this->tpl, $var);
            if (!$found) {
                return false;
            } else {
                return true;
            }
        }
    }

    /**
     * Replaces a template or block variable/placeholder with a given value.
     * If a block is previously opened the replacement is made against the opened block.
     *
     * @param string $var The variable to replace
     * @param string $value The user value
     * @param bool|true $useBraces If true (default) the variable name contain braces
     * @throws VariableNotFoundException. If variable/placeholder was not found
     * @throws NotInitializedViewException If template was not loaded
     */
    public function setVar($var, $value, $useBraces = true)
    {
        $this->checkLoadedTpl();

        if (empty($value))
            $value = "";

        if ($this->currentBlock != "" && $useBraces) {
            $found = strpos($this->blocks[$this->currentBlock]["last_parsed"], "{" . $var . "}");
            if (!$found)
                throw new VariableNotFoundException("Variabile $var non trovata nel template.", 101);
            $this->blocks[$this->currentBlock]["last_parsed"] = str_replace("{" . $var . "}", $value, $this->blocks[$this->currentBlock]["last_parsed"]);
        } else if ($useBraces) {
            $found = strpos($this->tpl, "{" . $var . "}");
            if (!$found)
                throw new VariableNotFoundException("Variabile $var non trovata nel template.", 101);
            $this->tpl = str_replace("{" . $var . "}", $value, $this->tpl);
        } else if (!$useBraces && $var != null) {
            $this->tpl = str_replace("" . $var . "", $value, $this->tpl);
        } else {
            $this->tpl = $value . $this->tpl;
        }
    }

    /**
     * Replaces the given template block name with the given content.
     * It replaces current opened block when no block name was specified.
     * If also no content is given it uses the parsed content located inside $blocks,
     * a class attribute defined as an array.
     * From $blocks, it selects the value where key = the given block name or, alternatively,
     * the value where key = current opened block when no block name is given.
     *
     * @param string|null $content The content to replace to the block.
     * @param string|null $block The block to replace with content. Current opened block if null
     * @throws BlockNotFoundException. Block name not found into the template
     * @throws NotInitializedViewException
     */
    public function setBlock($content = null, $block = null)
    {
        $this->checkLoadedTpl();
        if ($block == null)
            $block = $this->currentBlock;
        if (!array_key_exists($block, $this->blocks)) {
            throw new BlockNotFoundException("Blocco non trovato o nessun blocco aperto correntemente.", 100);
        }

        if ($content == null)
            @$content = $this->blocks[$block]["parsed"];

        $var = $this->blocks[$block]["template"];

        $begin = "<!-- BEGIN " . $this->currentBlock . " -->";
        $end = "<!-- END " . $this->currentBlock . " -->";
        $blockWithContent = $begin . $content . $end;

        if (!empty($content) && isset($var)) {
            $this->tpl = str_replace($var, $blockWithContent, $this->tpl);
        }

        $this->closeCurrentBlock();
    }

    /**
     * Outputs to the screen the previously loaded and parsed template.
     *
     * @return void
     * @throws NotInitializedViewException If template was not loaded
     */
    public function render()
    {
        $this->checkLoadedTpl();
        $this->closeCurrentBlock();
        echo $this->tpl;
    }

    /**
     * Gets the parsed template.
     *
     * @return string The parsed template
     * @throws NotInitializedViewException If template was not loaded
     */
    public function parse()
    {
        $this->checkLoadedTpl();
        return $this->tpl;
    }

    /**
     * Replaces all tpl content with the given string.
     *
     * @param string $content
     */
    public function replaceTpl($content)
    {
        $this->tpl = $content;
    }

    /**
     * Opens a template block and sets it as current block.
     *
     * @param string $block the name of block to open
     * @throws BlockNotFoundException If block name was not found
     * @throws NotInitializedViewException If template was not loaded
     */
    public function openBlock($block)
    {
        $this->checkLoadedTpl();
        // Block never opened before
        if (!@array_key_exists("$this->blocks[$block]", $this->blocks)) {
            $regex = "/\<\!-- BEGIN $block --\>(.*?)\<\!-- END $block --\>/s";
            @preg_match_all($regex, $this->tpl, $result);
            if (empty($result[0][0]))
                throw new BlockNotFoundException("Blocco non trovato", 100);
            $block_template = $result[0][0];
            $block_structure = $result[1][0];
            // $block_array = array("template" => $block_template, "structure" => $block_structure, "last_parsed" => $block_structure);
            $this->blocks[$block]["template"] = $block_template;
            $this->blocks[$block]["structure"] = $block_structure;
        }

        $this->blocks[$block]["last_parsed"] = $this->blocks[$block]["structure"];
        $this->currentBlock = $block;
    }

    /**
     * Hides a block or global content.
     *
     * @param string|null $block The block name to hide. If is null hides all content
     * @throws NotInitializedViewException If template was not loaded
     */
    public function hide($block = null)
    {

        if (!empty($block)) {
            $this->checkLoadedTpl();
            $regex = "/\<\!-- BEGIN $block --\>(.*?)\<\!-- END $block --\>/s";
            @preg_match_all($regex, $this->tpl, $result);
            $this->tpl = str_replace($result[0], "", $this->tpl);
        } else {
            $this->tpl = "<!-- Hidden -->";
        }
    }

    /**
     * Returns the parsed content of the current opened template block.
     *
     * @return string The parsed content of the current opened block or null if no
     *                block was previously opened.
     * @throws NotInitializedViewException If template was not loaded
     */
    public function parseCurrentBlock()
    {
        $this->checkLoadedTpl();
        $block = $this->currentBlock;
        if ($block == "") {
            return "";
        } else {
            $error_reporting = error_reporting();
            error_reporting(0);
            $this->blocks[$block]["parsed"] .= $this->blocks[$block]["last_parsed"];
            $this->blocks[$block]["last_parsed"] = $this->blocks[$block]["structure"];
            error_reporting($error_reporting);
            return $this->blocks[$block]["last_parsed"];
            //return $this->blocks[$block]["parsed"];
        }
    }

    /**
     * Closes the current and opened block.
     *
     * @throws NotInitializedViewException If template was not loaded
     */
    public function closeCurrentBlock()
    {
        $this->checkLoadedTpl();
        $this->currentBlock = "";
    }

    /**
     * Parses for errors
     *
     * @param array $errors An array containing the error's messages
     * @throws BlockNotFoundException If block <!-- BEGIN/END ValidationErrors --> was not found
     * @throws BlockNotFoundException If block <!-- BEGIN/END RecordErrors --> was not found
     * @throws VariableNotFoundException If placeholder {RecordError} was not found
     * @throws NotInitializedViewException If template was not loaded
     */
    public function parseErrors($errors)
    {
        $this->checkLoadedTpl();

        if ($errors[0] == "" || !isset($errors)) {
            $this->hide("ValidationErrors");
        } else {
            $this->openBlock("RecordErrors");
            foreach ($errors as $error) {
                $this->setVar("RecordError", $error . "<br/>");
                $this->parseCurrentBlock();
            }
            $this->setBlock();
        }
    }

    /**
     * @return null|string The tpl file name
     */
    public function getTplFileName()
    {
        return $this->tplFileName;
    }

    /** TODO
     * Cleans a string against XSS attack
     *
     * @param string $string String to purify
     * @param string $charset Charset string. Default is CHARSET constant
     * @return string
     */
    public function xssCleanString($string, $charset = CHARSET)
    {
        return null;
    }
}
