import os
import re
import sys
import glob
import json
import html
import validate
import cairosvg
import subprocess
import collections

def guard(result, message, metadata = None):
    """Make sure the result is a positive result. Exit otherwise.
        
    Args:
        result (bool): A boolean indicating a positive or negative result.
        message (str): The message to show if the result is negative.
        metadata (obj): Optional metadata to show if the result is negative.

    """

    if result:
        return

    print('Guard failed! {}'.format(message))

    if metadata:
        print(metadata)

    sys.exit(1)

def generate_html(manifest):
    """Generate an HTML content structure based on the given manifest.
        
    Args:
        manifest (obj): The issuer icon manifest data.

    Returns:
        str: The HTML content

    """

    contents = """
        <!DOCTYPE html>
        <html lang="en-us">
            <head>
                <meta charset="utf-8">
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <title>Issuer Icons for Raivo OTP</title>
                <meta name="description" content="Scalable Vector Graphics (SVG) logo's and app icons of organisations around the world.">
                <meta name="author" content="Mobime and Raivo contributors">
                <link rel="shortcut icon" href="https://raivo-otp.com/img/app-icon.png">
                <meta name="apple-itunes-app" content="app-id=1459042137">
                <style>
                    html{font-size:62.5%;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Helvetica,Arial,sans-serif,"Apple Color Emoji","Segoe UI Emoji","Segoe UI Symbol";line-height:1}body{font-size:2rem;background-color:#fff}.subPageBody{background-color:#fff}h1{font-size:3rem}h2{font-size:2rem}h3{font-size:2rem}body{-webkit-font-smoothing:antialiased;-moz-osx-font-smoothing:grayscale;background-color:#fff}a:link,a:hover,a:visited,a:active{color:#ec1438;text-decoration:none}*,*::before,*::after{-webkit-box-sizing:border-box;-moz-box-sizing:border-box;box-sizing:border-box;margin:0;padding:0}@font-face{font-family:octicons-link;src:url(data:font/woff;charset=utf-8;base64,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) format("woff")}.markdown-body .octicon{display:inline-block;fill:currentColor;vertical-align:text-bottom}.markdown-body .anchor{float:left;line-height:1;margin-left:-20px;padding-right:4px}.markdown-body .anchor:focus{outline:none}.markdown-body h1 .octicon-link,.markdown-body h2 .octicon-link,.markdown-body h3 .octicon-link,.markdown-body h4 .octicon-link,.markdown-body h5 .octicon-link,.markdown-body h6 .octicon-link{color:#000;vertical-align:middle;visibility:hidden}.markdown-body h1,.markdown-body h2,.markdown-body h3,.markdown-body h4,.markdown-body h5,.markdown-body h6{color:#000}.markdown-body h1:hover .anchor,.markdown-body h2:hover .anchor,.markdown-body h3:hover .anchor,.markdown-body h4:hover .anchor,.markdown-body h5:hover .anchor,.markdown-body h6:hover .anchor{text-decoration:none}.markdown-body h1:hover .anchor .octicon-link,.markdown-body h2:hover .anchor .octicon-link,.markdown-body h3:hover .anchor .octicon-link,.markdown-body h4:hover .anchor .octicon-link,.markdown-body h5:hover .anchor .octicon-link,.markdown-body h6:hover .anchor .octicon-link{visibility:visible}.markdown-body{-ms-text-size-adjust:100%;-webkit-text-size-adjust:100%;color:rgba(0,0,0,0.8);line-height:1.5;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Helvetica,Arial,sans-serif,"Apple Color Emoji","Segoe UI Emoji","Segoe UI Symbol";font-size:16px;line-height:1.5;word-wrap:break-word}.markdown-body .pl-c{color:#6a737d}.markdown-body .pl-c1,.markdown-body .pl-s .pl-v{color:#005cc5}.markdown-body .pl-e,.markdown-body .pl-en{color:#6f42c1}.markdown-body .pl-s .pl-s1,.markdown-body .pl-smi{color:#24292e}.markdown-body .pl-ent{color:#22863a}.markdown-body .pl-k{color:#d73a49}.markdown-body .pl-pds,.markdown-body .pl-s,.markdown-body .pl-s .pl-pse .pl-s1,.markdown-body .pl-sr,.markdown-body .pl-sr .pl-cce,.markdown-body .pl-sr .pl-sra,.markdown-body .pl-sr .pl-sre{color:#032f62}.markdown-body .pl-smw,.markdown-body .pl-v{color:#e36209}.markdown-body .pl-bu{color:#b31d28}.markdown-body .pl-ii{background-color:#b31d28;color:#fafbfc}.markdown-body .pl-c2{background-color:#d73a49;color:#fafbfc}.markdown-body .pl-c2:before{content:"^M"}.markdown-body .pl-sr .pl-cce{color:#22863a;font-weight:700}.markdown-body .pl-ml{color:#735c0f}.markdown-body .pl-mh,.markdown-body .pl-mh .pl-en,.markdown-body .pl-ms{color:#005cc5;font-weight:700}.markdown-body .pl-mi{color:#24292e;font-style:italic}.markdown-body .pl-mb{color:#24292e;font-weight:700}.markdown-body .pl-md{background-color:#ffeef0;color:#b31d28}.markdown-body .pl-mi1{background-color:#f0fff4;color:#22863a}.markdown-body .pl-mc{background-color:#ffebda;color:#e36209}.markdown-body .pl-mi2{background-color:#005cc5;color:#f6f8fa}.markdown-body .pl-mdr{color:#6f42c1;font-weight:700}.markdown-body .pl-ba{color:#586069}.markdown-body .pl-sg{color:#959da5}.markdown-body .pl-corl{color:#032f62;text-decoration:underline}.markdown-body details{display:block}.markdown-body summary{display:list-item}.markdown-body a{background-color:transparent}.markdown-body a:active,.markdown-body a:hover{outline-width:0}.markdown-body strong{font-weight:inherit;font-weight:bolder}.markdown-body h1{font-size:2em;margin:.67em 0}.markdown-body img{border-style:none}.markdown-body code,.markdown-body kbd,.markdown-body pre{font-family:monospace,monospace;font-size:1em}.markdown-body hr{box-sizing:content-box;height:0;overflow:visible}.markdown-body input{font:inherit;margin:0}.markdown-body input{overflow:visible}.markdown-body [type=checkbox]{box-sizing:border-box;padding:0}.markdown-body *{box-sizing:border-box}.markdown-body input{font-family:inherit;font-size:inherit;line-height:inherit}.markdown-body a{color:#ec1438;text-decoration:none}.markdown-body a:hover{text-decoration:underline}.markdown-body strong{font-weight:600}.markdown-body hr{background:transparent;border:0;border-bottom:1px solid #dfe2e5;height:0;margin:15px 0;overflow:hidden}.markdown-body hr:before{content:"";display:table}.markdown-body hr:after{clear:both;content:"";display:table}.markdown-body table{border-collapse:collapse;border-spacing:0}.markdown-body td,.markdown-body th{padding:0}.markdown-body details summary{cursor:pointer}.markdown-body h1,.markdown-body h2,.markdown-body h3,.markdown-body h4,.markdown-body h5,.markdown-body h6{margin-bottom:0;margin-top:0}.markdown-body h1{font-size:32px}.markdown-body h1,.markdown-body h2{font-weight:600}.markdown-body h2{font-size:24px}.markdown-body h3{font-size:20px}.markdown-body h3,.markdown-body h4{font-weight:600}.markdown-body h4{font-size:16px}.markdown-body h5{font-size:14px}.markdown-body h5,.markdown-body h6{font-weight:600}.markdown-body h6{font-size:12px}.markdown-body p{margin-bottom:10px;margin-top:0}.markdown-body blockquote{margin:0}.markdown-body ol,.markdown-body ul{margin-bottom:0;margin-top:0;padding-left:0}.markdown-body ol ol,.markdown-body ul ol{list-style-type:lower-roman}.markdown-body ol ol ol,.markdown-body ol ul ol,.markdown-body ul ol ol,.markdown-body ul ul ol{list-style-type:lower-alpha}.markdown-body dd{margin-left:0}.markdown-body code,.markdown-body pre{font-family:SFMono-Regular,Consolas,Liberation Mono,Menlo,Courier,monospace;font-size:12px}.markdown-body pre{margin-bottom:0;margin-top:0}.markdown-body input::-webkit-inner-spin-button,.markdown-body input::-webkit-outer-spin-button{-webkit-appearance:none;appearance:none;margin:0}.markdown-body .border{border:1px solid #e1e4e8 !important}.markdown-body .border-0{border:0 !important}.markdown-body .border-bottom{border-bottom:1px solid #e1e4e8 !important}.markdown-body .rounded-1{border-radius:3px !important}.markdown-body .bg-white{background-color:#fff !important}.markdown-body .bg-gray-light{background-color:#fafbfc !important}.markdown-body .text-gray-light{color:#6a737d !important}.markdown-body .mb-0{margin-bottom:0 !important}.markdown-body .my-2{margin-bottom:8px !important;margin-top:8px !important}.markdown-body .pl-0{padding-left:0 !important}.markdown-body .py-0{padding-bottom:0 !important;padding-top:0 !important}.markdown-body .pl-1{padding-left:4px !important}.markdown-body .pl-2{padding-left:8px !important}.markdown-body .py-2{padding-bottom:8px !important;padding-top:8px !important}.markdown-body .pl-3,.markdown-body .px-3{padding-left:16px !important}.markdown-body .px-3{padding-right:16px !important}.markdown-body .pl-4{padding-left:24px !important}.markdown-body .pl-5{padding-left:32px !important}.markdown-body .pl-6{padding-left:40px !important}.markdown-body .f6{font-size:12px !important}.markdown-body .lh-condensed{line-height:1.25 !important}.markdown-body .text-bold{font-weight:600 !important}.markdown-body:before{content:"";display:table}.markdown-body:after{clear:both;content:"";display:table}.markdown-body>:first-child{margin-top:0 !important}.markdown-body>:last-child{margin-bottom:0 !important}.markdown-body a:not([href]){color:inherit;text-decoration:none}.markdown-body blockquote,.markdown-body dl,.markdown-body ol,.markdown-body p,.markdown-body pre,.markdown-body table,.markdown-body ul{margin-bottom:16px;margin-top:0}.markdown-body hr{background-color:#e1e4e8;border:0;height:.25em;margin:24px 0;padding:0}.markdown-body blockquote{border-left:.25em solid #dfe2e5;color:#6a737d;padding:0 1em}.markdown-body blockquote>:first-child{margin-top:0}.markdown-body blockquote>:last-child{margin-bottom:0}.markdown-body kbd{background-color:#fafbfc;border:1px solid #c6cbd1;border-bottom-color:#959da5;border-radius:3px;box-shadow:inset 0 -1px 0 #959da5;color:#444d56;display:inline-block;font-size:11px;line-height:10px;padding:3px 5px;vertical-align:middle}.markdown-body h1,.markdown-body h2,.markdown-body h3,.markdown-body h4,.markdown-body h5,.markdown-body h6{font-weight:600;line-height:1.25;margin-bottom:16px;margin-top:24px}.markdown-body h1{font-size:2em}.markdown-body h1,.markdown-body h2{border-bottom:1px solid #eaecef;padding-bottom:.3em}.markdown-body h2{font-size:1.5em}.markdown-body h3{font-size:1.25em}.markdown-body h4{font-size:1em}.markdown-body h5{font-size:.875em}.markdown-body h6{color:#6a737d;font-size:.85em}.markdown-body ol,.markdown-body ul{padding-left:2em}.markdown-body ol ol,.markdown-body ol ul,.markdown-body ul ol,.markdown-body ul ul{margin-bottom:0;margin-top:0}.markdown-body li{word-wrap:break-all}.markdown-body li>p{margin-top:16px}.markdown-body li+li{margin-top:.25em}.markdown-body dl{padding:0}.markdown-body dl dt{font-size:1em;font-style:italic;font-weight:600;margin-top:16px;padding:0}.markdown-body dl dd{margin-bottom:16px;padding:0 16px}.markdown-body table{display:block;overflow:auto;width:100%}.markdown-body table th{font-weight:600}.markdown-body table td,.markdown-body table th{border:1px solid #dfe2e5;padding:6px 13px}.markdown-body table tr{background-color:#fff;border-top:1px solid #c6cbd1}.markdown-body table tr:nth-child(2n){background-color:#f6f8fa}.markdown-body img{box-sizing:content-box;max-width:100%}.markdown-body img[align=right]{padding-left:20px}.markdown-body img[align=left]{padding-right:20px}.markdown-body code{background-color:rgba(27,31,35,0.05);border-radius:3px;font-size:85%;margin:0;padding:.2em .4em}.markdown-body pre{word-wrap:normal}.markdown-body pre>code{background:transparent;border:0;font-size:100%;margin:0;padding:0;white-space:pre;word-break:normal}.markdown-body .highlight{margin-bottom:16px}.markdown-body .highlight pre{margin-bottom:0;word-break:normal}.markdown-body .highlight pre,.markdown-body pre{background-color:#f6f8fa;border-radius:3px;font-size:85%;line-height:1.45;overflow:auto;padding:16px}.markdown-body pre code{background-color:transparent;border:0;display:inline;line-height:inherit;margin:0;max-width:auto;overflow:visible;padding:0;word-wrap:normal}.markdown-body .commit-tease-sha{color:#444d56;display:inline-block;font-family:SFMono-Regular,Consolas,Liberation Mono,Menlo,Courier,monospace;font-size:90%}.markdown-body .blob-wrapper{border-bottom-left-radius:3px;border-bottom-right-radius:3px;overflow-x:auto;overflow-y:hidden}.markdown-body .blob-wrapper-embedded{max-height:240px;overflow-y:auto}.markdown-body .blob-num{-moz-user-select:none;-ms-user-select:none;-webkit-user-select:none;color:rgba(27,31,35,0.3);cursor:pointer;font-family:SFMono-Regular,Consolas,Liberation Mono,Menlo,Courier,monospace;font-size:12px;line-height:20px;min-width:50px;padding-left:10px;padding-right:10px;text-align:right;user-select:none;vertical-align:top;white-space:nowrap;width:1%}.markdown-body .blob-num:hover{color:rgba(27,31,35,0.6)}.markdown-body .blob-num:before{content:attr(data-line-number)}.markdown-body .blob-code{line-height:20px;padding-left:10px;padding-right:10px;position:relative;vertical-align:top}.markdown-body .blob-code-inner{color:#24292e;font-family:SFMono-Regular,Consolas,Liberation Mono,Menlo,Courier,monospace;font-size:12px;overflow:visible;white-space:pre;word-wrap:normal}.markdown-body .pl-token.active,.markdown-body .pl-token:hover{background:#ffea7f;cursor:pointer}.markdown-body kbd{background-color:#fafbfc;border:1px solid #d1d5da;border-bottom-color:#c6cbd1;border-radius:3px;box-shadow:inset 0 -1px 0 #c6cbd1;color:#444d56;display:inline-block;font:11px SFMono-Regular,Consolas,Liberation Mono,Menlo,Courier,monospace;line-height:10px;padding:3px 5px;vertical-align:middle}.markdown-body :checked+.radio-label{border-color:#0366d6;position:relative;z-index:1}.markdown-body .tab-size[data-tab-size="1"]{-moz-tab-size:1;tab-size:1}.markdown-body .tab-size[data-tab-size="2"]{-moz-tab-size:2;tab-size:2}.markdown-body .tab-size[data-tab-size="3"]{-moz-tab-size:3;tab-size:3}.markdown-body .tab-size[data-tab-size="4"]{-moz-tab-size:4;tab-size:4}.markdown-body .tab-size[data-tab-size="5"]{-moz-tab-size:5;tab-size:5}.markdown-body .tab-size[data-tab-size="6"]{-moz-tab-size:6;tab-size:6}.markdown-body .tab-size[data-tab-size="7"]{-moz-tab-size:7;tab-size:7}.markdown-body .tab-size[data-tab-size="8"]{-moz-tab-size:8;tab-size:8}.markdown-body .tab-size[data-tab-size="9"]{-moz-tab-size:9;tab-size:9}.markdown-body .tab-size[data-tab-size="10"]{-moz-tab-size:10;tab-size:10}.markdown-body .tab-size[data-tab-size="11"]{-moz-tab-size:11;tab-size:11}.markdown-body .tab-size[data-tab-size="12"]{-moz-tab-size:12;tab-size:12}.markdown-body .task-list-item{list-style-type:none}.markdown-body .task-list-item+.task-list-item{margin-top:3px}.markdown-body .task-list-item input{margin:0 .2em .25em -1.6em;vertical-align:middle}.markdown-body hr{border-bottom-color:#eee}.markdown-body .pl-0{padding-left:0 !important}.markdown-body .pl-1{padding-left:4px !important}.markdown-body .pl-2{padding-left:8px !important}.markdown-body .pl-3{padding-left:16px !important}.markdown-body .pl-4{padding-left:24px !important}.markdown-body .pl-5{padding-left:32px !important}.markdown-body .pl-6{padding-left:40px !important}.markdown-body .pl-7{padding-left:48px !important}.markdown-body .pl-8{padding-left:64px !important}.markdown-body .pl-9{padding-left:80px !important}.markdown-body .pl-10{padding-left:96px !important}.markdown-body .pl-11{padding-left:112px !important}.markdown-body .pl-12{padding-left:128px !important}.icon{margin-right:0 !important}.icon>svg{width:1em;height:1em;stroke:currentColor;stroke-width:2;stroke-linecap:round;stroke-linejoin:round;fill:none}.table{display:table !important;width:100%}.table tbody tr td ul{margin-bottom:0px}.anchor-hover h1,.anchor-hover h2,.anchor-hover h3,.anchor-hover h4,.anchor-hover h5,.anchor-hover h6{line-height:1 !important}.anchor-hover h1 .icon,.anchor-hover h2 .icon,.anchor-hover h3 .icon,.anchor-hover h4 .icon,.anchor-hover h5 .icon,.anchor-hover h6 .icon{display:none;font-size:80%;position:absolute;margin-top:2.5px;margin-left:-1.5%}.anchor-hover h1 small,.anchor-hover h2 small,.anchor-hover h3 small,.anchor-hover h4 small,.anchor-hover h5 small,.anchor-hover h6 small{opacity:0.6;font-weight:normal;font-size:0.6em}.anchor-hover:hover{text-decoration:none !important}.anchor-hover:hover .icon{display:block}.icon-link{display:table}.icon-link .icon{vertical-align:text-top;margin-right:2px !important}.icon-link:hover{text-decoration:none !important}.icon-link:hover .text{text-decoration:underline !important}.headerBackground{height:115px;background-color:rgba(0,0,0,0.1)}@media only screen and (max-width: 768px){.headerBackground{height:80px}.appDescription{min-height:225px}}.subPageHeaderBackground{background-color:#fff}.container{display:grid;margin:auto;max-width:1170px;padding-left:15px;padding-right:15px;grid-template-columns:repeat(12, 1fr);grid-template-rows:115px 876px auto auto;grid-column-gap:30px;grid-template-areas:"h h h h h h h h h h h h" "p p p p p i i i i i i i" "c c c c c c c c c c c c" "f f f f f f f f f f f f"}.subPageContainer{grid-template-rows:115px auto auto auto;background-color:#fff}@media only screen and (max-width: 768px){.container{grid-template-rows:80px 811px auto auto}}@media only screen and (max-width: 992px){.container{grid-column-gap:0px;grid-template-columns:1;grid-template-rows:115px auto auto auto auto;grid-template-areas:"h h h h h h h h h h h h" "i i i i i i i i i i i i" "p p p p p p p p p p p p" "c c c c c c c c c c c c" "f f f f f f f f f f f f"}}.page{margin-top:30px;margin-bottom:0px;grid-column:1/14}@media only screen and (max-width: 768px){.page{margin-top:30px;margin-bottom:70px;grid-column:1/-1}}header{grid-area:h;display:flex;margin-right:-15px}.logo{display:flex;justify-content:flex-start;align-items:center;height:115px;margin-right:30px}.logo>p{color:#fff;white-space:nowrap;display:flex;font-weight:bold;margin-left:15px}@media only screen and (max-width: 768px){.logo{height:80px;margin-right:0px}.logo>p{display:none}}.headerIcon{width:50px;height:50px;-webkit-clip-path:url(#shape);clip-path:url(#shape)}@media only screen and (max-width: 768px){.divider{position:relative;min-width:1px;max-width:1px;background-color:rgba(127,127,127,0.2);display:inline-block;margin-left:15px}}.scroll{display:flex;width:100%;align-items:center;justify-content:flex-end;height:115px;white-space:nowrap;overflow-x:auto;overflow-y:hidden;-webkit-overflow-scrolling:touch;-ms-overflow-style:-ms-autohiding-scrollbar}@media only screen and (max-width: 768px){.scroll{height:80px;padding-top:100px;padding-bottom:100px;margin-top:-60px;justify-content:flex-start}}.scroll::-webkit-scrollbar{display:none}nav>ul{color:#fff;display:flex;list-style-type:none}nav>ul li{padding-left:30px}@media only screen and (max-width: 768px){nav>ul li{padding-left:20px}}nav>ul li:first-child{padding-left:0px;padding-left:15px}nav>ul li:last-child{padding-right:15px}nav>ul li a:link,nav>ul li a:visited{text-decoration:none;color:rgba(255,255,255,0.6)}nav>ul li a:hover,nav>ul li a:active{text-decoration:none;color:#fff}.subPageHeaderBackground .logo>p{color:#000}.subPageHeaderBackground nav>ul li a:link,.subPageHeaderBackground nav>ul li a:visited{color:rgba(0,0,0,0.6)}.subPageHeaderBackground nav>ul li a:hover,.subPageHeaderBackground nav>ul li a:active{color:#000}.appInfo{grid-area:i;display:flex;flex-wrap:wrap;padding-top:140px;align-content:flex-start}@media only screen and (max-width: 992px){.appInfo{padding-top:50px;justify-content:center}}.appIconShadow{display:flex;filter:drop-shadow(0px 5px 10px rgba(0,0,0,0.1)) drop-shadow(0px 1px 1px rgba(0,0,0,0.2))}.appIconLarge{width:120px;height:120px;-webkit-clip-path:url(#shape120);clip-path:url(#shape120)}.appNameSubtitleContainer{display:flex;flex:0 1 auto;flex-direction:column;align-items:start;justify-content:center;margin-left:30px}.appName{color:#fff}.appSubtitle{color:#fff;font-weight:normal;margin-top:6px}@media only screen and (max-width: 768px){.appNameSubtitleContainer{width:100%;margin-left:0px;align-items:center;justify-content:center}.appName{margin-top:30px;text-align:center}.appSubtitle{margin-top:13px;text-align:center}}.appDescriptionContainer{font-size:2.5rem;font-weight:normal;width:100%;align-items:flex-start;margin-top:30px;flex:0 1 auto;line-height:1.5}.appDescription{color:#fff}@media only screen and (max-width: 992px){.appDescription{text-align:center}}.downloadButtonsContainer{display:inline-block;margin-top:42px;filter:drop-shadow(0px 5px 10px rgba(0,0,0,0.1)) drop-shadow(0px 1px 1px rgba(0,0,0,0.2))}@media only screen and (max-width: 992px){.downloadButtonsContainer{text-align:center}}.playStore{height:75px;margin-right:24px}@media only screen and (max-width: 992px){.playStore{margin-right:24px;margin-bottom:0px}}@media only screen and (max-width: 528px){.playStore{margin-right:0px;margin-bottom:24px}}.appStore{height:75px}.iphonePreview{grid-area:p;background-size:400px auto;background-repeat:no-repeat;margin-top:68px}.iphoneScreen{width:349px;-webkit-clip-path:url(#screenMask);clip-path:url(#screenMask);margin-left:26px;margin-top:23px}.videoContainer{width:349px;height:755px;-webkit-clip-path:url(#screenMask);clip-path:url(#screenMask);margin-left:26px;margin-top:23px}.videoContainer>video{width:349px;height:755px}@media only screen and (max-width: 1070px){.iphonePreview{background-size:370px auto}.iphoneScreen{width:322px;margin-left:24px;margin-top:22px}.videoContainer{width:322px;height:698px;margin-left:24px;margin-top:22px}.videoContainer>video{width:322px;height:698px}}@media only screen and (max-width: 992px){.iphonePreview{display:flex;background-size:260px auto;background-position:center 0;margin-top:47px;justify-content:center;padding-bottom:75px}.iphoneScreen{width:226px;height:488px;-webkit-clip-path:url(#screenMask);clip-path:url(#screenMask);margin:0px;margin-top:17px}.videoContainer{width:226px;height:488px;margin-left:0px;margin-top:17px}.videoContainer>video{width:226px;height:488px}}.features{grid-area:c;display:flex;flex:0 1 auto;align-content:flex-start;justify-content:flex-start;flex-grow:1;flex-wrap:wrap;margin-top:93px}.feature{display:flex;padding-top:63px;padding-left:15px;padding-right:15px;width:calc(100%/3)}.feature:nth-child(-n+3){padding-top:0px}.feature:nth-child(3n){padding-right:0px}.feature:nth-child(3n+1){padding-left:0px}.iconBack{color:#e6e6e6}.iconTop{color:#ec1438}.socialIconBack{color:#e6e6e6}.socialIconTop{color:#666}.featureText{margin-left:18px}.featureText>h3{color:#000}.featureText>p{color:#666;margin-top:8px;line-height:1.5}@media only screen and (max-width: 992px){.features{flex-grow:1;flex-direction:row;flex-wrap:wrap;margin-top:11px}.feature{display:flex;padding-top:41px;padding-left:15px;padding-right:15px;width:100%}.feature:nth-child(-n+3){padding-top:41px}.feature:nth-child(1){padding-top:0px}.feature:nth-child(3n){padding-right:15px}.feature:nth-child(3n+1){padding-left:15px}}@media only screen and (max-width: 375px){.features{flex-grow:1;flex-direction:row;flex-wrap:wrap;margin-top:11px}.feature{display:flex;padding-top:41px;padding-left:0px;padding-right:0px;width:100%}.feature:nth-child(-n+3){padding-top:41px}.feature:nth-child(1){padding-top:0px}.feature:nth-child(3n){padding-right:0px}.feature:nth-child(3n+1){padding-left:0px}}footer{grid-area:f;display:flex;flex-wrap:wrap;justify-content:center}.footerLinks{display:flex;text-align:center;padding-top:70px;padding-bottom:10px;font-size:1.75rem}.footerText{color:#666;font-size:1.5rem;display:block;line-height:1.5;width:100%;text-align:center;padding-bottom:10px}.footerIcons{padding-bottom:70px;display:flex;flex:0 0 100%;justify-content:center}.footerIcons .padding{padding:0px 5px}.footerIcons .padding a{font-size:2rem;font-weight:100;display:block;background:#e6e6e6;color:#666666;padding:10px;border-radius:50px}.footerIcons .padding a .icon{display:block;width:1em;height:1em;overflow:hidden}.footerLinks a{margin-right:30px}.footerLinks a:last-child{margin-right:0px}@media only screen and (max-width: 768px){.footerLinks{flex-direction:column;padding-bottom:70px}.footerText{padding-bottom:25px !important}.footerLinks a{justify-content:stretch;margin-right:0px;margin-top:20px}.footerLinks a:first-child{margin-top:0px}}@media only screen and (max-width: 992px){.footerText{color:#666;display:block;line-height:1.5;width:100%;text-align:center;padding-bottom:10px}.footerIcons{padding-bottom:70px;display:flex}}.hidden{display:none}
                    
                    small {
                        color: rgba(0,0,0,0.6);
                    }

                    .issuer-icon { 
                        width: 152px;
                        display: block;
                        float: left;
                        text-align: center;

                        margin: 0px 5.5px 10px 5px;
                    }

                    .issuer-icon a {
                        background: #f5f5f5;
                        display: block;
                        padding: 23.5px 0px;
                        border-radius: 5px;
                        width: 153px;
                        height: 153px;
                    }

                    .issuer-icon a:hover {
                        background: #222222;
                    }

                    .issuer-icon img {
                        width: 100px;
                        height: 100px;
                        margin-bottom: 0px;
                    }

                    .issuer-icon p {
                        width: 152px;
                        text-align: center;
                        text-overflow: ellipsis;
                        overflow: hidden;
                        white-space: nowrap;
                    }
                </style>
            </head>
            <body class="subPageBody">
                <div class="headerBackground subPageHeaderBackground">
                    <div class="container subPageContainer">
                        <header>
                            <div class="logo">
                                <div class="appIconShadow">
                                    <img class="headerIcon" src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAABAAAAAQACAYAAAB/HSuDAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxMBAJqcGAAAtGFJREFUeJzsvQu0ZVV9pxse8pAqoCgKQbEQpYXEOARCUUAw8REjwXRr346JQaOJRoFcEjWaSCc9Etu+UaOGtKgk2sTceC8YbB/DqBiCJOYmab23TdIhKcUgj0IeRfGQsgoQtOrc36pam9pnn/1Ya+855/8/5/y+Mb6RdFrhnD3XnGv+fmuufb7v+wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAlNx+x4XHyB+WL5a/KS+V/l38tN8nb5f2tO1sH/+/b2//MX7f/nUvbf0bzz3qG3N/69wMAAAAAAACoEoXyY+XL5R/K/ykflkuRfLj9dzT/rlc0/27r3x8AAAAAAACgSBS695Fnyd+VN0QM+129of1Zmp9pH+vPBwAAAAAAACBrFK7Xy7fJmxyE/kne3P6M660/LwAAAAAAAICsuOWIDT8kPyG/J5cGOgj7Kxz6+Zqf9ePNz279+QEAAAAAAAC4RuH5++V/l7tuWR6up5o46M9yV/s7fL/15wkAAAAAAADgiluOOH2VAvOlo0/8M3enfJ9cZf35AgAAAAAAAJij8P9ceYtcWq55gF/AZb9L87s91/pzBgAAAAAAADBDwfjN8nsrw/84rUN958A/zuZ3/DXrzxsAAAAAAAAgKQrD+8nLugV/T8XAwj/vB5rf3frzBwAAAAAAAIjOrbvf9z/9s2HCf5Y2vzvfCwAAAAAAAADlovB/jPyKXBroIJAnc+j3bj6DY6zHAwAAAAAAACA4CrzPkJtvXR6EV2gd0iMF/nE2n8UzrMcFAAAAAAAAIBgKus+X35oV/nMuBeb83R5oPhvr8QEAAAAAAABYGAXcX5SPzhv+PZYCgX+X5rP5RetxAgAAAAAAAJgLhdr95CWhg3+sssD6Z2w/K/5CAAAAAAAAAOSDguxq+TkHoTo3r24+O+vxAwAAAAAAAJjJ5iNOP14h9l8chOlcbT67463HEQAAAAAAAGAiCv9ny61yaaCDQJ2NQ59b8xmebT2eAAAAAAAAACtQYL1APrJ5eZBdogzoHPpHbT7L863HFQAAAAAAAGA3m9eefoCC6genBX/KgM6hf5zNZ3uA9TgDAAAAAABAxSj8HyP/Vi49Zv+AW3whMO9nMvS5Np/xMdbjDQAAAAAAABWiQHq6vH1Z+B91gfBrHdpNAv/y0D9q81mfbj3uAAAAAAAAUBEKoq+WD08N/wHLAK/FQKjfqcfn2Hzmr7YefwAAAAAAACic29aefpC8vFfwj1gGxC4IYv6cC36GlzdjYX09AAAAAAAAQIEocB4v/0EuDbtwGRC5EHBjgM9p5LNvxuJ46+sCAAAAAAAACkJB80Xy/tHwH7wIKKkQCPh5TPncmzF5kfX1AQAAAAAAAJmjcLmffJvcNS38Ry0DcigFIvy+PT7vXe0Y7Wd9vQAAAAAAAECG3LZ249EKlX/ZJ/gnKwNSFwSJfocFP+tmrI62vm4AAAAAAAAgIxT+XyDvkkt7XDicpi8EMjDk59qOVTNmL7C+fgAAAAAAAMA531y7cT/5Nrlzb/gflTLAWegfdWc7hrwSAAAAAAAAACtRYHyS/Eu5NGrsIqDUUiDWZzRuPMaM2181Y2p9XQEAAAAAAIAjFBTPlVvHhf/ZRUC8MiCnUiD2ZzDps58xZs2Y8lcCAAAAAAAAakfh8CD5XrlrVvjvVgTELwMsy4HUv9u0z7nHeO1qx/gg6+sNAAAAAAAADFAgfIa8vk/w914GlGGQ0D/OZqyfYX3dAQAAAAAAQCJuX7txH3mhwuBDi4Z/ygD3oX/UZswvbK4B6+sQAAAAAAAAIqLgd5T8tFwaGLoE6FYG1F4ITP9sYo3J0Lg318BR1tcjAAAAAAAARECB79/Ju29fHgSjFwHdyoDSC4HZv3/Mz3/CmDfXwr+zvi4BAAAAAAAgEAp5q+Xlk4J/yiKgXyGQaynQ7XdL8Tl3HPPm2lhtfZ0CAAAAAADAAijYnS1v7hr+UxcB/QsBL+VA/5835ec5x3g318jZ1tcrAAAAAAAA9ERh7iD5TrlznvBvWQYsVgr40OLzCjDOO9trhj8XCAAAAAAAkAN3rN24UW66I0woNC8CvJYD1p9BwOC/zPba2Wh9HQMAAAAAAMAEFNoOku+U371jT5B7zNAh0VMZsGhxYP0zeQj9bfAf9rvttcRpAAAAAAAAAE+0T/2/Ohr8UxQBuZYBuRlr7GZcM5wGAAAAAAAA8IDC2cHyXfJ7s8J/iiKAMiCP0N8h+I+eBvjd5lqzvt4BAAAAAACqRIHsOfLGPsE/ZRFAGeAv9PcM/qM219pzrK97AAAAAACAarjzyDMOl5fLXYuE/9RlAIWATeBfMPSPuqu99g63ngcAAAAAAABFo+D1v8k75NKwuRUBNRcCKT/fkNfFyDV3Z3MtWs8HAAAAAACA4lDYOlZ+cjT4l1IElFoKWH2GEYP/qM01eaz1/AAAAAAAAMgehav95Rvk9lnhP1YRYFkG5FIMWH82oUN/h+A/7Pb2Gt3fer4AAAAAAABkiQLVBvmPfYJ/7CLASxmQqiiw/l1Sh/6ewX/U5lo93XreAAAAAAAAZMNdR55xuHy/3LlI+I9dBORQBpRorLEMdK3tbK9dviQQAAAAAABgEgpN+8hXyjvl0rDeiwDKgDxDf8DgvzRyzd4lX9Vc09bzCgAAAAAAwBUKSqfIvxsN/jkWAZQB/kN/xOA/anNNn2I9vwAAAAAAAMxpj/t/QH5vVvjPsQigEPAT+BMH/2G/117jvBYAAAAAAAD1seXIM/ZVIHqNvKdP8I9RAqQuAygE0gb+kKF/jvA/7D3tNb+v9fwDAAAAAABIgsL/c+Q/yKUt84epIoqAWkoBq8/USfB/zPaab67951jPQwAAAAAAgGgo9KyXn9yyNwgthSoBYhQBlmVArsWA9WcVI/SHCP5D4X/YZi4cZz0vAQAAAAAAgqGQs0r+H/KhceGfIiCfgsD6dy0k+A/7UDs3VlnPUwAAAAAAgLlRqNlfvlbeOSv451IEeC8DajPWGCcI/qPe2c6V/a3nLQAAAAAAQC/uPvKMc+SmvsE/dAkQswigDCgr9IcK/nOE/2E3NXPHev4CAAAAAADM5O4jzzxF/qVCzNLARUqAXIoACoE8A7+j4L/boXnzV81csp7PAAAAAAAAK1BYeYr8iNwpl/a4LNC4KQJSlQEUAn4Df8jQHzj4t+6eQzvbOfUU6/kNAAAAAADQBP918r3ykb3Bf9RwJUCuRQCFgH3g9xj8V4b/sXPokXaOrbOe7wAAAAAAUCFbjzxzlQLJb8vtk4P/+BLAYxFgUQbUWghYfM6hr5XwwX9i+B+2mWtvlfzFAAAAAAAAiI+C/8Hy9XLr1k7BnyKg1mLA+jMsKPgvU/PunnYOHmy9HgAAAAAAQIEobBwkf0XesXVPCFmmdQkQowjwUgZ4LQisP4NUoT9U8F80/I+Zd3e0c/Ig6/UBAAAAAAAKYOu6Mw+UFylk3D4u+M9fAuRVBHguAzDemHsI/hPC/7DN3LyomavW6wUAAAAAAGSIwsQB8pfkN+XSY84OIwuVADkUAZQBPow5vvGC/8JP/Ve6d35+s52zB1ivHwAAAAAAkAH3KDzICxQiblsW/KOWAPFOA8QuAigEygj8oYN/gqf+o+F/2GbuXnAPRQAAAAAAAIyjDf6vk7fKpXsmhX+KAMqAhKYas0KC/2O2c/jWdk5TBAAAAAAAwO7gf6A8X95yz97gsMz0JUDcIiBlGUAp4C/sxwj9IYJ/iPA/YQ7f0s5xviMAAAAAAKBGFAYOlb8u75wU/EOWABQBFALWgT/74D9f+B/2rnbOH2q9/gAAAAAAQAK0+T9avkM+0CX4+ygC4n1RoIcyoPRiwPozjRH6xwf/SMf9Fw/+oz7QrgFHW69HAAAAAAAQgXvXnfk0+Yfy4XmCf+cSIOFrAbGKAC9lQE7lgPVnkyr0hwr+gd/1n9eH2zXhBOv1CQAAAAAAAqDN/SnyT+X35NKwUYuAaKcB0hYBXsuA2IWB9c/uLfQnD/7hn/ovc2QtaNaGq5q1wnq9AgAAAACAObh33VnPk9eMhv6kJUBhRUDOZUDJxh5zT8E/Qvgf9Zpm7bBevwAAAAAAYAbauD9Onie/Ipf2GrcECHUaYP4iwKYMoBAoM/CHDP29gn/kp/4dwv/QmrF7DWnWksdZr2sAAAAAADDEfevOOkK+RZv125cH/+4lgKfTADkWAZQB+Yd+r8E/wVP/0fA/bLOmXNysMdbrHAAAAABA1WhTfqK8TO6QSwMtSwDPRUDKMoBCII/APzn0lxH8Fwz/S0Pryo52rTnRet0DAAAAAKgGbcD3lS+Sfy533bd8k96hBHDwSkCyEsBHEUAp4Cfsxwz+eYb/yevEhLVlV7v2NGvQvtbrIQAAAABAkWizfbj8VfmNSaE/ZAngsQgorQyosRSw/qzNQ7+b4D9X+B+1WYveLNdYr48AAAAAAEWgzfVp8r+NHvPPqQTwVwT4LQNKKAesP7MUod9r8E8Y/od9sF2jNlivlwAAAAAA2XH/urNWy/Pl398/34bcZREwswRwVAR4LAOsiwLr3zFM6PcX/DN76j/Vds1q1q7V1usoAAAAAIBrtGn+Ifkhuf3+PZvpZZZQAvgtAsopA2o1Zuj3HPw9hP8x69X2di07zXpdBQAAAABwgzbIq+Tr5FfGhX5vJUAdRQBlQC7GDv1lBP/k4X/Ur7Rr3Crr9RYAAAAAwARthk+VfzjpaX/MEsDTaYBYRUCqMoBCwFPgNwz9iYO/h6f+HcP/6KmADzZrn/X6CwAAAAAQHW18j5JvlP/YN/R7LAHqKwIoBEoN/OUFf3fhf9R/bNfCo6zXZQAAAACAYHxr3VkHypfKT8tHFw3+oUsAb6cBYhYBccqA2YUApUCosB8+8M8d+g2Cv5en/gHC/7CPal38s3aNPNB6vQYAAAAAmItvHXXWmfL98j5tbJeGDVkClHoaoHMR4K4M6FYI1F4KdP2MYo1RzNBvF/zdP/Vf4cjaeF+7Zp5pvX4DAAAAAMxEG9f18i3ya3JpmdmXABQBsQuBEouBvr977LGoMfhnEv6XRtbMZg29uFlTrdd1AAAAAIDH0AZ1jXy1vFbuXBH8p5QAoYsAL68E5FoEpCkD5isFPJYEi/4OqT7rRa4H/8E/uyP/Y9fBKevmznZtbdbYNdbrPQAAAABUyLeO+uHV2oy+XP6ZfGRq6C+yBLA7DdCrCFiwDEhbCIQrB3yZ/vNbdMz7XF8xrm9PT/2Nw/+ozVr7mXbtXW19HwAAAACAgnngqB8+WP6Uwv/H5UOyz8Y1aQlAERC+CLArA7wXBNafRaDQT/BPf+S/X/gfcfca3KzFP9Wszdb3BwAAAAAoAG0sD5T/Vl4ht8ulB3YH/2EpASxfC7AqA3wVAvUZagytQ3/I4/4Vhf/HbNfk7e0a3azV/CUBAAAAAOhO+6T/xfKP5bce2LvJXGYuJQBFQJoygEIgk8DfM/QT/N2H/1GbNfv/bNdwTgYAAAAAwEq0UVwjXyE/IR+cFPpTFQGeS4BcigDrMoBCwFHgdxT6vQb/ZOE/XvAf54Ptmv5zzRpvfZ8BAAAAAEO0IXyi/CV5rXy0T+jP9TRA2hIg4yIgQhlAMZAo6C8Q+nMK/t7Df+Kn/l18tF3rmzX/Sdb3HwAAAABIgDZ+J8q3yC/LXYuE/lxLAM+nAWIXAXOXAZELgRqKgRSf37xjG/ua6zsHcg7+TsP/qLvae0BzLzjR+r4EAAAAAIHQ5u4g+ePyUnljyMBPCZB3EbBQGZCoEMitHEj9mSwyfimuL8/Bv+LwP84b23tEc684yPq+BQAAAAA92HbUDx8nL9BG7tMP9Hyfv7YSgCIgUBlgUAjEKBCsf8aYgT9V6K81+Gcc/kdt7hl/Ji9o7iXW9zMAAAAAGEGbtP3lj8p3yevl0sDU4T9oCVD0aQC/RUCQMiCDQiALA4xDyusmdPDPJfyHDf6m4f8xh+4j17f3luYes7/1/Q4AAACgSrQRO1H+7/KT8oFtyzdsS5QA5RYBqcuAYIUApUD0sJ868M8X+ssJ/hWE/1EfaO85zb2H7w4AAAAAiIU2W0+Q58kPy9umBX5KAF8lQElFQNAyoNZiIMLnZ3Ed5BL8Cf/Bwv84b2vvSc296Wjr+yQAAABAtmgztUqeKy9pj2Du6hv6fZYAFAExigCrMiBaITCsdWB3EvKtA//8oZ/g3yf4ZxT+R93V3qsuae9dq6zvowAAAABu0Wbp0HbT9A75d/LRRQO/pxIg19MAtkVAPqcCkpcCs8wozHsN+yme9ucW/Et96h8g/I/z0fZe9o723nao9X0WAAAAwAxthtbKF7dPS/5efi9G4K+pBMjtNEDsIsBLGeCmFMhA6zFaPPTbBv/kT/0J/338Xnuvu6S99621vg8DAAAAREObnfXyZ+X75T9vC3Ckv/oSoJDTACmKAG9lQO3FgPVnHj70lxv84zz1ry78j3NXey98f3tvXG99nwYAAACYC21kDpZnyTfJT8jbLcO+/xIgr1cCfBQBZZwKKLkcsP7s0gT/7tdrjsE/Tvj38b6/cfif5O3tPfNN7T30YOv7OQAAAMAKtu19uv/78svbIr2/Twng55WAmCVAqiIgx0LAuiyw/h3tA7+f4G/y1J/wn9pH23vq72/jlAAAAABYoA3Ik7bteX/xbfJz8m4Hm6RsS4CVRUBerwTELgIoA5DQX8JTfz9H/jMK/5O8u733vq29Fz/Jel8AAAAAhaCNxTHyJ+Vb5WfknQ42P5QADk8DlFYEUAaUYYjroPrgv47wn4F3tffot7b37GOs9w8AAADgGG0WHiefJV8h3yX/Qm5xsKmpuATI8zSAvyIgTBlAIZCHoca67zWWe/BP8dSf8J/cLe29/F3tvb25xz/Oer8BAAAAidm25wj/C7bt+ZKhj8j/JR9xsFlxofUGdWURkOdpgBRFgGUZQCHgw5Dj6S30mwb/dWU99a8w/E+yudc39/zm3t/sAZq9AK8QAAAA5I5u6PvIp8hz2pv85dv2fJHQAw42IO613qgGLwEMTwOkKgKsywAKgRwDv8/Qnyr4x3vqT/jP0GZv0OwR/mjbnj1Ds3do9hD7WO9nAAAAYAjdnA+Xp8nz5G/LK+RX5A4HG4rstd60pnglgCIgThlAKeAx7M8X+qsJ/uvCBn8P4d/6HlKAzV6i2VM0e4vf3rZnr9HsOQ633v8AAAAUi2606+RG+TPyN+SH5d9tK+Db93PQegO7sgiIcxqgxCJg/jIgXiFAMZAy6M8f+FOG/jKCP0/9K3Trtj17kWZP0uxNXrZtz17lKOt9EwAAgGu0UTn6gT2NevPnfF4v3yv/TP7zNp7ku9B6I7uyBMj/NEDqImCxMiB+IVB6OZDy81tknFNfkynnWy1P/Qn/Lmz2Ls0eptnLNHuaZm/T7HFOa/Y81vsuAACAKOgmt1qeKM+W/0FeJH9HfkR+UX5DfodNSx5ab2ijlACVFgGLlwFpCwHvJYH1Z7HoWFpcf2UEf8I/zn0f/U67B/piuyf6nXaP9B/aPVOzd1ptvY8DAIDK0c3ocfLI9sb0bPlT7Q3rbfJD8tPyS/JW+RAbl/K03tiOLwICbeQnhITSi4AwZYB9IVCXi4+X1bVWTvBfHv6t10TuoXk4x5g+1O6pvtTusT7U7rkuavdgz273ZOse4E8cAgCANjr7P3DUWWtan9p6ijxD/ph8ifwZ+Tr5Ft083i4vk1fKz8m/k5vk7XIHGxiccwOToASIexogdRFgWQaEKwQoBbyEfcvAnzr0Tw3+68p96s+907+JroMd7Z5tU7uHa/ZyzZ6u2ds1e7y3tHu+n2n3gD/W7glPHdonHtHuG/e33scCABSFNiL7yafI58rXyN+S75NXyS/Kf5I3ybvk/XLn0Cbmu+3/bdht4zY7WsB7ar+JYSPjX+vrY9hSTwN4KALCFwKUAinCvnXgtwr+tT71557pX+vrY6/99oRj5tK2MfvP7w79/+9s/293tXvYf2r3tFe1e9zfave8z2v3wPtZ78cBAKKgTcnR8kXyN9pF8Gvy0alPKQJKAYClb2qinAagCEhUCNRWDsT57KyviZqCv6fwb30PwLzulQuG/xg2e+Eb2r3xb7R7Zb4kEQDyQ4vXsfLn5R/Jr3c+mhhRSgAsf2MTqQRwVgR4KwPiFgK5lQTpPgfrMfcQ+tME/+Xh33qN4x6Zn9bXyV7dhf9p99Svt3voZi99rPW+HgBgLFqgninfJv9p7g1LRHkVAOvY4KQ/DWBZBHgsA2xKgbK1HktvoX/mfbTwp/7cG/PR+jrZ68JH/1OG/3E2e+v/0uy1rff7AFA5WoieKH9dXh908xLJHAsANjr5aH2djEoR4E/rIJ2D1mNE8Pcd/Lkn5qP1dbLcrMP/qNe3e+8nWucAAKgILTpnyavko1E2MRGlBMCaNj3RSgDHRUAuZUDN5YD1Z51b6E8X/H2Hf+u1HfO8D3oP/wvcMx9t9+JnW+cCACgYLTLnyC8n2dBEklcBsL7NT71FQK5lQO4FgfVnVUroJ/hzD8xV6+tlr9kf/e9qszc/xzonAEBBaFF5tvxS0o1NRHMsANgA5af19TLOmouAksqAmCWC9c9orfU1SvDn3pe71tfLcn2H/wj3x6YI+BHr3AAAGaPN0JO1kFwpd5lsciJKCYD1bYT2GLUEyKQIoBDAgdbXoF3wJ/xj6fe86sL/wF3t3v3J1jkCADLivnVn7qON0WvlNvMNTyRzfRWADVGeWl8z46QIoAyoVetrjeDPfa5Era+Z5VZz9H+a39Z69zq5j3WuAADnKPw/Qf65TLZZsmpeKQCw3s3RXi2LAK9lAIVAWVpfS3Pf+yoM/tzj8tX6ulmu7/Cf4t63dw088xp5tHW+AACnaIF4trx9T/g/M+kGihKADVINWl83k1xZAlAEUAjkq/W14j34E/6x7Hsb4X/5mrh7T39Hs8e3zhkA4AwtDK+V390b/peXAC42RRHkVQBks7TX6CVAIWXAyg0WEvgXvL/FmmtHEfyxpvsZR/+X35+W7embPf7rrPMGADhgz/v+Z/7OyuDPKQBKAKxn07RcioBFN11Yc9jvdV+rPPhzH8tb62tnuf7Df4r72vI1c+zevtnz870AALXShv/3TQ7/dZwC6F8CWN/k2DyVovW1M0svRUCOZcD4zRiWHPZ73ccI/ty/Mtf62lkp4X/5PWfq3v59lAAAlaLJf+ns8J+2BLBalCkBkI3UZJMVAR02ZtbhLvwmDa3HItn9K+a8OYrgjzXfs/yH/xT3r47hf+Cl1jkEABKjiX9x9/DPqwDeSwDrzQCWuKFaqacioKQyYPpGLn+tP0vz+xXBn/tVYVpfQ8vl6P/Ke0bn/f3F1nkEABKhCf9yuatfAcCrAJQAWN/GarxJiwDKgDk2fwR68/tT7DlxFMEfuUcR/sfdA3rt7Zss8HLrXAIAkdFEP0U+2D/813EKoH8JYH3zY4NVotbXUVc9FgGUARjDzvcQgj/3poK1vo5W6j/8p7gnzfn0f2CTCU6xzicAEAlN8FXypvnCP6cAKAGQzdZ4kxcBlAGYQD+h/yyCP5prfS2tlKf/AxcI/wNvbjKCdU4BgAhocv/XxcI/JQAlALLpmuzKIoAyAPPSc+jPKfhzHypL62tppYT/gQHC/8D3WucUAAiMJvbZcmfIAoBXASgBsOYN2GTNioCeGz3rsIn29rovpLqGx8wf6znNvadera+nleYR/lPcYxY8+j9qkxHOts4rABCIe4888yBN6q+FCf/1nALoXwJY3xTZiJWu9fXUV9MiYI6Nn3UYRYf3k5TX65j5Yj2HuefUrfX1NF7C/8CA4X/gDfIg69wCAAG498gzL5YhF4hqSgBOAaBHra+rvo4vAhKXAT03g9ZBFY3uH6mvyQlzw3rOcq9Bn/eaPJ7+p1jXIoT/3TaZwTq3AMCCaCIfKu9rCoDwJQCvAlACIJuzfrooAubcGFoHWYx0r7C4/sbMA+u5yf0Ffd9f8gj/Ke4VgY/+D4f/pTYzHGqdXwBgATSJ/9Mg/HMKYH4pAdCr1tfWvLo5FUAhkLX5BP6zJl7z1nOR+wr6v68Q/oeN+PR/4H+yzi8AMCeawIfLbw0XADmfAph7sxfInAsANmvla319zau7ImCBQoBiwOnab30tTbjGrece9xLM515C+B8Y+en/wCY7HG6dYwBgDjR53zwa/mOeAii9BMj9FAAbtzq0vsYW0W0ZEGhTaR2iczDIem19rRQY+rl/1KH1NTbePJ7+5xz+xxQAjW+2zjEA0BNN3H3ljeMKAF4FmF9KAMxB62sshK7LgIClQI3lQNB12foaKDz0c9+oR+trbLx5hP/0BUD08L/UZoh9rfMMAPRAk/ZHJ4V/XgVYTEoAzEXr6yyEk4sAZ2VAhFIgx5Ig6vprPbY9Qj/BH3PS+jobL+F/fPiPevR/1OdZ5xkA6IEm7eVpCwBKAEoA9Kr1tRbK7MqARMVAcVqPVeWhn/tDXVpfa+Ml/KcI/x0KgA9b5xkA6Eh7/P/uWQUArwIsJiUA5qb19RbSrMsAygH7z5zQzz0Bza+38eYT/tMXAEnD/1KbJXgNACAHNFk3dgn/vAqwmCUUAGz46tP6eothUWVACSWB9WdF6OdegBnfC/IpAHJ++t8jJ2y0zjUA0AFN1rfZFQCUAJQAmIPW11wsp5cBBRUCGMHp1471tc36j+Wv/4T/FOG/ZwHwNutcAwAd0GT9cp8CIPdXASgB2ARiiRvBMFIG4HTrC/ys+XVrfd1NlvA/uQAwC/+NX7LONQAwA03UA+TDXgqAGk4BUAJgCVpfeynkdADW+pSfdR59r/P5hP/Knv4vtZniAOt8AwBT0CQ9rW/4pwQIIyUAlqD19ZdSCoEaJPCztqPvtZ3w7zj8DzzNOt8AwBQ0SS+ctwDgVYDF7F8AUAKgX62vQQtnFwKUAr6dPX7W1xjrOVpofQ1Otv++qfTwv7wAcBH+Gy+0zjcAMAVN0g94LAAoAfIqANg04kDr69DSboUApYDXsF9r4GcNx2Gtr8PpEv4zePrf+AHrfAMAU9AkvWaRAqCEEsDyBkEJgKVqfS16sXspQDGQMujXHvaHtV4r0I/W1+J08wn/qQoAp+G/8RrrfAMAU9Ak/ddFCwBeBVhcSgAsWetr0pv9SgHKgUVCPmF/vNZrAvrT+pqcLuF/egHgKvw3/qt1vgGAKWiSfttnAUAJQAmApWl9XXp2vlKg1IJgsc/Ceiw9a70GoE+tr8vpEv5Thv9ABcC3rfMNAEzgniPPPPCeAOG/lFcBKAHYaGIara/N3Fy8HLAqDuL93NZjkpvWcx79an1tTpfwPz38u3z6v9smY1jnHAAYgybnE0MWALwKYFEAUAJgvlpfn6UYtyCw0fozLUXrOY6+tb4+p9t/P2S9h8v96X/IPNBkDOucAwBj0ORcf8+eSeq4AKAEoATAGrS+TmuTMF+u1nMZ89D6Op0u4T91+A/89L/xOOucAwBj0OR8ZugCgFcBKAHYkOIiWl+riLlqPXcxH62v1ekS/meHf/dP/xufaZ1zAGAMmpxPv2fvRKUEoARgc4qutL5mEb1rPUcxP62v2ekS/gsJ/40nWuccABjD8CsAvArgrwCYrwSwvnmzWcXwWl+3iN60npOYp9bX7WzzCv82BYD7o/8D11vnHAAYgybn2lgFACUAJQAbV4yh9fWLaKX13MO8tb5+Z0v4zzn8jykAjrTOOQAwBk3Ox48WAP5LgPpeBaAEQByv9XWMGFvrOYZlaH0dz5bwPzv8Z3P0f+DjrHMOAExgXAFACUAJwMYWc9P6ekYMpfVcwrK0vp5nS/gvMPwvWecbAJhCngVA+lcBPJQApX0pIBtdjKH1NY04r9ZzB8vT+pqeLV/6160AyOroPwUAgHcmFQCUAJQAbHqxBK2vb8RJWs8NLFvr63u2hP9Swz8FAIBzphUA/ksAXgWgBEDsrvV1jmg9B7AOra/z2eYX/lMWADkf/acAAMiAvAsASoCSSgA2x5hS62sd69H6Wsd6tL7Wu0n4twr/FAAAsJtZBUBOJUCqxZkSgM0ylqf1dY/laH0tY51aX/fdJPx3LwDyDf8UAADO6VIAUAJQArCBxpq0vv4xP62vWaxb6+u/m4T/WsI/BQCAc7oWAP5LgPSvAlACsKHGOrSeC+hP62sScaD1XOgm4b97+M/3vX8KAIBMuOfIMwopACgBKAEQ02g9L5C1CLHRel50k/BvGf4pAABgBXsKAEqA3AsASgBEW63nCbLmYF1az5Nu5hn+UxYAJYb/JldY5xsAmIIm6YOllgC1nQIouQRgU445aj1nkHUFy9N6znSX8N+vACgn/MuHrfMNAExBk/QBywIgTglQ76sAlACIvrWeQzVrPfaIi2o9h7pL+Ld++h96r96zAHjAOt8AwBSWFwDlnQKgBKAEQMxB63lVktZjiRhD63nVXcK/dfg3fvpPAQDgnZUFACUAJYB/rTdiiKm1nnOetB4LxNRaz7nuEv4J/xQAAO7xUgBQAlACEAQQF9N6PjKnEcNqPR/7SfgvLfxTAAAUyvgCoMwSIOUCTwlAYEDMVeYhor3W91bCf+wCoNjwTwEA4J3JBUBJrwJQAlACICIi5qH1PZXwn1/4d3L0nwIAIAc0Se9ftADIowSo+1UASgBERET/Wt9Lawn/qQuAnI7+99n/T8gQ37LONwAwBU3SzZMLgDJfBaAEKLcEoAhARMQctb53Ev4J/4Ge/jduts43ADCF2QUAJQAlACUAIiJiLK3vmYR/wn/A8E8BAOCdrZ0KgDK/D4ASoOwiwHpDh4iIOEvre2Xs4E/4r+a9fwoAgFzYeuQZXw9ZAORWAqS8EVACUAQgIiI2Wt8bCf8pC4CSwn+nAuBG63wDAFPYeuQZ12/tVACU9ioAJQAlACIiYnqt74mEf8J/xKf/S022sM43ADCFQQFQZwlg8yoAJQAlACIi1qn1vZDwnzL8V/Xe/yD8UwAAeGe4AKAEoAQovQSgCEBERAut732Ef8J/ovBPAQDgHU3Sv41VAOT2fQCUAJQAiIiIobW+5xH+ywj/oQuAwO/9DxcA/8M63wDAFDRJ/2br8klb4SkASgBKAERExPBa3+sI/4T/xE//G//GOt8AwBTGFQCUAJQAtZQAFAGIiBhD63sb4Z/wbxT+KQAAvKNJ+lfzFwDllgCpbx7llAB5FgHWG0VERCxH63tayuBfe/hfXgDUF/4nFAB/bZ1vAGAKmqSfHVcAeP8+AEoA70WA9UaGIgAREdNqfQ9LHf6t9yiE/5jhf+6n/42ftc43ADCFaQVAvSWA3asAlAD2Wm8gERExP63vXYR/y/Bf2jf+LxT+KQAAvDOrAOD7ACgBaiwBKAIQEbGL1vcqwn+Z4d+2AFgo/FMAAHhHk/RjYQoASgBKAEoARESsR+t7FOGf8G8R/jsUAB+3zjcAMAVN0itmFQD1vgpACRCuBKAIQETEMrS+J1kEf8J/DeE/yNP/xius8w0ATEGT9E+6FACUAJQAlACIiFi71vciwj/h33n4b/yIdb4BgClokl7WtQCgBKAEqL0EoAhARKxT63sP4Z/wn0n4b7zMOt8AwBQ0ST/QpwDw/qWAlAC5FAHWGyFKAERE7Kb1Pccq/FvvMQj/qQqAoOGfAgDAO5qk74hTAFACUAKUXQJQBCAilq31PYbwT/j3EP7nKADeYZ1vAGAK8xQAdb8KQAkQtgTIvwiw3qAiImJ4re8tVsGf8J82/McoAAyP/lMAAOSAJulvzFMAUAJQAlACUAQgIpam9b2E8E/4zzz8N/6mdb4BgClokl48bwFACWBXAFAC+NV684qIiP21vncQ/v2E/+UFAOF/Di+2zjcAMAVN0l+JXwBQAtRUAlAE2G9kERGxu9b3DII/4d9b+F+wAHi9db4BgClokl64SAEQowQIvShSAlACUAQgIuKo1vcIwj/hP30BED38N15onW8AYAqapL+waAGQw6sAlACUABQBiIjYaH1PIPwT/j0//Q+QC15tnW8AYAqapOeFKAAoAWy/FLDcEqCcIsB6w4uIiPb3Ag/Bn/A/LfyX+uf+koX/xvOs8w0ATOHutWe8NFQBQAlACRCvCLDeaFEEICLmrPXa7yX8W+8FCP/Fh//Gn7bONwAwhbvXnvEimbgAoASgBKi7BKAIQERMo/VaT/gn/OcQ/kMWAE22sM43ADAFTdKfaAqA2kqAeAUAJUDcEqCsIsB6Y4yIWLLWa7yX4E/4twv/MQoA5+G/8VzrfAMAU9AkPTt0ARCjBMjrFICPEsBzEUAJQBGAiBhL6zWd8O87+Occ/vsVAGnD/1ABcLZ1vgGAKWiSnjkoAGxKgFJfBaAEiF8EWG/IKAIQET1pvYZ7C//W93jCv/+n/xHCf+NZ1vkGAKagSfrM4QKAEoASIK8SgCIAEbF2rddsb8Gf8E/4Nwz/jc+0zjcAMAVN0qeNFgCUAJQAlAA+tN5UIyJ61nqNJvwT/gn/KzOEPME63wDAFDRJjx1XANT2pYCUAD6kCKAIQEScpfWaTPAn/Occ/iN86d+oT7bONwAwBU3Sw2IXAJQAlACUABQBiIiLar0GE/4J/4T/TgXAYdb5BgCmoEl6wKQCgBIgbhFACRCzBKAIQEQsRes113PwJ/x3Df9x93aE/2UeaJ1vAGAGmqi7KAFsSgCKgNhFgPXGjiIAEXFerddY7+Hf+h6dR/An/CcO/0vWuQYAOqDJ+sC0iez9SwEpASgBKAIQEcvRek31HvwJ/4R/Z1/6N+wD1rkGADqgybplVgFACUAJYC0lAEUAIpat9RqaQ/i3vhcT/gn/M7zbOtcAQAc0WTd3KQC8lwAxFmJKAF9yGoAiABHL03rNzCH4E/7LD/8xjv4nDv+Nm61zDQB0QJP1er8FACUAJQAlAEUAIpao9RpJ+Cf8lxz+jQqAf7bONQDQAU3W/9m1AKAEKL8EoAgoT+tNPiLisNZrIsG/nOBP+HcV/hu/Yp1rAKADmqx/06cAoASgBPAiJQBFACLmo/UaSPgn/BP+o4b/xr+xzjUA0AFN1s/1LQAoASgBPEkRQBmAiD61XusI/oR/wn+y8N94tXWuAYAOaLJ+cp4CgBKAEsCTYUoAigBExBBar205Bn/CP+E/8/Df+CnrXAMAHbh77cYr5i0AKAEoAbxJCUARgIh2Wq9luYZ/63sn4Z/wHyD8N15hnWsAoAN3r934h9JFAUAJ4LcEyKkI4DQAZQAiptN6zco5+OcU/q33IIT/NOF/sQJg4wetcw0AdECT9T2LFACUAOlKAA9FgPUGJH0RYL3BpAhARJ9ar1G5h3/re2RO4X90L0T49xj+dxcA77HONQDQgS1rN751TwFACeC3BPB1GsB6I5K+BKi7CKAMQMSB1muRvXU99fcX/uPvtwj/c4f/xrda5xoA6MCWtRvfvLcAyLEE6LeoUgJQBFAEUAQgYj+t1x57Cf6E/1Thv1sB4DD8N77ZOtcAQAe2rN14/pZlBYCfEiDGKQBKAEoAioAwWgcSRIyr9Rrjw3D3Det7IOG/nPDv6Ev/luUH5YkLrHMNAHRAk/W8LXsmLSUAJUDxJUDYIsB6Y+pH66CCiGG0Xkt8WV/wJ/wT/hcI/43nWecaAOiAJutPbtk7cSkBKAEoAigC5tY6vCDifFqvHb4k+BP+Cf9zhP/Gf2udawCgA5qsz9myfPIGKQAoAdKWAF6KAOsNjF0JQBEwqnWgQcTpWq8R/qzzuL+X8J/6m/4J/2EKgJEM8VzrXAMAHdBkPW20AAhVAoRcqCgB8jkNYL2RoQjwp3XQQcQ9Wq8FPq03+PsM/2n2TjWH/1BP/8fkh9Oscw0AdECT9aTpBQAlQB5FgK8SgCKAImCS1gEIsTat57xfCf4eLOWpf23hf0IBcJJ1rgGADmiyPmlcAUAJQAlACUAJEFvrYIRYqtZz27+Efw8S/osK/43HWucaAOiAJuthkwoASgBKgFpLAIqA9FoHJsTctZ7DeVh38Cf8E/4jhv/Gw6xzDQB0ZFoBQAmQdwlAEUARkKvWYQrRu9ZzNC8J/tZ7gPHBn/BfUPhv3Mc60wBARzRht1MClFYCcBqAIqAcrYMWohet52J+Evz9hv80eyHCf7Lwv8M6zwBADzRp75pVAFACUAJQBIQuASgC5tE6gCGm1nrO5WnYtdr63pN78Cf8Fx/+G++yzjMA0ANN2hu7FACh/jwgJQAlQM4lAEWAL63DGWJoredU3hL8Cf+E/zDhf28B0DEjfMM6zwBADzRpv9K1AKAEyL8EoAigCChZ6/CG2FfrOVOGBH//wZ/wX3D4b/yKdZ4BgB5o0n5hvgKAEiDnIsB6Y1BKCUAR4FvrcIc4qvWcKEuCfx7hP//gX2P471kAXGedZwCgB5q0n+hTAFACUAJQAlAE5Kp1+MP6tL7my5TgT/gn/DsK/2O9K4B3HjHO0zt7x3K/K++Xt8h/kV+Ql8lfkS+UR1lnMoBkaJL+0TwTmxKAEoAiIEUJQBEQW+uAiOVofS2Xb/j11fqeUVLwJ/wT/h2H/y7ukv98+xGnv1e+WB5indEAoqGJesm8E7zGEoDvBaAEsCsCKANSaR0k0b/W12g9xllLre8RpYV/3vcn/Gce/h/z9r0+KK+S58p9rfMaQFA0WX9rkYlOCZBzCcBpAIoA7Kp14ETCfl0S/HMI/ivDf7r9C+HfX/gPUQA4Cf+j3iLfIFdb5zaAIGiyvn7RyV5GCeDrlQBKgHJKAIqAsrUOp0jQL0uCP+E/t/DffZ9J+J8n/HcvACKF/yE33CvfIg+2zm8AC6EJ96oQk54SIPciwO8rARQBFAE5ah1mkaCflwT/fIN/GUf+Cf95h/9FCoCO4X/YzfJnrTMcwNxo0r3krkCTnxIg9xLA92kAigDKgFK0DsA1aD3G2MV466D1Gl9q8F8Z/st46k/4J/z3LAB2+801G/5CPsU6ywH0RhPvOYMJ6KkECL1YUgJQAniUIgAnaR2gPWs9NrioBH/CP+HfrgDwFf4dv/c/K/wPfEC+1DrPAfRCE++U4UlICUAJMFoCUARQBKB/rUM5gR5nS/AvI/gT/gn/hP8xvk8eaJ3rADqhyXf86GSkBKAE4DQAZQAi4uIS+ssK/2n3IIR/wr+38D+lAGj8klxrne0AZqIJuGbcpKQEmL8EKK8I4DQARQAiYh8J/mUF/9qf+hP+04R/V9/43zf8D/yaPM463wFMRZNw30mTM2wJsNjiRQlACUARQBmAiJ6Nu1ZZr8W1BH/Cf2nhf28BUEr4N/zSv67eJr/fOuMBTEWTcVusAoASoKQSYHkRYL1BqakEoAhARL8S/MsN/2n3GIR/3+E/RAFgGf4jvPc/zTtuW7PhBOuMBzARTchbY54CCFkC2BcBlACcBii9CKAMQMRZxl+HrNfauoM/4b9P+PcZ/An/huF/t7et2XCjXGed8wDGokn599MmbNgCgBKgxCLAeuNSYxFAGYCIaSX0lxj8V4b/soJ/7eHfy9H/XMP/vAXAbXv9sjzIOusBrEAT89pZE5cSwP9fCLAuATwXAdYbvzKKAMoAxPpMs7ZYr6G1hv/Sn/rPF/5zP/LvL/xPLgCKD/8D/8g66wGsQBPzqi6TlxIgj9MAlAAUAZQBiDi/6dYR6zWz1uBfQ/iv831/wr/Rl/5NC/8Df8467wEsQ5Pzsq6TmBIgjxKAIqDuIiBtGWAdVhAxjIR+gn/+wZ/wT/h3GP4bvy2Pt858AI+hCfpf+kxmSgBKgBJKAIoACgHEuk27NlivhbUHf8I/4T+38J/hl/7N8hrrzAfwGJqkb+w7qSkBKAFKKQKsN4yUAYiYTkJ/jeHfOvgT/gn/Pp/+R33vf5LnWec+gN1oor5qnslNCZBPCcBpAIoA2yKAMgDRzvTz3XqNI/j7Cf+p9jiEf8J/BuG/8Tb5eOvsB9AUAD857ySnBMirCOA0AEUAZQBiDRL6Cf489Q8R/An/JYb/ZO/9T/It1tkPoCkAzlhkstdSApRyGoASgCLAVxlAIYC4uDZz13rNIvgT/nN76k/4rz78N967ec2GVdb5DypHk/bpi056SoC8SgCKAIoAn0UAZQBid+3mqfUaRfCvO/gT/gn/1uF/kQJg8x7fYJ3/oHI0QY8MMfkpASgBSi0Bai0C7MsACgHEvdrOReu1iOBP+Cf8E/4LCf/ytFvlvtYZECpGk3TfZqJ6KQFCFwG5lgBlvxJAEZCj1gGEQgDr0nqu2a85BP88gn+q8D/vXqqM8B82+JcS/jP40r8xBcBpA8+xzoBQOZqs91ECxC4BOA0wqwiw3khRBORWBlAIYElazyVCf07Bf2X4Lzf4p3jqT/gn/CcO/41/ap3/oHI0YW8MVQBQAlAC1HIagCJgj9aBhUIA89R6rhD6h7W+l8wf/An/hP/04X9yAdA9/Bf+5/5mhf/GHfIQ6wwIFaNJ++XB5KUESFEE+C0BKAIoAhbROshQCKBfrecCoX+c1vcOgr9t+I+5LyT8W4d/d+/9j/Ml1hkQKkYT99PDk7jkEsDPaQC/3wtACUAREELrgEMpgHZaX+OE/lla3ysI/zGCv4/wH2KfSvivIvw3ftA6A0LFaPJePjqZKQFSlACcBqAIqEfr8EMpgHG0vnYJ/H20vjcQ/O2f+hP+Cf/GR/+HvdE6A0LFaEK+fdykpgSgBLAtASgCStU6GFEMYH+tr0VC/yJa3wtyDf6Ef8I/4T9a+B/4BOscCJWiSfnGPZOTEiCnEqCOVwLyLwEoAmZrHZooBjC3oE/g76b12k/4jxn8Cf+E/6y+9G+S51rnQKgUTcxXTCoAKAFSFgG+SwCKAIqAFFoHKoqBWrS+Vgj9MbVe6wn+vsJ/zH1eneE/3Z/7yzP89yoALrbOgVApmpwv2DtRKQFyPA1QTwlAEVCb1mGLgiBXrceawG+h9dqee/DPI/z7eOpP+I8f/i3+3F/kL/0b5+XWORAqRRP0lOUTlhIgxxKgriJg5abJehNHGZBG6zBGSeBB6zEi8HvSeg0PF/x56k/4J/znGv7nOPq/21sPP+2vrHMgVIom6TErJ26dJUDMIoASgNMAFAHhtQ5rPrQO5IT5RbSeQ7lqvWaHDf889c8l/IfaaxL+q/3Sv+Hw3/g16xwIlaKJesD4CUwJkGsRkPJmThFAEeBN60CHOEnruVGC1ms0wT91+PcR/An/hP8I4b9xq3UOhIrRhL2/nBKAVwIoAigCcLnWwQ/r0/qaL03rNZngX+9T/3Dhf/lelfBfffhv3G6dAaFiNGm/PnlC51ECxDgNQAmQYwlQVhFAGRBP64CI5Wh9LZeq9dpbWvAn/PsI/yH3vYR/6/C/cAGwZJ0BoWI0cf9m+sSmBKAIyK0IKOeLAikC0mkdJNG/1tdoDVqvteGDv334J/gT/gn//sI/BQCYosn7idkTvN4SgNMAuZYAZRYBlAHptQ6dSNAvXes1tdTgT/i3DP6E/9jhf94CYJHwH+joP68AgD2awB/sNtEpAXIvASgCyikBKALstQ6pSNDPXes1NF74t77n5RT8Cf+E/+L/3N8k77bOgFAxmsRv7T7hKQH8lACcBqAIoAzwrHW4RftrAFdqvVaWHvzzCv/99z2Ef8J/IeGfPwMItmgint9n0scuAUIWAbmVAClPA6TeIFAEUATgcq3DcQlajyF213ptJPjnHf5j7708h/+Q+2vCf7rw36EAuM46A0LFaDK++I5eBUDOJQBfDkgRUEcRQBlQntZhm0CPfbVeAwn+HoN/qeHf95/5I/y7C/+Nl1tnQKgYTcgNg4lZagmQ42mAkl8J8FUClF8EUAbgJAnrGFrrta6m4J9f+OfIP+E/x/Af/Oj/wIutMyBUjCblk4cnaL8SgO8F4DQARUBuWgcERCxL6zWN4O89+Jf61N9/+A8R/An/UcJ/4znWGRAqRhPzgNGJSgmQewmQ12kAigCKAETMT+s1jOCfQ/j39dSf8E/4N/zSv1GPss6AUDmaoFspAXyXADWcBqAIoAxARN9ar1UE/1yCv7+n/oR/wr+j8P916+wH0BQA14+buPWUAPkUAZwGsC8CKAMQsSat16S0od9X8M8z/Nfx1J/wHyr8jy8ACg7/jX9gnf0AmgLgLyZN4JJLgFxPA6QsASgC6i0CKAMQ69V67SH45xj86wn/ofeihP98w/+cBcBLrLMfQFMA/PG0ieytBOCVgHmLgMVu7BQB9RYBFAKIZWu9thD8cw/+HPm3DP6Ef9d/7m/UHbccftrjrbMfQFMA/O6sCU0J4LMIqOU0AEWAP60DCyIupvUaQvD3Efx56k/4J/wnC/+NH7XOfQC70aR9Q5eJTQngswTgNIC1dRcBlAGI+Wi9VhD8fYV/nvoT/gn/yd773+0th5/2QuvcB7AbTdyf7jrB6yoBSn8lgNMAsYuAGssACgFEP1qvBX5CP8G/9Kf+YcO//y/7I/xnGf5vkvta5z6A3Wjy/kifiV56CZDzaYDUJQBFQPcywHpTTBmAWIfWc95X8Le+F5QW/H2G/xye+hP+ff25P4Pw3/jL1pkP4DE0gZ/Wd8JTAvgtASgCPEkRQCGAGFfrOW0twb/e4E/4J/xnFP7vkausMx/AY2gSHzTPxA9VAnguAuorAfJ9LSC3IoAyYI/W4QkxN63nrAdzOuZvHfwXD//z7SUI/ymDP+E/9pf+LRj+G3/NOu8BrECT+R5KgC4lQC1FQL6nAXwXAePLAOvNtCetwxWiN63npCdzCv35B//6nvoT/gn/IQuAofC/mT/9By7RhP7HeReC/iUArwTkUQJQBFAE2GsdvhBTaz3nPErw9x/8cw//NRz5J/wn/XN/wwXAy6xzHsBYNKk/s8iCQAngvwSo9TRAjkUAZcBkrcMZYmit55RXczvm7yH4W4X/VHsYwj/hP8Pw/+fWGQ9gIpqsf7DowlBnCZBfEVDraQD/RcD4MsB6E56D1gEOsavWcyUHcwv9NQf/3J/6E/4J/5HD/za53jrjAUxEE/Y3bw+wQHgtATgNQBGQexFAGdBP66CHaD0HcjLHp/0E/7zDv+fgT/gvIvw3nmed7wCmokn7C4PJu+hCMV8RQAnAaQCKAMqAuFoHQixX62s7R3MN/V6Cf+nhP/SeqsbwH2I/T/ifO/x/yDrbAcxEE/fHhiexpxKAVwIoAigCKAJiah0eMR+tr9USJPgT/K2e+nsP/56+6Z/wv1D4/7I8wDrbAcxEk/ek0clMCVBXCTB/EbD4ZsR6M5VXEUAZkErrsIkE/VLMOfSXFfznC/8p9yGEf8J/mPDfvwAIFP5vlOuscx1AJzSBV42b1DYlQJ5fDljCKwGWpwEoAigDctE6nCJBPwdzD/0E/5yf+sc98p8u/Kd/35/wv1D4/6Z8qnWmA+iFJvK3aisBOA1AEVBOEUAZ4EXrUIsEfCtLCP0Ef576pwr+eYf/sMG/kPB/onWWA+iNJvQ/T5rolACcBqj1+wFKKgIoA/xoHYxL0HoMcY/T1hv7tTDP4B8m/M9//84z+Of11J/wn3f4HykAviGPt85xAHOhSX3NtAlPCVBnCcBpgJyLAMqAkrQO3IR5HFhS6C8v+Nf41J/wT/g3+9K//yGPtM5wAHOjif2hWRPfWwmQQxEQ85UAigDKAMoAREwhoZ/g7z34533kn/CfYfj/oDzQOr8BLIQm9292WQBCLDbzFQH5lgCcBqAI8CdlACJOt7TQT/AvN/zn/dTf5pv+Cf9zh/8H5HnWuQ0gCJrIr+jzzZ/eTgPkVwJQBFAEeJEyABH3WGLoJ/iXG/zzD//en/oT/kfC/9VyvXVmAwiGJvOz90xqSgBKgBRFQJgNkPUmrqwigEIAsTZLDfxlB/88jvsT/gn/BYX/zfJnrLMaQHA0odfvndz5lgB5FgHhb5J5lADlFgGUAYjoVUJ/fcE//6f+dR/5J/ybhf+tCv5vkgdb5zSAKGhS7ye/SwnAaQCKAIqAvmUAhQCiX2fNXfu1heBfSvDnqT/hv5Dwf6P8ZQX/Q6zzGUB0NLlvWT7ZKQFilwClnQagCKilDJheCFgHHsTarSHwew39BP98nvoT/gn/Q35b/t/yx+S+1pkMIBma4F9cOenTlgChi4DQCzunASgCKAL6lQEUAojxreUpP8HfX/DnqX+X8G8T/An/U90p/0G+R557K8f8oVY0yf9k/OTPuwTgNECupwHKLwLKLAMoBBBjW1vg9xz6vQR/nvrz1J/wvyz8P9S8wy9vkn8v/1y+V14ony/XWucuABdosr5t0gJACWBVAlAEhNtY2W8S6ysCTAuBu+Rn5NvlK+WPyBPlWrlfs+bpf+4r18j18gfa/8wr2//OZ9p/hnnYq0zGbYwZBP675Gfk2+Ur5Y/IE+VauXvc9D/3lWvkevkD7X/mle1/5zPtP6PS4G/21P8u+Rn5dvlK+SPyRLlW7me5JwUAgAR8c82G1zSN3eQSgO8FKOWVgJpfC/BeBJRfBswuBOYsBXbIj8tflE8NtS42/6z2n/nx9t9hHgYLk3Ebscv8sJ/DZ+6QH5e/KIONW/PPav6ZWgc/LndYr8UFBv8d8uPyF2WwcQMAgExR+H/+8NGdvEuA/L8XgCKAIqCOMmChQmCn/IL8ebkq9hrZ/Dvaf9cX2n+3eVjMVMZtyEwCf+NO+QX58zL6uGn9WyV/Xn5B7rReizMO/jvlF+TPy+jjBgAAGaHQf8Lo+zteSgBOA5RXAlAEUAQsUAg8LN+v8PQUq/Wy+Xffv+dneNg6QGbkw/dXPm7dwr6bwD9Q8+3M90uzcdMa+BT5fvkwwb+zD8v3S7NxAwAA5yjwHyh3UgL4KwFKLgIWKwHCFgGUAZ7dHYx2yHfLY6zXywEKdUfLd8vtDgK2V3e0n1FV45Zp2B+2Oeav+Xamm3HT+ne0fLfcnlfoXyz49wz/zTH/d0s34wYAAI5R4L9t3Dd55l8C8EoARUB5RUBFZcAueaVcb71GTuL+PV9Id6Xc5SBwe3FX+5kUPW7dg777wJ/NfNPat15eKXcR/B9zl7xSuh03AABwiML+Fyf9OY+SSwCKAB8lAEUAZcAYb5EvsF4bu6JA+AJ5i4PwbW3zGRQ3boUF/eznm9a9F8hbKg/+jbfIbMYNAAAcoaD/4Wl/0zNECWBfBJRRAqwsAso5DUARQBnQ+jF5mPW62BcFxUPlxxyEcCub3z3bcev/RD/bsF/EfNN6d6j8WKXBv/FjMrtxAwAAJyjk/+a0AsDbnwnMpQTI/TRA3kUAZYCDYNHX78iLrNfDRVGYvOj+ur4k8DvN72z9uS+KAv1F7RdNlhr0i5xvWusuurfjlwR6C/1zBv/vyOzHDQAAjFHA/9lZBUA5JUDJpwEoAigCsi0D7pUbrdfCUCgQb5T3OgjnsW1+x2LGrbkG22vRej4w33qgNW6jvLfw4N94ryxm3AAAwBCF+zO6FAB8L4DPIqD0EoAioPgy4FZ5kvU6GJr71p11krx19Gmyg9AeyltlgeN25kntNWk9L5hvPdDadpK8tdDg33irLG7cAADACIX6I/sUAN5KAF4JoAiwLAIoAxZyk3yS9RoYCwX+Y+VX53u/3HVxsEkWPG5natzO/KqD+cF864HWtGO1Hn+1sODfuEkWO24AAGCEQv02ixKA0wB5lgAUAWUVAUaFwI3yidZrX2ya37H9Xdvfe/EywFj9LmdVOG7ZW8V80zr8RHljIcG/8UZZ/LgBAIABCvT/q28BwPcC+CwBKALsiwDKgJlulSdYr3upaH7X9nee8/MyD/0D9TucxbjlZ1XzTevvCXJr5sG/causZtwAACAxCvOfmKcA8FgCUATU9VpAuCKAUwGJCoGH7yvoC8i6ct+eL5h72EEYZNx6wLjlidbejfLhlKE/YPBvfHgrX/gHAAAxuW3NhnfNWwBQAlACUATUVQYsWAi82nq9s0K/+y84CISMW09yHbd2rlY7blpzfyHD4D+w2nEDAIBE3LZmw4W3LVAA8L0AFAEUAZQBHfyI9VpnjT6DP7EOhnPIuGUybiNzs/px01r7J5kF/8bqxw0AABKg8P/CpgAosQTIrQiIWQLUVgTkUAZYh/eEhcDNcrX1WmdN8xm0n4V5WOwo4/Z9fsdtyjy8WVY/blpjV8ubMwj9A2+W1Y8bAAAkQMH/+EEBELcE4JUAD0VAyhKAIqDuMmCoEPhx63XOC/osXiB3WYfHjjJuLR7Grce8Y9xatG6/QO5yHvwHMm4AAJAGhf795COUAP1LAE4D1FoEUAb08ArrNc4bCnNXOAj3s2TcRkg5bsy3cGi9vsJ58G9k3AAAIC0K/ZuGCwBPJUAORUCupwEoAnwXAQWUAQ/KJ1uvb95QwDtWPugg5E+y+dkYtxFijhvzLR5ap4+VDzoN/o0PSsYNAADSosD/6dECIEQJUNP3ApR1GqCOIoAyILrvtF7bvKLQ904HQX+SjNsEQowb8y09Wp/f6Sz0D8u4AQBAepo/BTiuAPB2GsBHCcBpAIoA2yIgkzJgu1xrvbZ5RUHwCLndQdgftfmZGLcJ9B035psPtC4fIbc7C/6N2yXjBgAA6VHIf920AsBTCeCnCLA5DUARkFMRUHUZ8PvW65p3FBAvcRD4R2XcZjBp3JhvvtF6fImj4D+QcQMAABsU8H90VgFACVDXaYDai4DwZUD8IsBRGbBTHme9rnlHofE4udNB6B/Y/CyM2wyaa7u9xq3nGfOtB1qHj9O6vtP6vjJQ97udknEDAAAbFO6f1KUAKPl7ATgNQBGQpgioogz4vPWalgsK3Fc7CP4DGbeO6Bq/2kHwZ771ROv51db3k6H7HOMGAAC2KNxv91ECcBogjxKAIoAyYKI/Y72e5YJC9087CP4DGbeO6Br/aQfBn/nWE63jP+0g+A9k3AAAwBYF+3/sWgDwSgCnAWouAnIvAyIXAjvkIdbrWS4odD9e7nAQ/pufgXHriK7xx7fXunX4Z771QGv34+UO4+DfuEMybgAAYIsC/ZV9C4CSS4BcTwOU/FqAxyKAMmCFn7Jey3JDwftTDgoAxq0nzbXuoABg3HqiNftThsF/IOMGAAD2KMz/53kKAF4JqPc0AEVAWUVAoELgtdZrWW4ofL/OQQHAuPVE1/rrHBQAjFtPtFa/zjD4D2TcAADAHgX5l89bAJReAnAagCKAMqCzJ1ivZbnRfGaLhMBABQDj1pNFxy2QjFtPtD6fYBj8BzJuAABgj0L8hkUKAF4J8HsagCKgtCLApgzoUAhssV7HcqX57AxDJOM2J4xbnmht3mIU/BsZNwAA8MHmNRtWhygAOA3g9zRADUVAnWWAm0Lgc9brWK7os/usYZBk3OaEccsTrcefTRz6h2XcAADAD5vXbNhMCcBpAIoAyoA5fYf1GpYrCnPvMAySjNucWI4b821+tAa/wyD4D2TcAADAD5vXnPbnmwMVAB5LAIoAigCPFlQGvMZ6DcsVBbpXGxYAjNucpBo35ltYtO6+2iD4D2TcAADAD5vXnPb7cilkCeCxCKAEoAjwatwyIHoh8DzrNSxXFPKeY1gAMG5zEmvcmG9x0Vr7HIPgP5BxAwAAPyj8n7+nANhTApR8GiB0CUARkE8RQBkQrQx4pvUalisKfScZFgCM25yEGDfmW3q0vp6UOPQPy7gBAIAfFPyfvbcAKL8E8HcagCKAIsCqDAhSCBxnvYblioLgOsMCgHGbk77jFrhwY9zmRGvqOoPgP5BxAwAAPyj0H7m8APBZAnAaoKwSwEMRQBkQpAxYa72G5YrC4SrDAoBxm5Np4xb5dRvm2wJoHV2VOPQPuZFxAwAAXyj037OyBCj/ewE4DeC1COBUgH0Z0LkQOMB6/coZwwKAcVuARGGf+RaYxKF/WMYNAAB8obD//4wvAHyeBpheApRwGoAigCLAWxkwvhCwXrtyx6oAsP69c8co/DNuC2IQ/Hdr/XsDAACsQEH/DycXAD5LAE4DpCkBai0CKAM6FwKrrdevXDF+BYBxmxNd86usCgDm2/xoTV+VOvgP3dcYNwAA8IVC/humFwDhS4BaXgmgCKAIKLwQOMp6/coVhfCjDAsAxm1OFMKPSh/8mW+LorX8KIPgP5BxAwAAXyjgv3B2AeD3ewEoAigCKAPMyoDvt16/cuVe2z8DyLjNiQL5SYnCPvMtIFq/TzII/gMZNwAA8IWC/ZO6FwC8EuC5BKivCKAMMC4Enme9fuWKQvjzDAsAxm1OFNKflzDwM98CoTX7eTFC/4zgP5BxAwAAfyjY32dZAnj9gsBci4BUJYBFEcCpAFdlwGus165cUfh7zSAEGhQAjNucDI9b5LDPfAuI1urXJHraP07GDQAA/KFQ/8V+BQCvBNiXABQBnoqAEsqAOQqB37Veu3JFgfAdhl8mx7jNSbdxi/bKDeM2J1qf32EQ/AcybgAA4A+F+cvmKwByLQFKKgL8vBZAEVBWGdChEPi89dqVKwqKnzcsABi3OVk5btHCPvMtIFqTP28Q/AcybgAA4I9bDz/twvkLAL+vBORyGoAigDIgF0cCyVbrtStXFB63GhYAjNucNNd84tDPfAuA1uGtiUP/sIwbAAD449bDTztbLlAA5FwC+CkCUrwWQBFAGRC4EHi69fqVGwrgTzcM/wMZt54017ph+B/MOcatJ1p3n24U/Idl3AAAwBcK/4c1BcDARYuAkCUApwHy/34Af0UAZUBAL7Bev3JD4ftCBwUA49YThfALE4d95lsAtNZeOCv0Rwz+Axk3AADwh4L/5pAlAKcBKAI8FQGeTwVkXgZ8ynrtyg2F7085KAAYt54omH8qcdhnvgVA6+unEj/tHyfjBgAA/lDov3q4ACi5BIhxGsB3EeDrtQCfRQBlwJw+KFdZr1+5oOB9iHzQQQHQ/AyMW0cU1DVuZzxoEPiZbwugdf0Q+aBh8B/4oGTcAADAFwr87xotABYvAXJ+JaC+0wC1FQHeTwVkVAa8zHr9ygWF7pc5CP8DGbeOKLi/LHHQZ74FQOv5ywxD/zJ1v2PcAADAFwr7PzeuAOA0QGmnASgCcjwV4LwMuMZ6/coFBcNrwh8jn7sAYNw6MmvcmG8+0Rp+jfW9Zeg+x7gBAIAvFPJPnlYAeCwBaj0NUHIRQBmQZRmwU55gvYZ5R0HxaXJnqi+S62DzszBuMxgeNwdzjfnWEa3lT5M7jUP/sDsl4wYAAH5QwD9QfjduCeD7lYDcTgNQBFAEOCoELrVew7yjAHmpg9A/KuM2g+badjC/mG890Rp+qZPgPyzjBgAAvlDA/6dZBUCoEsBzERDjNID/1wIoAigDFnK7XGu9hnlFQXut3O4g8I/a/EyM2wSaa7q9tq0DP/OtB1qz18rtjoL/wO2ScQMAAD8o3H+kSwFQwysB9Z4GoAgoqQxIXAi803oN84pC9jsdhP1JMm4TaK5pB2Gf+dYTrdPvdBb8h2XcAADADwr1v9qnAPB6GiBdCUARYFEEeCkDJhcBfsuAyIVA8yfK1luvY95QwF5/T6A/IRfJ5mdj3EZoruX2mrYO+sy3HmhtXr9lz5/d8xT6h31QMm4AAB74xmGnPkH+hHyj/AN5nbxB3iu/J5fGedNj/tBc3nz4Yt4y09Om2jfw51ICUARQBFAGmJQBH7Ney72hcH2Vg5A/S8ZtBF3LVzkI+cy3nmg9vspp8B+WcQMAsODGw049RP57ean8l0kBv4u5lwCeigBKAIqAnIuAnMuAgIXAOdbruxeazyLEZ5qoBGDcWkKNWyIZtxatv+c4D/7DMm4AAClQ0N9XPl/+sdwul4YtuwTgNABFQPlFAGWAi0Jgszzcer23Rp/BYe1nYR0QGbceMG55ovX2MLnZeegfdrOsftwAAKKhcH+ofLO8dTT0l1QC8EpAuhIg1yIgXAlAEVBbGdCzFLjCet23pvkMHITDvjJuGYzbmHlZ/bhpjb0ik+A/bPXjBgAQHAX6VfI/yvtmBf9QJcA3FiwBeCUgryIgVglAEUAZkItTgsprre8BVjS/u3VIXEDGzZkd52K146Y19bUZhf5l6v5Y7bgBAATlXw89dR/5s/KbfYI/JQAlgLciIJ/XAroVAdZlgHXwL70MGCkFHpFnWN8PUqPfeWP7u5sHxzll3PyH/XE+IqsbN62hG+UjOQX/kXviI7K6cQMACMq/HnrK8Qr+fyGXBlIC2L4S4PULAtOfBqAI8FIEUAYkc6s8wfq+kAoFuBPkVusQGcDmd2DcfAf+6ueb1swT5NYcQv+Y4D/sVlnNuAEABEXh/6Vym1waLgDsS4C8vxeg9NMAvBZAEeDVQgqBm+Sx1veH2CjMHStvchDeQ9n8Loybz6Bf/XzT+nisvCnz4D/sTbpvFj9uAADBUOA/UL5vT/AflhKgvBIgj9MAFAH5FQGUAVG9oeRQ0obIGxyE9tDeUHIJEGvcmG9xacP/DZ5Df4/gP3y/vIESAACgAwr6T5D/78rwP74EsCoCFi0BeCWg1NMAfl8LqLkIyLsMcFsI3CyfZn3PCI0C39PkzQ7Ceiyb341x8xn0q5tvWv+eJm/2GvrnDP7D3iyLGzcAgGAo4P8befPk8O+rBCjlewE4DVDXaYDwRUBeZYB12C+sELhHnmV97wiFQuBZ8h4HIT22ze9Y7bg5mDfMt+/bHf7Pkvd4DP4Lhv5R75HFjBsAQDC+fugpP6Bwf+fs8E8J4PmVgJpOA1AE5F0EUAYE8zvyAut7yKIoGF4gv+MgnKey+V2LHTcH84L5NgWtbRfI73gK/RGC/7DfkdmPGwBAMBT+T5B3yqWvdy4APJYA3r8XgFcCYpQAFAH5FwG5lAHOC4GPysOt7yd9UVg8TH7UQSC3svndsxy39pqzDuQG7p7r+t03ZjduWsMOkx/1FPojB/9RPyqzGzcAgKAo8K+TNw7CPyWAfQng7ZWAXIqAmN8PUHoRQBlQTCGwWb7I+r7SFYXIc+VmByHc2uYzyGbcdI2d215rDsJ4ssA/Tn0GG7MZN61X52p93Wy9vvcP/UGC/7Cbdb/NZtwAAIKioH+Q/PJo+K+5BOCVgLTfDUARkKoIyPNUQG5lQLdCIEkp8DG53voeM4nmZ5NXTfr5HQRyKz8msx23Mpw9f8fM+49Jt+OmdXS9vMp6Le8f/MPfB0futR+TbscNACAKCvkfnhT++5cAfv5MYD7fC5DTKwF5nQaI/VpAqiKAUwF5lwHGhcBD8vfkOut7zYDmZ5HvaX+2oL+vg/Aeyofk78kqxs3euQL/OB+SvyfdjJvWzHXyPfIh67Xb8Gn/rHvsQ/L3pJtxAwCIhsL9K2eFf8sSoJ4vB+Q0AEUARUAtZUD3QiBoKbBDXipPsLrfNP/u9mfYYR/4spFxC263ubfgHN8hL5Vm46b18QR5qdxhvU47Dv6j7pCXSrNxAwCIikL98fLbXQuARUsAXgkoswTIqQiI/VpA3kUAZUAlhcBOea18hVwd+z7T/Dvky9t/50778JetjNvcJgn8k9wpr5WvkNHHTevgavlyea3cWWvonyP4j7pTXitfIaOPGwBAMhTor+kT/ikB/JwGCFUC5FAE2JwGoAjIuQjIvQxIWAo0R7k/J98kT5X7Lnpfaf4Z7T+r+WdefXeRx8XNZdwmahr2Z9m8HvA5+SZ5qlx43LTW7StPlW+SV99V+TH/AMF/0usBn5NvkqfKhccNAMCEGw495bx5wj8lgJ8SgNMAFAHxiwDKAG92LwXmKgbul9fJy+Tr5QvlD8pj5Bq5vzxEHiWf2gbGF7b/2cva/+799iGwOisct+7zwHrOTvF+eZ28TL5evlD+oDxGrpH7y0PkUfKpW/YE/RfK18vL5HXyfuv11UPojxT8x6r7+v3yOnmZfL18ofxBeYxcI/e33uMDAKxA4f/x8rYbFigAci4BvHwvgLdXAigCyigCPJwK8FwGlFYIJCgFEI0tIuzPpfVaGS70Z/O0f1rwX9DZ+4wY+5tYe7HNj7n4vjHkPrbLvnnW3nton75L3i83y03yOvkH8o3yJ+QTrDMdQGcU/N/ShP+BOZYAnAbgNABFAKcCKANClQIUA+jBftes9RyrMfT3D/55P+3POfjXFv5DBP9b5tvH/4u8VP57eYh1xgMYiwL/KnnvcAGwaAmQ/18IoASgCOhbBCy+KaitCKAM8GH/YoByAEPa//qznjOEfh9P+/ML/rPDP8E/+/A/6nblgj+Wz5d8VwT4QWH/DaPhnxLARwngsQgIUwLkWQSkOA2QugigDKAQCFsMUA7gOOe7lqznAIGf0J8q+Nd83L+Qp/5jHckGt8o3y0Otsx9UjkL+vvLWSQWAdQmQ+/cCcBrAtgSgCMizCMilDKitEFi8GKAgKNfFrgnra5rAHzL0E/xLDv489Z8r+I96n/yPcpV1DoRKueHQk8+ZFv4pAfycBghVAnAagCLArggo81RA7YVA+IKAosCPYcbS+pr0oPXaFD/4x7tvEPx9hP+8n/q7Cf/D3il/Vu5jnQehMm449OQ/lTMLgJxLAF4JyOU0AEWAtyKAMoAyIIbhCgIKA6tAT8DvpvUaROgn+Jcc/L099e8Z/oe9Vj7VOhNCJSj4HyJ37CkAupUANf+FgBJfCajxNABFgMcioPwygFKgn/FKghzLBNvPwvpayEXrtaWE0E/w9xX8PYf/jJ/6T/DUbcoZL7XOhlABX1t98kv2hn+vJYCvLwf0VAJwGoAioMwyoNzvC6AQiKd1SM5J67EqReu1I23o52k/wd9H8C/oqf9w+B/OGR+QB1lnRCiYr60++b/JJYsSgO8FKLsEoAjoUwRwKiBkGUAhgItoHc4J7n61XhvSB36e9i8S+gn+4YN/eeH/1NHwP/D/k0db50QoFIX/G5sCYHwJwPcCpCgBSi8CcioBKAIoA7xqHXwQa9N6zhP6Cf6pg7/38F9W8J8a/gfeLP+NdVaEwlDof8Ig/FMC8EoARUD8EoAigDKAQgDRp9ZzuvTQT/D3Hf7zD/75hf+OGeNO+QPWmREKQoH/3NECgBKgplcCcj4NQBFAGUAZQCmAOJ/Wc5XQn0voJ/jnEf677Xl9hP+ZT/0nlQAnWOdGKASF/YvHFQD5lABevxyQ0wAlnQagCKijDCixEKAYwNq1nnv+An/ZoZ/gz1N/v8F/7vA/8Ca5zjo7QgEo6F8+qQDIuQTwUwTkVALwJYEUASWWAZwOoBTAWrSeU35Df5q11vqeQvAvI/iX+dS/15H/aX5ZOeVg6/wImaOQf920AmByEVD2XwjgNECupwEoAnIsA9IVAZwOoBjAErSeI/4Df7rQbx38Q94HCf724Z+n/tNtM8qHrfMjZI7C/fVdCgBKgNpKgNxPA5RRBHAqgDLAk9ahD+vT+pon9PsM/aU97Sf4l/3UP3D4H/gq6wwJGaNgf2vXAoASoJxXAjyfBojxWgBFAGUAhQDlAPrU+tq1NpfAX1borzH4537cP+1Tf0dH/seF/8Zvy+OtcyRkikL9vX0KAL4XgNMA+ZUA6U4DUARQBlAIjNc6ZCIB34uLri21hX6CP0/9eeo/0b+Q+1hnScgQBfpH+hYAlABllQD1nAagCKAMoBDwrHVQRcJ9DHML/GWGfoJ/nsE/z6f+icL/wPOssyRkiML8d+YpACxLgLJeCfBTBIQ8DUAR4LEIKO9UQK5lAIXA4loH35K1HtvcDbE+1Br6wwf/7vdHgn9+wb/Up/49wn/j7fLx1nkSMuOrc7wCEKIEyP17ATgNYFMCUARwKsBfGRCmEKAUSKN1uCbIl2WouW+1dlmv2XFCv5+n/QR/nvpHDP7DXmydJyEzvrr65Fu+ukABkFcJ4P00wOKLEKcBKAI4FWBdBlAIIJZq7oHfU+gPH/z9PO3POfjz1L9/+Dd46j/qfcozq6wzJWSEwv/1TQGwaAmQz18I8F4C5HgaoJQiIN1fDCi9CPBWBpRSCFAKIKYz5Lwl8McK/bUHf99P/b0Gf29P/RcM/4Ms86vWmRIyQsH/ukEBEKcEqON7Aby9EsBrARQB85cB4Tdo1htOX2VA2EKAUgBxcUPPSes1xnqNjR/6fR3zJ/jXdNzfxZH/0fDfeKvc1zpXQiYo9F8+XADUVQJwGqCk0wAUAZwKyLMMoBRATGlpYd9r6I8T/H097Sf489Tf+Kn/iKeca50rIRMU+H99tACwLAF4JaCU0wClvBaQvghIVQb0KwLqKANKLgQoBrA2Y80h+7XBb+C3Dv0lHfOPGfw57l9k+G+8yjpXQiYo7J8zrgCgBPDxSgCnAfwVAalu/LWcCvBcBvgqBCgFECdZetj3HPjjhP56n/avDP++g3/54d9P8J8R/hsflIdYZ0vIAAX9dZMKgBBFgGUJEOJ7ATgNQBFAEUAZ4LMMiFsKUA6gN2Nf6/ZzmdDv8Wk/wb/W4O8r/E/KKGPyzEussyVkgkL+12OWAJZ/IaDM0wC+SoDuRUD3mwVFgG0R4OFUgPcywG8hEL8UoBjAmKa4fu3naH6BP17o9/m0n+AfIvin/5K/kOHfQ/Dv8NR/1A9Z50rIBAX8y2YVAHw5oLcSwF8RkMNpAIqAPE8F5FAG+C4E0pQCFATYxdTXov3cyzfwewn9BH8fwZ+n/u7Df+ON1rkSMkHh/iVdCgC+F8DPKwH5nwYotwigDKAMyKMQsCkGKAjK1+qasp9LZQR+Qn/K0E/w9xb+Q+7TEx35H+cTrLMlZICC/ePl9hpKAE4DeDsNQBGQYxHQvwyIt6G03iiXVwjYFgMUBX61vhZyCvq5Bf64ob9f8E99LyP4pw3+PPWP8tR/1BdZZ0vIBAX7K7sWAJQAlABWpwHy+X4AigDKgNIKAX/lAKVBCWE+/5Cfa+D3FPoJ/v7Cf+j3/D0/9S8s/Dc552LrXAmZoFD/gj4FQIgSoITvBaAI4LUAr0VA7WVAjoVA3qVAPgVBjoWC9WdAwM8/7McP/IT+EoJ/jKf+BP/oR/6Hw3/jh61zJWSCAv2+8sa+JQCnAcotAXgtgCKAMsCH1kGHggBrDPi5h31CP8HfOvj7PO6fS/ifK/gP/CvrXAkZoTD/y/MUAJQAvkqA2k4D5FcE7C0DKAIoAygEKAny0/raIPDnFvrrC/5x9g72wZ+n/hmE/8avWWdKyAgF+VXyHkoAXgmgCCi7CKAMKKsQqKsYoDQgzBP00wd+Qj/BP+fgn8tT/yDhv3GrdaaEzFCQ/7V5C4B43wvQrQTwVAR4KQG8nwagCKizCPBcBpRUCFAKzKt1KCe4x9B6LuYX+PMJ/QT/8oI/T/3nCv4Dt1vnSciMTatPPlhBfrO/EoDTAGGKgNxPA9RRBFAGUAZQDCDOp/XcIvTXEvoJ/jk89fcf/IOH/91a50nIkE2rT37ppgUKgHpfCajrNABFAEVAbWVAyYUAxQDmpvVcKSfw5xX6Cf4E/1qf+ncN/xQAMDebVp98NSWAn1cCPBcBtb8WELcI2FsGWGx48iwDKAQoB7AUra95An/doX958I93n4+5P+G4v23wT3Tkn1cAIAybVj9r/abVJ3/LQwngoQgo95UAv6cBKAJ8FQGUARQClAMYQ+trt47AT+gn+Jcc/H2G/5RH/ke82zpHQsZsWv2s85oCgBLA1ysBtRUBub0WEL8MsH09wLoIWKwMoBCgIEACvo25BH7r0O/jaX9+x/xjBn+O+/s/8j8ifwYQFmPT6md9UFICLFQEcBqAIqDMIoAygEIgpNZBFQn3IbVYXwj9BH/vwb/08G/41H/Y66zzI2SOwv8B8kuhSgDr7wXgNABFQHlFwN4ywHID5aEMyLEQoBSYT+vgW7LWY5ujVmtHzoHfOvQvD/7x7s8E/zqCv4On/sNebp0foQAU/tfJGwclAKcByj4NQBGQfxFAGZBvGUApkEbrkE2Az1frtYHQ7/9pP8E/n+Bf0FP/x1TOeYt1doRCUPg/Xn6TEoDTAJ5KAIoATgWkKQMoBBBr1XruL7p2Wa+9/oJ/3HsvwT+f8F/YU/9B+G88xzo3QkEo/J9YWgnAKwFlFAH9SoA6iwDKgLIKAUoBxPBaz+mSAr+v0E/w56l/2eF/JNuss86MUBgK/0/Z+zqAn+8FyPc0QJxXAryfBqAIsCgD/BQB3sqAUgoBigHEblrP0VIDv4fQnzL4p9gX1Bv8c3vqb3rkf9ivW2dFKBQF/yPl31ICcBqAIiDHIoAyIH4Z4KsQoBTAmrWee6EDP6HfNvQT/PMM/iU+9R8T/hsvs86JUDAK/o+Tl8hdvBJQ12mAXF4LoAjIrwjwWAaUWghQDGBpWs+lmgK/l9BP8LcP/nUf93fz1H/Yl1hnRKgAhf8Xy3vzKAHqPg0Q77UAioD8iwDKgPSFgN9SgHIAvWo9J1KEfa+Bv9bQT/BPF/w9HvfP5Kn/wB3y8dbZECpB4f/J8rMhSwAPpwG8lQC1nAagCKAMyKkMCFsI+C8FKAcwttbXduqw7znwE/oJ/nGDfx1P/ROF/8aPWmdCqJBNq5/1U3JzSd8LQBFAEdC3COBUAIVAjYUABQF21fraJPDnEfhLDv599xQEf4J/x8zy49ZZECpF4f9g+UZ5N68E1FcCUASUWgRQBvgpBPItBSgJytf6mvIY9r0HfkI/wb+E4E/4P+UbyjX7WudAqJxNq591iPylTatP/ldOA9RXBORUApRSBFAG5FEGxCsFyikGKAr8aX0N5BL0cwj7hP60oT/n4J9L+K88+Lee/MvW2Q/gMTatftY+m1af/Hz5fynEf5sSwG8JQBFAEVBSEUAhUHYpQGlAmPcQ9gn8BH+Cf97Bv5Dwf69cZZ35AMayafXJByvE/4R8t/yK3GlVAnj5gkCPRUDtrwWUVARQBuRZBsQvBeotBkJKOPdqvHljvSYQ+v2HfoI/x/3Thf/H8stbrDMeQGcU5tfI58kL5H+Vn5fXyzvk/fJRTgOUcxqAIqCWUwF5lAE5FgLxSwHKAczF+PPAeq6XFPitQr/vp/0Ef576L/TUf+BmebB1pgMAqB7d2PeRa+ST5UnyufJ8eYm8Wm6p41SAqzJgi7xaXiLPl8+VT5drB+OmDfQ+co18sjxJPleeLy+RV8st1ht822LApBzYIq+Wl8jz5XPl0+Vj46b/fR+5Rj5ZntT+Z85v/ztXt/8MB6G1KgOM24YtBP18A38NoZ/gz3F/w6f+A1+aeo8LAABzok3AM+RF8pNye/lFQPIyYLv8pLxIPiPUuGnT/Qx5kfyk3G4dAHyUAkHLge3yk/IiGWzcmn9W+8/8ZPvvsA7IpRlp3DZo3Dbon7lB/+wN22sO+4R+gn+NwT9++J/vFK7xU/+Bnw+11gIAQGK0MVglXyWvlTtTFwEFlQE75bXyVTL6F+JoM75KvkpeK3daBwO/xcDMkmCnvFa+SkYft+bf0f67rm3/3dbhOVcTj9sGjdsG/bs26N+5YWfpQT+XwF9T6J8v+Pd/nZDgz1P/DuH/ga+tPnl97HUXAAASoA3DcfLdcpvFqYBMy4Bt8t3yOKtx00b9OPluuc06MGRUDGyT75Zm46bgepzUz3D6NgeBOhM9jNuG45qfIef5lnPgJ/QT/GsL/o6e+u9W4f88q/UXAAAioQ3EEfLtVq8H2BQBvcuA5pj/2+UR1uM1QBv4I+Tbv+n89QDjcqA5yv126WbcFGyPkPqZanw9IN9xK2G+WQf5XEJ/qcE/ZujPLfh7Df+OnvoPvsT8Q9ZrLwAAREQbimPklXKX1ZcGeigDRgqBXfJK+UTr8ZmEfrZjtLm/Uu6yDhiOCoJd8krpdtz0sx2z52c8fdf/3969R2t2l3WCtyjIpXIxJBRkQQxGChpCgnSQCVFcjVw0yLQicoku2jZio2mcGcOE1bQzti5tCULDTAMSxvZCa6dXd7wE10BoHWAGkYuQBOmWFggVEkCBEqKSm4Gkap6dOm9y6tS5vJe99/P7/fbns9b3r7ZDnXe/+7ef73Pec05+MV+5sPeV4q/b5yq537IL/GqFX+lX/H3XP/m7/l0+FDkm+8wFYAQxYDwjsj/zrwcUsgzYH3lG9vWYVwzRz4js/2wB5SN5QbA/Us116/6ta//mjMJdUqq6bvFee0Zkf/b7vdbCP9XSP9Z3++ss/j7un13+1/358hsie7PPWQBGFAPHyZGrNhtEMgamhGXAVZGTs6/DomKoPjly1WbDdnZBGSlXRaq7blF8T45cVUAJz0r3tVd33br32tp7TtmfM9mFP6v0K/71F/+WP+6/ofx/LvIt2ecrAEliALkkcudWw0mDy4A7I/88+3VfVQzal0Tu3G4QL6Cs95k7I9VftyjBl0TuLKCQD5rWrlt8DZesfS0K/4aU8F3++kq/4l9a+Z/Id/1n5f8x2WcqAMliGHly5AvbDSuNLAK+EPkfsl/vvsTw/eTIFxYZ1gso8svkC5FmrlsU5CdHvpBd0nco7q7bBvG1PHnta5ps2S+t9NdX/Msr/Yp/K9/1n6v8dx/7/+bssxSAQsRwclbkU/MMMZUuAz4VOSv7de5bDONnRT616kBfQMnfKp+KNHfduq9p7WvLfn1dtwUsct2yS3qrhb++0q/4l1j8W/+u/ybl//2Rh2SfoQAUJgaV0yMfX2SwqWQZ8OeR07Nf36HEgH565ON9D/0FlMg/jzR73bqvLfLxAl5n120Bm1237ILeeuGvs/Qr/op/Ed/1Pxh5feRB2WcnAIVaWwLcsOigkzWczbEMuKHl8j+ztgS4YYxyMFKJvKHlEjmzViZvKKC0u24LGPN+U/prK/6Ll37FP7v411j+5/qu/5cjz80+LwGoQAwj3xKDzIFlB6DMoW3DQuBAZF/26zmWGOK/JXIgq0D0WCIPRCZz3eJr/Za1rzm7vLtuC8i+3xT+kkp/ud/tr7X4+67/St/17/L2yDdln5MAVCSGk/MjdxweVJYfihKHuTsi52e/jmOLwf78yB3Z5WKF5cAdkcldt+5rXvvas0v8spnkdSv1fqul8E+x9Cv+iv/A3/W/OfLC7LMRgErFoHLx0cNLNYuAi7Nfvywx7F+cXTZWWA5M9rp1X3sBRX7ZTPa6lXq/lVr4Syj9qxX/KZZ+xb+C7/p/KfLyyPHZZyIAlYvB5Tc3H2iK/lTAW7Nft2xRAH4zu4Askbdmv27Zokj/ZgFlftG8Nft1y5Z9vx1d9ssq/PWXfsW//uLf7Hf9PxV5WeSE7HMQgEbEAHNS5Mbth5yilgE3Rk7Kft2yRSk4KXJjAaV+3nT/1slftyjTJ0VuLKDUz5vu3zr56zb2/abwK/2Kf+sf99/2u/5fjfx25BmRXdnnHwANioHmWZGDOw8+qw1bPQx7ByPPyn69ShFF4VmRgwWU+53S/RtdtzVRqJ8VOVhAud8p3b/RdVsz1P1WQ9lvq/Qr/lMv/gV+3P+eyLVR9l8beXbkuOzzDoAJiOHmysWGoZRlwJXZr1NpokBcWUDB3ymu2wZRrK8soOAfFddte33cbwq/0j9+6Vf8k7/rf1vkS1Hy93dFP/LOyP8R+cnI06PwPzj7bANggmLQeUTk9uWGpFGWAbdHzsh+nUoT5eERUSpuL6Dkb5Xu3+a6bfBZ161Ki163msq+0q/4l1/8B/+4/z/KPmMAYFQxiFx+4woD08DLgFdnvz6lilJx+caSUUCBnMV120K8NpcXcH1ctwVtdd1qLPulFf4plf4Win8Nv+BvgfL/pcju7PMFAEYVA8mpkVtnw0nmImDDMuDWyKnZr0+pYog/NXLrkYP95oVk5BJ5a8R120L32qy9Rtll33VbQNxHcb89+dYay36Jhb+f0q/4ry/9tRX/Qj7u/6vZZwsApIjh5HWbDSzJy4DXZ78upYvB/nXzFYBRFwOu2w7iNXpdAYXfdVvQ/PdbGcku+Er/ON/tV/xX+jn/C7PPFQBIEYPKIyP3bDfIjLwMuCfyyOzXpXQx5D8ycs9yBWGrjy+vVCLvibhuO+heo7XXKrv0u24LWO1+m2bhn2LpV/yrKP5d/iZyTPa5AgBpYmh5xzyDzWqLgLmXAddkvx61iMH/Hf2XiaUXA67bnOK1ekcBxd91W9Aw91s7Zb+U0t/ed/vH+/n+Wn7BXw+/3f83s88TAEgVA8wLFx14BlwGvCj79ahFlIEXjlc8tv7UwNqCwHWbU7xWLyyg+M/ius1p3PutnsI/1dKv+FdZ/Gf53uzzBABSxSCzJ3LbskNQj8uA2yInZL8etYhysCdyW/Z3JG9+8JPj3/Bk121OUbr3RG4roPx3/wbXbU5j3W/ZZX68wq/0Z3/Mf4LFv8tXIg/KPk8AIF0MNlf3MRytONBdnf061CYKw9X5C4Bvc90WFMX76gIWAK7bgvq+37KLvNI/zeLfd+mvpPjP8mvZ5wgAFCEGnB/ve2BaYrj78ezXoTZRIn68gAWA67agKN8/XsACwHVb0Cr3W3aBr7Xwt1366/6Yf2XF/9AnDue7s88RAChCDDv7hhyk5hz09mW/DrWJYrGvgAWA67agKN/7ClgAuG4LWuR+yy7wSr/iX2PxH7j8/3XkgdnnCAAUIwafL44xXG0x8H0x++uvVZSNLyaWf9dtSVHAv9hXmd/plzRuEtdtSRvvt+zSXkbhV/pL+5i/4n9E8Z/lLdnnBwAUJYagt489dK0b/t6R/fXXKgb5t68f6kdeALhuS4oS/vYlivs2cd3GsPF+qyn9Fv76S3+L3+1X/Dct/rM8I/v8AICixDB0eeIgdnn211+rGOwvn2f4H2gB4LotqXvtEj+54botad77rYSUWPinVvoV/yKKf5cvRnZnnx8AUJQYWi6eDS8JQ9lLsr/+WsWgf/EqJWHFIum6LSleu4sTFwCu25JWvd/qKvxKv+JfR/Hfofx3eWP22QEAxYnh5WmbDTQjDWjflf311yoG/6cNUSbmLJKu25LitXta4gLAdVvSUPdba4V/WqW/jY/5N1r8Z7kg++wAgOLEEPPY7QacgYe1c7O//lpFEXjs2MVjXZF03ZYUr91jExcArtuSMu43hb/U4t/Gd/sbL/5d9kd2ZZ8dAFCcGGj2zjv4DDC0PTL7669VFIO9id+FdN2WFCV8b+ICwHVb0pj3W+mFX+lX/Asv/rP86+xzAwCKFMPNicsMRD0NcKdlf/21iqJwYuICwHVbUpTwExMXAK7bkoa634Yp++1+l3/c0q/411v+n9jl7OxzAwCKteqQtMIwd0z2116zxAWA67aCxAWA67aCsst+u9/lV/oV/wWLf5drs88LAChan8PTgoPdsdlfe80SFwCu2woSFwCu2wrKKvttF/7xS7/iX3fxP6L8d3l59nkBAMXav+SPAPS0DPCR5CUN9ZHkOeO6LSnjRwBct9Vtd7/VWPaV/tzSr/gPVvy73BN5ePaZAQDFiiHooWMMWlsMfN+c/fXXKkrBQxMXAN+c/fXX6ibXrUqz6zZO2Z9O4Vf6Ff8ei/+9+YuTnvju7PMCAIq2f4c/AzjwMsCfJVtSlITHblUeRiiSrtuSbkr8c3Ku2/K2u99KLfulFv6c0t9u8d++9E+n+K+V/y4/ln1eAEDRYhh6+v2D0ehD2dOzv/5aRXF4+qJFo8ci6botqXvtEhcArtuSlrnfFP5pl37Fv+/iv+13/We5PXJS9nkBAEWLIeUls2Fl/xEZZUB7SfbXX6soEi/pq5AsUSRdtyV1r13iAsB1W9Ly99twZV/hn17pr734D/Bx/4357eyzAgCKFwPLqzcbYkZaBvxy9tdfqygXrx7jZ5G3KJKu25LitXt14gLAdVvSfPfbsGW/9MI/5dKfX/zLLv0jFP9ZnpV9VgBA8WJweed2Q83+o9Lr4PbO7K+/VlE43jnGAmCLuG5LihL+zsQFgOu2pCPvt+GLfg1lv6TS77v90yj+O5T/z0UekH1WAEDxYog5MO+wM8Ay4ED211+rKAgHji4Noy0AXLclRQk/kLgAcN2WtPn9pvBPsfQr/sUV/1lelX1OAEDxYpB5zLIDUI/LgMdkvw61ibLwmMXKxSBLANdtQVHAH5NY/mdx3Ra0+P3WTtlX+scr/Yr/0sV/lsdmnxUAULwYai7pYzBacRlwSfbrUJsoEJf0U0RWWgC4bguK8n1JAQsA121Bq95v2eW9vsJfTulv5bv9jRf/Lh/MPicAoAox3Fzd97C0xDLg6uzXoTZRKq4e/meQd1wAuG4LivJ9dQELANdtQYvcb9nFvd7CP63Sr/j3VvxnsdgEgJ3EkHNC5PYhh6g5lwG3R07Mfj1qESXjhMjtwy8AdlwQxL/h21y3OcVrdcLaa7bqJy9Wjeu2gO3ut+zSrvTXV/oV/6V+s/9OuSPy4OyzAgCKFwPPRWMPWNssA34o+/WoRQzuF202zCctAly3OUXpvqiA8j+L6zanz2xxv9WY/LJ/dOnPLPy+21998Z/lyuxzAgCqEIPPH2YOXRuWAX+U/XrUIgb5P1xk6B94AeC6zSlK9x8WUPxncd3mtOj9VlLyi/60S7/iP3jxn+W7ss8JACheDCb7IvfcUMAgtrYMuCeyL/t1KV0M9fsi9/RRDnpaANwTcd12EIV7X+SeAor/LN2/xXXbQZ/325QLv9Kv+A9Y/vdHdmWfFQBQvBhS3rDZ4JI8oL0h+3UpXQz5bxijSCy4BHDddhBl+w0FlP6Ncd12MNb9Vn/ZV/rH/G6/4n9Efib7nACA4sWgclrk1p2GmYRh7dbIQ7Jfn1LF0H9a5Nbs4rHJAuDWiOu2hSjap0Vu7a+49/bpDddtG6Xcb2UX/vI+2l9e6Z/Kd/tHL/5dvv7fT3riI7LPCgAoXgwtr150yBlxeHtN9utTqigBr84uIdvEddtClOxXJ/2Cxnnium0h+37LL/b1fJd/7NLvu/3pxf/Qfz+ct2efEwBQvBhczozcscrgM/Agd0fkzOzXqTRRCM6M3FFA0d8q3b/NddsgCvaZkTsKKPpbpfu3uW4bjH2/5Zf6+gp/maVf8R+p+M/y3OyzAgCKF0PMVX0ORAMNdldlv06liZJwVQElf6e4bhtEub6qgJK/U1y3DYa83/ILvdI/9dLfQPHv8sXIg7LPCgAoWgwzFw45KPU86H1v9utViigNFxZQ7ueN67YmivWFBZT7eeO6renzfssv8vUX/ozSr/gXXfxnuTz7rACAosVAc0rk5rGGpx6Gvpsjp2S/btliOD8lcnMBxX7edP/WyV+3KNSnRG4es8S7bqvrXoO116Lhsl9+4Z9K6a+j+G9d+pOKf5d7ImdlnxcAULQYXK7shpeMoWqFIfDK7NctWwzrV243yBdQ+DfL5K9b9xoUcB1ctwXNc93yy3ubhV/pL6n05xX/Ocp/l2uyzwoAKFoMMC/dONBkDFpLLgNemv36ZYnB/aXLDvwFlMnJXrfuay/g9XfdFrTxuuWX9vYLf1bpV/zL+pj/AsXfL/8DgJ3EEPOUyF3bDTmFLwPuijwl+3UcWwzxT4nc1XcxGLFI3hWZ3HXrvua1rz27yLtuCxjqflP4lf6xSv9Ein+Xz0cemH1mAECRYvjYFzlw/zCy8/CTNZjtsBA4ENmX/XqOJQb6fZEDYxaIgcrkgchkrlv3ta59zdkl3nVbQMb9pvC3XfoV/0GK/yw/n31mAECRYgA5I7J/6wGlumXAjZEzsl/XocVwf0Zkf37B6G05cGOk+evWfY2R/QWU974yietW4v22U9mvofAr/aWX/iqLf5evR5o/lwBgYWvl/5PzDS3zDUuFLAM+0fISYK2MfDK/cPS+IPhEy2Vyrfx/soDS3neavm5l3m91lv16Sr/iX2nxn+UPss8NACjO2sf+P7PcIFP+MiDymU83+OMA+w9/DPkz+QVksCXBZ25s8GPlNx7+2P9nCijrQ6XJ61bO/abwK/25H/OvpPjP8j3ZZwcAFCWGkqdG/rqfAafoZcBfR56a/Xr3JYb+p0YZ+Ov8MjJ4uq+xmesWxfipkb8uoKQPne5rbOa6de/BnPut7rJfQuFX+idb/LvcENmVfX4AQBFiUNgVuTQGlK8NMfQUugz4WuTSSLUDQQzUuyKXRr62WTk4nPTS3nfia33SpZFqr1uU4V2RSyNfK6Ccj5WvrX3N1V637j239t772thFv8ayr/S3WfyHLv0DFP9Z/pfsMwQAihDDwmmRqzcOEEMNQgUuA94WOS37OiwqBuvTIldvN3g3vhR4W6S66xYF+LTI1QUU8qUy1evW/ZsjVyv7dRT+KZZ+xX/b3Prxk554SvY5AgDpYmB4TuSzOw0V2cuAERYCn4s8J/t6zCuG7OdEPrvsgN7QYuBzkWquW/dvjXy2gNctOxO8bu0V/XoLv9Jf0sf8By7+hz5+OG/JPkcAIFUMDWdGfm+ZYaPxZcDvR87Mvj5biWH7zMjvDTXIV7wY+P1Isdet+7dFfq+A16m0NHjd2i36pRX+KZd+xX/u4h/51i6Pzz5PACBFDA0Pjbwucseqw8dwi4D0ZcAdkddHHpp9vWZi6H5o5HWROzIG/koWA3dEXh8p5rp1/5bI69b+bdmvT6mp9Lq1X/RLLPyllP56v9vf9Mf8Nyv+Xf7f7DMFAEYXg8OjI2+K3DbEUDLsMmD+hUDPg+ZtkV+JPDrrusXw/ejImyK3ZZeAxRYDqcuB2yK/Ekm7bt3/duRNa/+W7IJdSwq8blu/v7PvvSkV/sVK/7DPIt/tr6r4z/L8rDMFAEYVQ8NJkRdH3hU5ONagUsoyoMeFwMHIuyIvjpw09HWLAfykyIsj74oczC4DwywHRlkQHIy8K/LiyODXrfvfWPvfetfa/3Z2oa41I1+38+K6nRf/W+fF/+Z5B6dS8tsp/Eq/4r9l8e9yc+SBQ58jAJAiHva7I0+KXBa5JnLnbAjIGmAaXAZ0Px5wTeSyyJMiu1e9bt1/I4bwJ0Uui1wTuTO7EDS4IOg+yn1N5LLIkyIrX7fuv7H237ps7b99ZwHlubUMcN3Oi+t2Xvy3pnW/lVz2lf72Sn8BxX+Wy1Y9MwAgTTzYT4w8LPKotaJ/YeTSyBWRd0du2WkwyBpqxlkGpCwEbom8J3JF5NLIhZFzIw+PPDhybOTEyMMij1pbGly49n/b/f9599p/44j/bnZJKCE7LwhWWhTcEnlP5Ir9h/+u+4WRcyMPjzw4cmzkxMjDIo9aK58Xrv3fdv9/3r3238guyFPL2nU7L67BeXEtzotrcl5ct/Piup0X1+28uG7nxXU7L67beY/69OHF2oWRSyNXRN4duSX7vT1mskt9LYU/s/Qr/oMU/y63RvzpP6A9//XEJ5we+d7IyyNv+W8nPuE9kU9Gvhw5tHW+de78eY+Z48BeIocfBkM/dMZ6eO70ED+c1YcFy4DBf5lgr8kuEqVm/kVBcb+0UI7I/Ncx+z1XarLPqH4Lv9LfR+mfaPGf5Q3ZMzpAL6Lknxj5gcgbIx+PHNqY7Yv/couAPpcAtS8CyloG9DNEtL8MaG8hYCmweBZbFlgeDFnilfrVkn321Fr42yj90/lu/5LFv8s9kX3ZMzvA0qLUPyDyzMhbI7duVvotAqa+CKj/UwE+HWApUGpWWxxkLB7y/r3Z16rFZJ8ptZf+7Gea7/aPWvxneVv27A6wlCjxJ0cui9w0b+lfbRGQ82MBFgE+FZC7DGh7IWAxUF8U8mkm+4xopfC3VfrL+W5/JcV/lqdlz/AAC1n7mP/PRG5ZpfhbBEx9EWAZYCFgMSBSWrLv+RYL/9RKv+K/ba7PnuMB5vaxE56wK/JPIn/VZ/Ef+scCLAJqWAb0N5hkD1jjLwMWXwjUvhSwGBBZPdn38Phlf9zC317pn953+3su/rP8SPY8DzCXKP3fEnlX5ND6WAQMtwjwqYA2lgEWAhYDIpnJvicV/rZLf4vF/+MnDVL8u/xl5JjsmR5gR1H0L458dWP5H3oJUMsvCrQIKH8ZkD2M5S4Dpr0QsByQKST73ppq4S+p9Pdb/Msq/Q0U/3sT8+Jl2TM9wLai3B8XeetWxb/MRUDepwFaWQRYBlgITGkpYDkgtST7Himv7Cv8Sn81xb/L30ZOyZ7tAbYUhf70yIfnLf9+LGCMRcD9ywCLAMuAEhYCrS8FLAhkzGS/x8sv+zmFv+3Sr/gPVfrXFf9ZXps92wNsKUr8oyM3LVP+y/o0gEVArYuAjGWAhYClgCWBDJXs96ayr/CXXvobLv5d7oqckT3fA2zqYyec+/jIF1Yp/xYB4y8CLAPaXwbUvBCY8lJgq2QXUlHq+86qZ4TCr/SPU/pHLf6z/Pvs+R5gU392wrn7Il+IRHmfxSLAIqCcRYBlQEkLAUuBsZNdeFtO9rWtLTWX/VIL/5RK/4SKf5eDkXOzZ3yAo0Tp3xu5qSv/s9S0BLAIyFsElP2pgOksA8pYCKy+FLAYGC/ZpVthLz993M/5Z1L+2Vxa6W/9u/2FFP9Z3p494wMcJcr+8ZEPrS//FgG1LwLuXwa0vgiwDCh9IdDPUsBiQGS49HWP5p815Rf+qZX+vOI/3Hy14Kz3HdlzPsBRouT/xlblf6glQFk/FjCNRYBlwLDLAAuBnKWAxYDI/Onzvss/Q6Zc+JX+sb/bv0Tx7/Le7Bkf4ChR7n90p/JvEWARYBnQ3jKgvIVA/4sBywGZYvq+h/LPhPoKv9I/+eI/y4XZcz7AEaLQnxX56iILAD8WMM4ioMUfD8hcBGQvA2pZCJS7FOh/MWA5IDVniPsh/x6vt+wPV/jLL/25xb/I0j/LddlzPsARosTvivzRMuX/6CWARUDtiwDLAMuA+hYCwy0GLAgkO0O+r/Pv2TYKv9Lvu/075PnZsz7AEaLA//Aq5b/mHwuwCLAIKGkZUNtCoI6lwPDLAUsCWTZjvC/z7732yv6whb+O0q/4z51PRB6QPesD3OejJ5y7J/L5vhYAFgEtLQIsA7IXAtkDbvtLgfGWAxYF08rY76n8e6jtsj986Z//eZP9bGy19A9Q/Gf50exZH+AIUf5fGTn00Z4XABYBFgGWARYCdS8F8hYEFgblJfs9kH8PTK/sD1v4lf4JFP9u7tsfeWD2rA9wnyj9J0a+MlsADLUIqHEJUOsiYMxlQMZAUMIyYLGFwLADZPZwPO2lQFlLAkuEkst7u+W+pbI/fOGfv/RnP+OySv+Rxb/O0v/nR857P5496wMcIcr+yzeWf4sAi4BaPhVQ3zLAQmCai4G6FgWi2E+h6JdW+JX+5op/l89Ejsme9QHuEwX/AZGbtlsA+LGAVRYB5SwDhl8E+FRAicuAlhYC01gMWBiUn+z3gqJffuGvr/SXUfyHn1NGKv3r85PZsz7AEaLcf+9O5d8ioK1FgE8FWAhkD98WA2MnuzCXmOxrUmay72GFf4qlv6nv9m/MZ333HyhOlPr/vMgCoKYfC7AIKGERYBmw2jLAQsByQKTfZN+LUyz8Sv/4pT+5+M/yz7PnfIAjRJE/IXL7MguAmj4NYBFgEVDqMmDxhcBYg23+AG85ILJ8su+p9sp+3d/lzyz9Ey3+Xf4yclz2rA9whCjxz122/FsEtLkIsAyoaRlgIWA5IFNO9r2h8Cv9Uyn9Cxb/tTzhZdlzPsBRorz/uz4WABYBOYuANj4VYBlQ60JgyksBSwIZOtnv6VIy9plWe+GfWukv6Lv964t/l5sifvYfKE+U9hv6XAAMvwSo7xcFtrAImNqnAkpdBtSwELAU2D7ZpVLyk/0eLDkZ51ULhT+79Ptu/xHFf5Z/lj3jAxzl+j3nPizSa/n3aQCLgBaXARYClgIZyS6qotQPmayzSOGvt/RXUvy7fDrywOw5H+AoUf6f0y0AZrEIKHkRUN7vCbAMKC+LLwMsBVpNdiGuMdnXrOXUUvaV/mmW/uWK/7Yz249kz/gAm4rS/8r1C4AhlwAWAXmLgLY+FWAZMM5SIHNgzy8rMn8Uc1mf7LOjpbI/9dJf0Xf7N+YvIruzZ3yATV2/55xf27gAqPHTAC0sAlr58YCsTwVYBgy9EMhfClgMiJST7LNg2bJfQ+Evs/RP/bv9cxX/WS7Knu8BtnT9nnPeE9l0AVDjpwEsAqa6CCjvUwEWAhYDIi0k+95W+LOLfzvf7R+h+He5Pua4B2TP9wBbivL/Xw8vAGaxCKh7EWAZYBmQuRAoaylgMSAyf7Lv1b7Kfi2FX+lvqvTfm7XZ7buzZ3uAbUXpv+nIBYBFgEVAS4uAcpcBNS0EWlsKWA7IVJN9z0257JdW+DNLf6PFv8u7sud6gB1F2f/y5guArZcANS4CPnaCRYBlgGVAWUuBchcDFgRSa7LvmSGLfm1lX+mvqfSvXPy7HIx8W/ZcD7CjKPp3bb0AaPnTAHUvAsb6PQFjLQKylwEWAqUsBMpfClgQSGay3/Njln2Ff4jC32bpH7P4bzOX/afsmR5gLlHw/37nBYBFQMnLgJZ+PMAyoK2FQH9LgboWA5YEskyy36uZRb/Wsl9i4Z9S6V+++K/83f6N+VrkUdkzPcBctv8RAIsAi4CpfSqgjmVAzQsBiwHLgikl+z2l6E+j8JdQ+lv8bv8cxX+WN2XP8wBzi0L/mcUXADlLgBp/UWA9iwCfCphnIZA94LW8EOh3KdDWYsDSoIxkX+vain4LZb/kwn906c95Rrb43f4Fin+Xr0Yelj3PA8zt6D8DaBHQyiKg1U8FWAZMZyEwzGJgOsuBZZJdshX2sdLvPZV9Rkyl8Cv9RXy3f2N+PnuWB1jIdXvOeddqCwCLgHYXAZYB8y4DLARaWQxYEEgrGebeyL7nFX6lP/m7/RvzpchJ2bM8wEKu23POr0UOXTfgEqDW3w9gEVDXIsAywEJg+MWABYGUkmHf49n38JQL/1RLf2XFf5afyp7jARYWxf8VswXA0IuAWj8N0NovC1xtGbD8A3o6y4A6FwKtLwXGWw5YEsiqGec9mn0vKvvlFf66Sv+oH/PfLJ+OPCh7jgdYWBT+CzcuAPpZAlgEtL0IsAxofRkwlYVA3oLAomB6Gfe9lX0vKfxKfymlv+fiP8tF2TM8wFKi7O/dbAFgEZC7CJjKMmDsoSN/GVD3QmCqS4HcBYGlQZnJfg/k3wvZyT4Lay78GaV/teJfTOmf5SORXdkzPMDSouh/crslQM2/H8AioI5FgGVAncsAS4HNk10MLRHKL+/K/WLJPuOU/sl/t399DkYuyJ7dAVYSBf/NOy0Ahv40gEVAmYsAywALAUuB4ZJdMkWxLzHZZ1dLhV/pHyRXZs/tACuLcv/98ywALAIWXQZYBKy6CJj2MqCthYClwOrJLq+i0Ped7DOpxcJfZ+mvovh3uT1yRvbcDrCyKPV7IrcusgSo/fcDtLAImNoyIGOgKWsZ0N5CwFJg/GQX5pKSfS2mluyzpuXCn1X6J/Dd/iMSc9fPZs/sAL2JQv8fF10AWARYBCy/CKhzGWAhYDEgItsn++xQ+Nss/cnFv8vNkeOz53WA3kSRf+ayC4D+lgAWAbUuAiwDSkm7CwGLAZGykn0WTK3wK/1pxX+WF2TP6gC9ihL/gMgNqywBWvj9AOMuAtpcBmQtAiwDprkQsBgQGTbZ9/aUC3/dpb/O4r/FTPXHEX/2D2hPFPj/adUFgEWARYBlQD0LgdaXApYDIvMl+x7NKfsKv9K/Y/Hvck/kH2bP6ACDiOJ+YuRAX0sAi4CyFgGWAZYB8ywFsod0ywGRYZJ9r+UX/uyzVumvqPSvz7/Lns8BBhWl/RV9LgAsAsr7PQF1LwLqXwbUthCY4lLAgkBqTPa9UlbZV/hLLv2VFP8ufxd5WPZsDjCoKOvHRW4udwlgEdDKIsAyoJaFgKXATskuftJ+st/jJaW2sl9K4Z966V+w+M/yv2bP5QCjuHbPOS8YYgFgEeDHA0peBpSyEMgeVC0Fhkt2iZTykv2eLD01lv2SSn8fz8UJlv5ZPhU5NnsmBxjNtXvOece1Ay4BLAIsAiwD2l8IWAosn+xiKkr92NnqDMk/2+op/Ep/L8V/lmdnz+IAo4ryf2bkb4ZeArS0CGjtxwMsA8pbBtS1ENh6MZBdNFpPdgFuIdnXsPXUXPRLK/xKf6+lf5bfzZ7DAVJE+f+hbgEwS+2LgLGWAD4VUOoioJ9lgIXAMEsBi4G6opzLPKn9u/ptF/7VS3+Dxb/LrZFHZM/gAGmi+L9l/RLAImDaiwDLgLKXAfUuBCwGRGpOS0W/xMKv9A9e+tfn0uzZGyBVFP5jIh/YuARoYRHQ2o8GjP3jAZYB5S8D2lgKWAyIlJIWi36JZV/pH730z/JnkQdmz94A6aLs743csNkSoJ5fFDjVRcB0PhXQ3zLAQqCebF1IsouSSK3Z7r7Kv+cV/rEKfymlf6Ti381LByMXZM/cAMWIon9W5HMZSwCLAJ8KaGEZUPJCoK2lwPaLAcsBkZ1KfjtFv+Sy32/hV/pXKP73Jual/yt71gYoThT9x0Q+u9USwCKg7GXA2J8KKGkZ0N9CYBrLgPYWAhYEMq1MqeDXVPhLLP3Zz+bM0v+x+2ekA5FTs+dsgCJFyf/myKe2WwJYBFgElLwI6G8ZMK2FQPtLgfmWAxYEUkLmeZ/m30vKfomFv7TSn1z8Z/mn2fM1QNGi4D8k8r6dlgAWAWUvAiwDyl0G1LIQmM5SwJJAxotyX2/Z77/wK/0Dlv5Z3hvZlT1bAxQvyv2DIq+PHLQIsAiwDLAQsBRYfElgWTCdLPJ+yH/v5if7DGuh8E+79M9V/Lt8LXJ29kwNUJUo9/9j5MslLAFaXQRYBlgG1L4QsBTYKYsWRIuDOoq8Uj9Pss+m/MKv9I9c+tfn1dlzNECVotyfEfmdeZYAFgH1LAKOXAaM+xDPHlgsBCwGys8qZXRay4S+X6v8a19nss8ahb/V0n9/8V9wxvl0ZE/2DA1QtSj33xO5wSKg5UXAtD8VMMwywELAYiA7Q5Tk0pL9Gk8n2WdH64W/1NI/fvFf+Lv963Mw8szsuRmgCVHsj4v8VORmiwDLgCksA2pZCLSwFLAYECkn2WdBuWV/OoW/stK/Pr+RPS8DNCeK/TGRl0Q+OcVFgE8FWAaUvgxoZSFgOSAyXLLv6XoKv9JfSfHv8sXIqdlzMkCzotjvijw98luRv7MIsAyY0jKgtoVAi0sBywGR7ZN9b9ZZ9qdV+Bso/evzguzZGGAy1n484MLIayIfidzd1iLAjwdYBmQsBHxKwHJAZPtk32P1F/7+z+3sZ9HESv8sb8uehQEmLUr+KZGnRX4i8rrINZGPRT4fuSVyu0VAvcuAjx2V8YeJ7OEpZxngUwIWBDK1ZN8jbZX9YQp/DaU/4zm9cVYYcC65JfLw7NkXANLEg/CFU/hUQC3LgJoXAlNcClgSyFjJfm9nZ4zza6qFP6/0j/Ld/o350ey5CwDSxQPx93M/GWAZkLMQGGcpYDGwebILpeQn+z1YYsY6k4Y8V7OfGUr/lrkme94CgCJ89IRzH/7Rwx+LG/thbBmwRFpYCFgKLJ7ssioKfd8Z87xR+Cdb+mf525hzzsietwCgGPFgvDj7FwdaBpS4EBh3KWAxMFyyy3ENyb5GrWbsM2ToMzH7zFf658+6ueYl2XMWABQlHo67Iv8l+5cGlrYMsBAoYylgMSBSfjLOhTHOu+wzvZ7CX0bp31D8D63NNruy5ywAKM7ajwJ8pZS/IGAZYClgOSBSVjLvc2Vf6V+g9M9yS+QR2fMVABQrHpQXlfjnBEtcBlgIlL0UsBwQWS7Z9+xYZV/hb7b0r89F2XMVABQvHpj/aZ4lQPYiwDKg1qVAGYsBywGZcrLvvbGLfs1lX+lfqvh3+c/Z8xQAVCEemqdG/nLeJYBlQJsLgfGXAuUsBiwIpPZk3zuZRb+Fsl9G4a+y9M/SzTCnZs9TAFCNeHBeGDm46BLAMqDthUDOUqC8xYAlgWQm+z1fUtFvpeyXU/irLv2zdLPLhdlzFABUJx6gb152AVDKIsAyoOWlQNmLAYsCWTTZ79HSi35LZV/pH6T4z3JF9vwEAFWKh+gJkRtWXQKUuwywEGhzKVDfcsCyoN1kv5dqLPmtlv1SC38jpX+WbmY5IXt+AoBqxYP0gsjdfS0BLAOmuxAoYylQ/3LA8kCJb6nkt1z2yyr8TZf+WbpZ5duz5yYAqF48UH+p7wVAacuAoxcC2YPaNBYC5S0G2l8QtLhgyH5Nyk72vTSNol9m4T+y9Gc/Ywcq/evzS9nzEgA0IR6qx0Q+OuQSoOxlgIWAxYAlgZSY7HtgukW/hsI/kdI/y591s0r2vAQAzYgH6+Mit4+xBChtGXD0QiB7wJvuQqCOxYAlgfSR7Peuol9H4T+y9Gc/K0cu/bN0s8njsuckAGhOPGBfOuYCoI5lQJkLgSkuBepbDlgYTCvZ7y0lv42yf2Thn3DpX5+XZs9HANCseND+TtYSoMRlwNELgezB0EKg3eWAxUEZyb7WSv5QyT6j5y392c+8Qkr/LL+bPRcBQNOu33Pug+OBe1P2EqCOZUDZCwFLgaOTXcLKSnbRVtb7TPa9VVqyz95FCr/Sv2W6WeTB2XMRADTv+j3nPjVydyT74V/0MqDGhYClwM7JLnIiG5N9T5Se7DO1hcJfUOm/N2szyFOz5yEAmIx48P6rbgGwPtkDQenLAAuB6SS7EEo7yX4v15jsM7OVwl9g6V+fn8uegwBgUuLhuzvy3o1LgNIWAXUtA+pYCFgK9JvsgilKfc3JPguXLfxK/9LF/9Da7LE7ew4CgMmJB/CZka9stgQodRlgIWAx0EqyS6zkvwemlOyzTeFPL/2zdDPHmdnzDwBM1vV7zvmByHYPa8uAJdPCQsBSoI5kF2mlXdYn+8xqvfBXWPrX5ZznZc89ADB58UC+4vASYL5FgGXAtBcCFgMi0iX7DJpS4a+79J+7Nl+cc0X2vAMAfMO9C4DjIx+9fwlQ/zKghoXA5kuB7KHYYkBEjkz2mTJE4c8++2st/EuU/ln+rJs1sucdAGBNPJj/QeSrRy8BLANyFwJtLAUsBkTKT/YZMVTZr6XwN1j6Z+lmi3+QPecAABvEA/qirRcA7SwDLATKS3bxEZlKsu91hb+Owt9D6V+fH8qebwCALcSD+i3zLQEsAywFLAdESkz2PavsK/3r8pbsuQYA2EY8rI+LXL/YEqCdZUCNC4EpLgUsB2Tqyb73lP02Cv9ApX+W7ncLHZc91wAAO4gH9qMif7vcEqCtZUCtC4EpLwUsCaT2ZN8zyn7bhX/g0j9LN0M8KnueAQDmFA/u748cXG0J0N4yoOaFgKWAJYHkJ/u9XkbaKfu1FP6RSv8s3ezw/dlzDACwoHiA/3I/C4B2lwG1LwQsBRZLdnmUcpP93iw3bZX9mgr/yKV/fV6TPb8AAEuIh/juyHv7XwIstgywEChlKWAxsEyyi6ko8+Nk8zMj+yxT+Ecr/bN0M8Pu7PkFAFhSPMhPj3x+uCVA28uAVhYCFgN5yS7DNSb7mrWddot+jYW/kNI/SzcrnJ49twAAK7puzzkXRO4afgmw+EIge/Ca+kLAYkCk1bRf9Gst/IuV/tGe23d1s0L2vAIA9CQe7C+NHJqlxGWAhUB52XoxYDkgkp+t78/ss0PhX7bwj1r6D62bC34ye04BAHoWD/hfv+7IB75lwEDJHpBzFwOWAyL9Zft7LfssUParLv2z/Hr2fAIADCAe8sdGPrzZEiBnGWAh0GIsB0QWzbRLvsKfUvpn6WaCY7PnEwBgIPGgPyNyYLslgGWApYAFgUifUfBbK/uVl/5ZulngjOy5BAAYWDzw/1Hk6/MsAWpYBrSyEJjyUmCxBYElgZSUnd+v2fdUCck+W3MKf5Glf5a7I0/LnkcAgJHEg//SRRYAucuA6S4ELAU2z3xLAosCWSXzvcey74USk31mTqnwL1j61+fS7DkEABhZDABXLrsEyF0GTHshYCkwf+ZfFFgYtJ/F3gvZ791akn0WTrHwr1D6Z/mP2fMHAJAghoA9kY+sugSobRnQ4kLAUmD1LL4ssDwovcQr9f0m+4wro/BXW/pn6Z75e7LnDwAgyXWHfyngX/W1BLAQKC/ZpWEKWX150NpiYbjXI/taTyHZZ5bC32vhX5/uWe+X/gHA1MVAcH7k74dYAuQvAywELAbqzbCLhWGT/drJzsk+gxT+UUr/LHd1z/rseQMAKEQMBj8y5AKgnGWAhYDFgMi0kn2mKPxppX99fiR7zgAAChMDwmvHWgJYCNSX7BIjItsn+4xQ+Isq/Ovz2uz5AgAoUAwJuyPvyFgClLMMWG4hMNWlgOWAyLjJvtfrK/v5hT+p9M9yTfdsz54vAIBCxaDwjZG/yFwCWAi0lezCJFJbsu/ZUqLwr5zuWf6N2XMFAFC4GBj2Rf6mgOGlwGXA8gsBS4HNk122RMZO9j1XYlY5V/OfCUWV/lm6Z/i+7HkCAKjEtXvOefp1h39rcPYQU8FCwFJg6GQXNpF5k32v1JAWyn6BhX99umf307PnCACgMtfuOeefRQ5dmz/MTGohYCmwXLKLn7Sf7Pd4jVn1LMw/y6so/Pdl7Zn90uz5AQCoVAwSr772/qGimmVAeQsBS4GSkl0kpZxkvxdbSktlv5bCv670z/LL2XMDAFCxGCZ2R/5g4xKgtmVAmQsBS4Hakl1WRZEvJa2V/ZoK/yalf5buWe03/gMAq4mBYk/k+q2WABYCZS0FLAbKT3ZZLinZ10K2Tx/nUf6Z2mzhX5+Pds/q7HkBAGhEDBZnRD43zxKgxmVA2QsBSwERGT4tl/3aCv+cpX+W7tl8RvacAAA0JgaMJ0VuX2QJUOsyoPyFQD9LAYsBkemlr7Mj/wxsq/AvWPpn6Z7J35Y9HwAAjYpB4/sidy+zBLAQqGsxYDkgUm/6PAfyzzSFf4t0z+Lvy54LAIDGxcBxyaoLgNqXAVNeClgMiJSTvu/t/LOq3bLfU+lfn0uy5wEAYCJi8Pg/+1wCWAi0sRiwIBDpN0Pdo/lnj8K/Yt6YPQcAABNy7Q5/HtAyoOalgOWAyJhR8tsp+wOX/ln8uT8AYHzXHv7zgB8aegnQ2kKg7qXAcIsBSwJpMWPcL/lnwrTL/giFf326Z64/9wcA5Lh2z+P3xjByw5hLAAuB0jNO6ckudiJjvdfz72mFP7Hwr0/3rN2b/dwHACbu2j2P3xc5EDl0OGnDUfpQaDFQxnLAokBWScb7NP/eVPQLLPxruffZ2j1j92U/7wEA7hWDyfmR2+9fAuQvA1pcCLS5FMhfElgatJvs91KL5b71sp9f+O8r/bN0z9bzs5/zAABHiAHleZGDRy8BLAQsBdpbElggTKnAT7fcT6HsF1j416d7pr4g+/kOALCpGFReuf0CoJxlQMsLgekuBupaFNS4cMj+WhV7Rb+Nwr9t6V+fV2Y/1wEAtvWR4x//5si8w42FQEKyy0W5yS6dotCPm+yzSOHfOt2zNPt5DgCwoxhadkf+oFsCrE9ty4ApLQQsBvpIdsGdQrKvcb3JPlumWfgXK/0bnpndM3R39vMcAGAuMbjsiXxo4xJguWVAOQuBKS4FLAdKS3YJV8xLTfYZoewvVvg3Kf2zdM/OPdnPcQCAhcQAszdyw1ZLAAuBdpJdfESmkux7vYRkn/kDFP716Z6Ze7Of3wAAS4lBZl/kwDxLgOWXAWUtBCwFjkx2YRKpLdn3bEnJPstXLfxzlv5ZumflvuznNgDASmKgOT9y+yJLgFY+HWAhMF+yC5fIWMm+10pP9lndR+lf8lnXPSPPz35eAwD0Igab50YOLrsEaG0hYCmweLKLm8hOyb5Hakv2GZxc+NenezZelP2cBgDoVQw4r+xjAdDijwxYCvST7AIo7Sb7vV17ss/Wvsp+T4V/Y16Z/XwGABhEDDpvHmIJ0OpCwGJg2GSXSlHoW0v2WVlJ4V+fN2c/lwEABhPDzu6PHP77xkMPVU0vBCwFcpNdXkWRLyHZZ2ClhX99umfh7uznMgDAoGLg2fORw3/neMxBq/mFgMVA3cku0sq6bJXsM62hwr8+3TNwT/bzGABgFDH47P3I4b93nDV8rbgQqGspYDEgIjsl+4wao+wnFv716Z59e7OfwwAAo4oBaN9HDv/d4+xhbJILAYsBkWkm+8yZYOFfn+6Zty/7+QsAkCIGofM/cvjvH2cPZZYCG5JdUkRktWSfIcr+UemedednP3cBAFLFQPR9kbsLGM5GWAjUvRSwHBApK9lnQWbZr6Dwr0/3jPu+7OctAEARYjC6pIABzVKgp2SXIpFWkn0vK/u95ZLs5ywAQFFiQLq8gCGtkIVAe0sBCwKRo5N9L5Za9hso/OtzefbzFQCgOB8+/vG7Ir/14fxhreClQNuLAQsCaS3Z91INRb+xsn9E4nn277tnW/bzFQCgSDEo7Y78wYcPD073JXuIsxQoM9nlTqab7Pe+sl9mNjy7umfZ7uznKgBA0WJg2hP50MYlwBSWAf0uBaa7GNgs2YVRyk/2e7Ss9HMGZZ+lCaV/lu4Ztif7eQoAUIUYnPZGbthqCTC1hYDFQE6yC6ko88NH0e+p8K9P9+zam/0cBQCoSgxQ+yIH5lkCTHUh0O9SwHJgiGQX4RqTfc3aS79nRPaZV2DhX5/umbUv+/kJAFClGKSeEPnbRZcAU14IDLcYsBwQKTf93+/ZZ1glhX99umfVE7KfmwAAVYuB6oLI7assASwFhl4MWA6IDJ9h7t3sM6nSsr8x3TPq27OflwAATYjB6rsjd/W5BLAQODrDLQcsCER2znD3X/bZUkoGeoZ0z6bvzn5OAgA0JQasH4zcPdQSwEJg+wy7HLAgkClk2Hso+4woMSM8L7pn0vOzn48AAE2KQevHIgfHWAJYCsyf4ZcDlgRSesa5B7Lv9ZKT8FzonkU/lv1cBABoWgxcL89YAFgILJ/xFgSWBdJnxn/fZt+rNSX7GdA9i7KfhwAAkxCD178sYPizFOgpOQsCC4PpJfs9ln+v1Zrss32T/Ez2cxAAYFJiAPu5AoZAS4GRkl3cLBBKSPY1Vu7HSPaZPUd+Pvv5BwAwSTGIvaqAYdBSoLBkF8HpLRqyXxOlvtZkn8VL5FXZzz0AgEmLgex1BQyFlgIVJ7tkikI/hWSfsT3kddnPOwCAyYuhbFfk3xYwHFoKTDDZhXaKyb7msnOyz84B8obuWZP9vAMA4BvuWwK8pYAh0WJAek122VbQZbtkn4Uj5S3KPwBAYdaWAL9awLBoMSAiTSX7bEvMFR9W/gEAyvSnMaj96cSXAJsluzyISB3JPqsKy6/8qfIPAFC2Pz3+7HuXAJFDsxQwSBaZ7LIhIjnJPntKzbrnhvIPAFCLw0uAs6+IHDqcIwa79CGzhmQXFBFZLdlnSA3Z+GxYe2a8sXuGZD/HAABYwNoS4I33LwHW5+jBL3sQrSnZxUZEDif7LKgpm537mzwf/q3yDwBQqbUlwOs3XwJYCAyR7EIk0lqy7+laM2fhX5/XK/8AAA2Ioe7ynZcAlgJjJbtQiZSS7HuxlSxR9jfm1dnPKQAAehQD3s8uvgSwEMhIdikT6SvZ91Kr6aHwr8+/yn4+AQAwgBj0XhE5uNoiwFKglGSXO5lust/7U0rPZX9jXpH9XAIAYEAx8L2s3yWApUAtyS6NUm6y35tyOAOX/fXpngEvy34eAQAwghj8XhK5e7glgKVAK8kupqLMt5oRy/7GdGf/S7KfQwAAjCgGwB8edwmw81LAYqDtZBfjkpN9bWS4bHXWJZ293Zn/w9nPHwAAEsQg+NzI3+ctASwFRKSdFFb2N6Y7638g+7kDAECiGAifGbmtgOHUYkBEqkjhRX+zdGf8M7OfNwAAFCAGw6dEbilgSF15KWAxICJ9ZbtzJv8MnDvd2X5B9nMGAICCxID4hMgXChhWLQZEZNQ0UvQ3S3emf2v28wUAgALFoPjoyE0FDK2WAyLSaxou+VulO8sfnf1cAQCgYDEwnhH5iwKGV4sBEVk4Eyz6m6U7w78p+3kCAEAFYnDcG7m2gCG2gGxfKLLLjsjUstM9mX9mpKc7ux+a/RwBAKAiHzru7JNiiHxPAcNs4bEgEOkzCv5K6c7sk7OfHwAAVOhDx519bOT3IodmKWDArSg7l5nssiUydua5L/Lv3Xqy7nzuzupjs58bAABULAbK3ZFfXb8EsBDoM/MVouzSJrJT5n0v599zdWeLs7g7o3dnPy8AAGhADJa7Iv96qyWApcDQWaRc5RdBaSeLvvfy75W2MueZ+0vdGZ39nAAAoDExZP505OC8iwCLgYwsXtqyS6aUXOiV+jGy5JnancU/nf1cAACgYTFwvjjytWWXAJYCJWbZYmiBUEeBV+ZLSk9nZ3cG/5Ps5wEAABMQg+ezI7f2tQSwFKg5fZTS+hcNY74O+ddc5s1AZ2R39j47+zkAAMCExAD65MiBoZYAFgNyZMYu2Yq5zJ8Rz8HuzH1y9vkPAMAExSC6L7J/zCWAxYCIZCX5rOvO2n3Z5z4AABMWA+npkWuzlwAWAyLSV7LPrk3SnbGnZ5/3AADQLQFOivxRAUOyxYCIzJ3ss2jOdGfrSdnnPAAA3CcG1GMi/6GAYdlyQETuS/aZsmK6M/WY7PMdAACOEoPqrshrCxiaLQdEJpTss2GgvKY7U7PPdQAA2FYMrf9z5O4CBmgLApEGkn1vj5zu7Pyp7HMcAADmFgPs8yJ3FDBMF5PsEiVSarLvzYLSnZk/kH1+AwDAwmKQ/fbIlwsYqqtKdhkT6SvZ91Jl6c7Kb88+twEAYGkx0D7mQ4f/fnX2cN1cssudTDfZ7/0G052Rj8k+rwEAYGUx2D4s8uEChuzJJrswSvnJfo9OON3Z+LDscxoAAHrzwePOPiHy9gKGbVkg2aVUFPnG8/bubMw+nwEAoHcx6O6OvDlyaH0KGMJlpGSX4xKTfU1kvGw4+7qzcHf2uQwAAIOKofeyyD0bFwEWAyLSQnY427qz77LscxgAAEYTA/APRu7YaQlgMSAipWaJ86s7856fff4CAMDoYhB+SuRLyywBLAZEZKz0dEZ1Z91Tss9dAABIEwPxoyKf6GsJYDEgIstmwHOoO+MelX3eAgBAuhiMT428d8glgOWAiHRJOGe6s+3U7HMWAACKEQPysZHfzlgCWA6ItJXss2NdujPt2OzzFQAAihOD8q7Iz0YOFjC4WxCIFJrse3+OHFw7y3Zln6sAAFC0GJovitxZwBBvQSCSkOx7d8V0Z9dF2ecoAABUIwboCz7Y418IKDHZJUskK9n33oDpzqwLss9PAACozgePe9xZkT+PZA/1FgUicyT7XsnNvWfVWdnnJgAAVCsG6m+M/OHhJcBmyR76y0p2AZT2kv2eLi+bnkXXdGdV9nkJAADVi8F6d+SNWy8BLAdWSXbBFGW+vCx01nRn0+7scxIAAJoSQ/ZPRe5efBFgMTBkskvtlJJ9rdvLSmdJdxb9VPa5CAAAzYqB+9mRr66+BLAcqDnZRVwZrymDnBXdGfTs7PMQAACaF4P3uZGbh1sCWBCI1JPRz4Hu7Dk3+xwEAIDJiAH89Mif5iwBLAhExkv2PX1EujPn9OzzDwAAJicG8eMjv1NAKVgg2WVKpLRk35Nzpztrjs8+9wAAYLJiIN8VeVUB5aDnZJcykVWTfQ/1loORX+zOmuzzDgAACB847nEvjtz5gfyyMHKyS55ML9nv+fGydqa8OPt8AwAANohB/YLIFz5weHA/KtllopxkF0gpJ9nvxTKyxZnRnSUXZJ9rAADAFmJgPyNy/VZLAMuBVZNdWEWJXy4LngnXdWdJ9nkGAADsIAb3EyJXLboEsBzITHaJVtRbSE/3fHd2nJB9jgEAAHOKAX5X5OcjB/taBFgQiORm4Hv54NqZ4Zf9AQBAjWKYf0HktqGXAJYEIqsl+R7tzogXZJ9XAADAimKwPy/yuewlgCWBTDXZ99YO6c6G87LPKQAAoCcx4J8e+WABZcOiQJpJ9r3QQ7oz4fTs8wkAAOhZDPrHRX6rgNJhYSBFJvu9OnJ+uzsTss8lAABgQDH0/4vIPQUUkKKTXUZFkR8o3b3/L7LPIQAAYCRRAP5x5KsFlJHJJLsUl5zsazOhdPf8P84+fwAAgJFFETgnsr+AUiIiw+fGyDnZ5w4AAJAkCsFDIu8toJyIyHDp7vGHZJ83AABAsigGD4pcUUBJEZH+093bD8o+ZwAAgIJESXhp5K4CCouIrJ7uXn5p9rkCAAAUKgrDd0S+WEB5EZHl093D35F9ngAAAIWL4nBG5NoCSoyILJ7rIt+UfY4AAACViAJxfOTKAsqMiMyf7p49Pvv8AAAAKhRl4hWRuwsoNiKydbp79BXZ5wUAAFC5KBbfE7mlgJIjIkenuze/J/ucAAAAGvH+Yx+3L/LxAsqOiNyfj3f3Zvb5AAAANCaKxkmRt0UObZUCCpFIU9nmfuvuxZOyzwUAAKBRUTh2RX4xcnC7RYAlgcjOWfIe6u69X+juxezzAAAAmIAoH8+P3LrsEsCSQFrPQPdGd889P/v+BwAAJiaKyBMi+4daAlgWSGlJfq9399oTsu97AABgoqKQnBb5f7KXAJYGsmiy34MLprvHTsu+3wEAgImLYrI78tr3r/B7AWpLdnmV/PfASOnuqdd091j2fQ4AAHCfKCkvitxWQGlqJtklWzFPTXcvvSj7vgYAANjU+w//XoBPF1CeRGpOdw/5eX8AAKBsUVxOjbyzgBIlUmO6e+fB2fcxAADAXN5/+PcCXP7+Cf1eAJEV090rr3q/n/cHAABqFGXmB99/+G+XZ5crkZLT3SM/mH2/AgAArCSKzeMjnyqgZImUmO7eODv7PgUAAOhFFJxTIu8ooGyJlJS3R07Jvj8BAAB6FUXnAZFfeL/fCyDS3QPdvfCA7PsSAABgMFF6nhv5uwJKmEhGuvf+c7PvQwAAgFFEAXpc5BMFlDGRMdO95x+bff8BAACMKorQyZHfL6CUiYyR7r1+cvZ9BwAAkCIK0a7Iz0TuLqCgiQyR7r39LyO7su83AACAdFGOnhX5cgFlTaTPdO/pZ2XfXwAAAEV5/7GPfWTk2siho5Ne5ER2yFHv2+69/Mjs+woAAKBIUZiOi/z65kuARZJdBqX+rPQe7N7Dx2XfTwAAAMX7k2Mf+xORv/+TlRcBFgcy3vtn7T37E9n3DwAAQFWiSJ0f+dyfHC5Wc2fchYFFQgvFfZtCv0i69+r52fcNAABAlaJQ7Y28Z9ElQJ1LAymozC+a7j26N/t+AQAAqFoUqwdGXhs5OPQiwDKhrGRfyznSvSf/Tfcezb5PAAAAmhEl64WRWwsofSJduvfiC7PvCwAAgCZF4To78okCyp9MO9178Ozs+wEAAKBpUbxOjvxOASVQppnfjZycfR8AAABMQhSwXZGXR75eQCGUaaR7r3XvuV3Z738AAIDJiTL2nZG/KqAcStv5QuQ7s9/vAAAAkxbF7PTIewsoidJmuvfW6dnvcwAAAL7hvj8V2P05tir+VKBUEX/iDwAAoFRR1p4X+bsCyqPUne499Lzs9zMAAADbiOL2mMh/K6BESp3p3juPyX4fAwAAMIcocHsi/6GAMil1pXvP7Ml+/wIAALCgKHMvi/x9AcVSyk73HnlZ9vsVAACAFUSxe0rk5gJKppSZ7r3xlOz3KQAAAD2Igrc38kcFlE0pK917Ym/2+xMAAIAeRdHbHfmFP/GnAuXwe6B7L+zOfl8CAAAwkCh9z4ncUkAJlZx01/452e9DAAAARhAF8KzIdQWUURk33TU/K/v9BwAAwIiiCB4X+bUCSqmMk+5aH5f9vgMAACBJlMIfi9xZQEGVYdJd24uz32cAAAAUIAriEyM3FlBWpd901/SJ2e8vAAAACvK+Yx97SuTt78svrdJD4jr+3901zX5fAQAAUKAojLsi/1vk7vcdLpG9JbsQl5yeX+u7167hruz3EwAAAIWL8vhdkQN9LwFk8Hypu3bZ7x8AAAAqEkXyEZE/LqDUynzprtUjst83AAAAVCgK5QMjr48cLKDgyubprs3rumuV/X4BAACgclEunxv5uwLKrhyZv+2uTfb7AwAAgIZE0Xx05KMFlF45nO5aPDr7fQEAAECDonAeF/nNAsrv1PMb3bXIfj8AAADQuCifPxq5s4AiPLXc0b322dcfAACACYki+sTIpwooxVNJ91o/Mfu6AwAAMEFRSE+OXF1AOW493Wt8cvb1BgAAYMKimO6KXBr5egFFubV8fe213ZV9nQEAAOBeUVK/M/L5AkpzK+ley+/Mvq4AAABwlCiseyPvKaA8157uNdybfT0BAABgS1Fcd0d+MXKwgCJdW7rX7Be61zD7OgIAAMBcosQ+J3JLAaW6lnSv1XOyrxsAAAAsLArtWZHrCyjXpad7jc7Kvl4AAACwtCi2x0V+vYCSXWq61+a47OsEAAAAvYiS+5LInQUU7lLSvRYvyb4uAAAA0LsovOdFbiqgfGenew3Oy74eAAAAMJj3Hf5Tgf9fASU8K93X7k/8AQAA0L4owA+KXFFAGR87b+q+9uzXHwAAAEYVZfji903j9wJ0X+PF2a83AAAApIli/MTIRwso6UOl+9qemP06AwAAQLooyA+MvDzylQIKe1/5ytrX9MDs1xcAAACKEmX5lMjPRQ4UUOCXzYG1r+GU7NcTAAAAihbl+djIP428M/K1Akr9Tun+jf9l7d98bPbrBwAAANX542Mee2rk4sg1kTsihwrJHWv/pu7fdmr26wQAAADNiKL9oMj5kZ+OXBX5TOTgCGX/4Nr/1lVr/9vdv8Gf8wMAAICxRBHfE/mHkRdF/vfImyJvi3wgckPkpsgtkds3Kfa3r/2/3bT2f/uBtf+/b1r7b71o7b+9J/vrBAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACO9v8DFwsrQ9rxlToAAAAASUVORK5CYII=">
                                    <div class="divider"></div>
                                </div>
                                <p class="headerName">Issuer Icons</p>
                            </div>
                            <nav class="scroll">
                                <ul>
                                    <li> <a href="https://github.com/raivo-otp/issuer-icons/tree/master#contributing" target="_blank" rel="noopener nofollow">Contribute</a></li>
                                </ul>
                            </nav>
                        </header>
                        <article class="page markdown-body">
                            <h1>Raivo OTP Issuer Icons<br><small><small><small>__AMOUNT_OF_ICONS__ vector icons that can be used within Raivo OTP!</small></small></small></h1>
                            <div class="issuer-icons">"""

    ordered_manifest = collections.OrderedDict(sorted(manifest.items()))
    amount_of_icons = 0

    for domain in ordered_manifest:
        data = manifest[domain]

        for icon in data["icons"]:
            amount_of_icons = amount_of_icons + 1

            contents = contents + """
                <div class="issuer-icon">
                    <a href="https://raw.githubusercontent.com/raivo-otp/issuer-icons/master/vectors/{}" target="_blank">
                        <img src="https://icons.raivo-otp.com/{}" alt="Scalable Vector Graphics (SVG) logo of {}" />
                    </a>
                    <p>{}</p>
                </div>
            """.format(html.escape(icon[0:-3] + "svg"), html.escape(icon), html.escape(domain), html.escape(domain))

    result = contents + """
                            </div>
                        </article>
                        <footer>
                            <div class="footerLinks"> <a href="https://raivo-otp.com/donate/" target="_blank" rel="noopener nofollow">Donate</a> <a href="https://raivo-otp.com/changelog/" target="_blank" rel="noopener nofollow">Changelog</a> <a href="https://raivo-otp.com/branding/" target="_blank" rel="noopener nofollow">Branding</a> <a href="https://raivo-otp.com/license/" target="_blank" rel="noopener nofollow">License</a> <a href="https://raivo-otp.com/privacy-policy/" target="_blank" rel="noopener nofollow">Privacy</a> <a href="https://raivo-otp.com/security-policy/" target="_blank" rel="noopener nofollow">Security</a></div>
                            <p class="footerText">This website and its content is a labor of love by Raivo's <a href="https://github.com/raivo-otp/issuer-icons/graphs/contributors" target="_blank" rel="noopener nofollow">community</a> on Github.<br><small>Copyright &copy; 2023 Mobime. All rights reserved.</small></p>
                        </footer>
                    </div>
                </div>
            </body>
        </html>
        """

    return result.replace('__AMOUNT_OF_ICONS__', str(amount_of_icons))

def main():
    """Run the validation & generation for all vector issuer icons."""

    validate.main()

    search_data = {}
    manifest_data = {}
    output_count = 0

    search_denylist = [
        'text', 'monochrome', 'filled', 'circle', 'rounded', 'square', 'icon', 'symbol', 'alt', 'alternative', 'main', 'second', 'secondary', 'light', 'dark', 'white', 'black'
    ]

    try:
        vectors = glob.glob('./vectors/*/*.svg')
    except:
        vectors = []

    # Generate a PNG for every SVG
    for vector in vectors:
        print('Processing {}...'.format(vector))

        domain = vector[10:].split('/')[0]
        filename = vector[10:].split('/')[1][0:-4]

        output_directory = './dist/{}'.format(domain)
        output_file = '{}/{}.png'.format(output_directory, filename)

        safe_output_directory = re.sub(r'[^a-z0-9-\./]+', '', output_directory)
        safe_output_file = re.sub(r'[^a-z0-9-\./]+', '', output_file)
        safe_input_file = re.sub(r'[^a-z0-9-\./]+', '', vector)

        subprocess.check_call(['mkdir', '-p', safe_output_directory])
        subprocess.check_call(['cairosvg', safe_input_file, '-b', 'rgba(255, 255, 255, 0)', '-f', 'png', '-W', '200', '-H', '200', '-o', safe_output_file])
        output_count = output_count + 1

        # MANIFEST
        if domain not in manifest_data.keys():
            manifest_data[domain] = {
                'domain': domain,
                'additional_search_terms': [],
                'icons': []
            }

        manifest_data[domain]['icons'].append('{}/{}.png'.format(domain, filename))
        
        # SEARCH
        for issuer_part in filename.split('-'):
            if issuer_part in search_denylist:
                continue

            if len(issuer_part) <= 1:
                continue

            if issuer_part not in search_data.keys():
                search_data[issuer_part] = ['{}/{}.png'.format(domain, filename)]
            elif '{}/{}.png'.format(domain, filename) not in search_data[issuer_part]:
                search_data[issuer_part].append('{}/{}.png'.format(domain, filename))

    with open('dist/manifest.json', 'w') as manifest_file:
        json.dump(manifest_data, manifest_file)

    print('Manifest JSON generation done!')

    with open('dist/search.json', 'w') as search_file:
        json.dump(search_data, search_file)

    print('Search JSON generation done!')

    with open('dist/index.html', 'w') as index_file:
        index_file.write(generate_html(manifest_data))

    print('Search JSON generation done!')

    print('Generation finished, everything looks good! Generated {} icons.'.format(output_count))

if __name__ == '__main__':
    """If this file is ran, run the main() function."""

    main()
