/*
 * Radon
 *
 * Copyright 2019 The Radon Authors.
 * Code is licensed under the GPLv3.
 *
 */

package shift

import (
	"context"
	"fmt"
	"strings"
	"sync"
	"testing"
	"time"

	"github.com/radondb/shift/xlog"

	"github.com/fortytw2/leaktest"
	"github.com/stretchr/testify/assert"
)

func assertChecksumEqual(t *testing.T, shift *Shift) {
	// check checksum.
	<-shift.getDoneCh()

	assert.Equal(t, false, readonlyLast)
	assert.Equal(t, 0, throttleLast)
}

func assertChecksumNotEqual(t *testing.T, shift *Shift) {
	// check checksum.
	<-shift.getDoneCh()

	fromConn := shift.fromPool.Get()
	defer shift.fromPool.Put(fromConn)

	sql := fmt.Sprintf("checksum table `%s`.`%s`", shift.cfg.FromDatabase, shift.cfg.FromTable)
	r, err := fromConn.Execute(sql)
	assert.Nil(t, err)
	from, err := r.GetString(0, 1)
	assert.Nil(t, err)

	sql = fmt.Sprintf("checksum table `%s`.`%s`", shift.cfg.ToDatabase, shift.cfg.ToTable)
	r, err = fromConn.Execute(sql)
	assert.Nil(t, err)
	to, err := r.GetString(0, 1)
	assert.Nil(t, err)
	assert.NotEqual(t, from, to)
}

func TestShiftInsert(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		begin = begin + step
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.insert.done")
	}

	// MultiInserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d',B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43'),(%d,%d,'%d',B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i, i+1, i+1, i+1)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

/*
func TestShiftInsertJson(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()
		defer shift.fromPool.Put(fromConn)

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into %s.%s(a,b,c,e) values(%d,%d,'%d', '{\"screen\": \"50 inch\"}')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		begin = begin + step
		log.Debug("test.shift.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}
*/

func TestShiftDelete(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Delete.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("delete from `%s`.`%s` where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		log.Debug("test.shift.delete.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where o = bit
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*3, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where o=B'1'", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.bit.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where p = text
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*4, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where p=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.test.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where q = blob
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*5, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where q=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.blob.done")
		shift.fromPool.Put(fromConn)
	}

	// MultiDeletes.
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("delete from `%s`.`%s`", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.multi.delete.done")
		shift.fromPool.Put(fromConn)
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftDeleteWithoutPK(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, false)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Delete.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("delete from `%s`.`%s` where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		log.Debug("test.shift.delete.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where o = bit
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*3, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where o=B'1'", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.bit.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where p = text
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*4, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where p=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.test.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where q = blob
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*5, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("delete from `%s`.`%s` where q=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.delete.blob.done")
		shift.fromPool.Put(fromConn)
	}

	// MultiDeletes.
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("delete from `%s`.`%s`", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.multi.delete.done")
		shift.fromPool.Put(fromConn)
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftUpdate(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Update.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("update `%s`.`%s` set b=1 where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.update.done")
	}

	// Update where o = bit
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*3, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=1 where o=B'1'", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.update.bit.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where p = text
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*4, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=2 where p=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		log.Debug("test.shift.update.test.done")
		shift.fromPool.Put(fromConn)
	}

	// Delete where q = blob
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*5, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=3 where q=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
	}

	//MultiUpdates
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("update `%s`.`%s` set b=1 where a in (1,3,5)", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.update.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftUpdateWithoutPK(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, false)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Update.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("update `%s`.`%s` set b=1 where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.update.done")
	}

	// Update where o = bit
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*3, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=1 where o=B'1'", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.update.bit.done")
	}

	// Delete where p = text
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*4, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=2 where p=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.update.test.done")
	}

	// Delete where q = blob
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, step*5, step, step)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("update `%s`.`%s` set b=3 where q=0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
	}

	//MultiUpdates
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("update `%s`.`%s` set b=1 where a in (1,3,5)", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.update.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftReplace(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// replace.
	{
		fromConn := shift.fromPool.Get()

		for i := begin - 2; i < begin+step; i++ {
			sql := fmt.Sprintf("replace into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftIntegerUnsigned(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, false)
	defer cleanup()

	fromConn := shift.fromPool.Get()
	defer shift.fromPool.Put(fromConn)

	// Inserts.
	//sql := fmt.Sprintf("insert into `%s`.`%s`(a,f,g,h,i,j,k,l,m,n) values(-2147483648,0,-9223372036854775808,0,-128,0,-32768,0,-8388608,0)", shift.cfg.FromDatabase, shift.cfg.FromTable)
	sql := fmt.Sprintf("insert into `%s`.`%s`(a,f,g,h,i,j,k,l,m,n,o,p,q,r) values(-2147483648,0,-9223372036854775808,0,-128,0,-32768,0,-8388608,0, B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable)
	_, err := fromConn.Execute(sql)
	assert.Nil(t, err)

	sql = fmt.Sprintf("insert into `%s`.`%s`(a,f,g,h,i,j,k,l,m,n,o,p,q,r) values(2147483647,4294967295,9223372036854775807,18446744073709551615,127,255,32767,65535,8388607,16777214, B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable)
	_, err = fromConn.Execute(sql)
	assert.Nil(t, err)

	//sql = fmt.Sprintf("insert into `%s`.`%s`(a,f,g,h,i,j,k,l,m,n) values(2147483646,4294967294,9223372036854775806,18446744073709551614,121,252,32761,65532,8388605,16777213)", shift.cfg.FromDatabase, shift.cfg.FromTable)
	sql = fmt.Sprintf("insert into `%s`.`%s`(a,f,g,h,i,j,k,l,m,n,o,p,q,r) values(2147483646,4294967294,9223372036854775806,18446744073709551614,121,252,32761,65532,8388605,16777213, B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable)
	_, err = fromConn.Execute(sql)
	assert.Nil(t, err)

	// update
	sql = fmt.Sprintf("update `%s`.`%s` set f=4294967291 where h=18446744073709551615", shift.cfg.FromDatabase, shift.cfg.FromTable)
	_, err = fromConn.Execute(sql)
	assert.Nil(t, err)

	// delete
	sql = fmt.Sprintf("delete from `%s`.`%s` where h=18446744073709551614", shift.cfg.FromDatabase, shift.cfg.FromTable)
	_, err = fromConn.Execute(sql)
	assert.Nil(t, err)

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}

}

func TestShiftXACommit(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftXa(log, true)
	defer cleanup()

	fromConn := shift.fromPool.Get()
	defer shift.fromPool.Put(fromConn)

	step := 7
	begin := 0
	// Inserts.
	{
		for i := begin; i < begin+step; i++ {
			// xa start.
			sql := fmt.Sprintf("xa start 'xc%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa commit.
			sql = fmt.Sprintf("xa commit 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Update.
	{
		for i := begin; i < begin+step; i += 2 {
			// xa start.
			sql := fmt.Sprintf("xa start 'xc%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("update `%s`.`%s` set b=1 where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa commit.
			sql = fmt.Sprintf("xa commit 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Delete.
	{
		sql := fmt.Sprintf("use %s", shift.cfg.FromDatabase)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		for i := begin; i < begin+step; i += 2 {
			// xa start.
			sql := fmt.Sprintf("xa start 'xc%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("delete from `%s` where a=%d", shift.cfg.FromTable, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa commit.
			sql = fmt.Sprintf("xa commit 'xc%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

/*
// This test will be fail when shift tables
func TestShiftXARollback(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftXa(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()
		defer shift.fromPool.Put(fromConn)

		for i := begin; i < begin+step; i++ {
			// xa start.
			sql := fmt.Sprintf("xa start 'xr%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("insert into `%s`.`%s`(a,b,c) values(%d,%d,'%d')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			if i%3 == 0 {
				sql = fmt.Sprintf("xa commit 'xr%d'", i)
			} else {
				sql = fmt.Sprintf("xa rollback 'xr%d'", i)
			}
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Update.
	{
		fromConn := shift.fromPool.Get()
		defer shift.fromPool.Put(fromConn)

		for i := begin; i < begin+step; i += 2 {
			// xa start.
			sql := fmt.Sprintf("xa start 'xr%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("update `%s`.`%s` set b=1 where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa.
			if i%2 == 0 {
				sql = fmt.Sprintf("xa commit 'xr%d'", i)
			} else {
				sql = fmt.Sprintf("xa rollback 'xr%d'", i)
			}
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Delete.
	{
		fromConn := shift.fromPool.Get()
		defer shift.fromPool.Put(fromConn)

		for i := begin; i < begin+step; i += 2 {
			// xa start.
			sql := fmt.Sprintf("xa start 'xr%d'", i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)

			sql = fmt.Sprintf("delete from `%s`.`%s` where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa end.
			sql = fmt.Sprintf("xa end 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa prepare.
			sql = fmt.Sprintf("xa prepare 'xr%d'", i)
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)

			// xa.
			if i%2 == 0 {
				sql = fmt.Sprintf("xa commit 'xr%d'", i)
			} else {
				sql = fmt.Sprintf("xa rollback 'xr%d'", i)
			}
			_, err = fromConn.Execute(sql)
			assert.Nil(t, err)
		}
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}
*/

func TestShiftInsertWithDump(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftWithData(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c) values(%d,%d,'%d')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		begin = begin + step
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.insert.done")
	}

	// MultiInserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c) values(%d,%d,'%d'),(%d,%d,'%d')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i, i+1, i+1, i+1)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftChecksumTable(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, false)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	// Delete.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("delete from `%s`.`%s` where a=%d", shift.cfg.FromDatabase, shift.cfg.FromTable, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}

	{
		assertChecksumEqual(t, shift)
	}

	err := shift.ChecksumTable()
	assert.Nil(t, err)
}

func TestShiftMySQLTable(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftMysqlTable(log, false)
	defer cleanup()

	// Delete.
	{
		fromConn := shift.fromPool.Get()

		sql := fmt.Sprintf("delete from `%s`.`%s` where user='%s'", shift.cfg.FromDatabase, shift.cfg.FromTable, "root")
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
		shift.fromPool.Put(fromConn)
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftMySQLTableWithDatas(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftMysqlTableWithData(log, false)
	defer cleanup()

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftWithCleanup(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftWithCleanup(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		begin = begin + step
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.insert.done")
	}

	// MultiInserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

func TestShiftWithRebalance(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.DEBUG))
	shift, cleanup := MockShiftWithRebalance(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		begin = begin + step
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.insert.done")
	}

	// MultiInserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

// Now we doesn`t support ddl
/*
func TestShiftDDLEvent(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftDDL(log, true)
	defer cleanup()
	fromConn := shift.fromPool.Get()
	defer shift.fromPool.Put(fromConn)

	begin := 0
	step := 7
	// Inserts.
	{
		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		log.Debug("test.shift.insert.done")
	}

	// Create database event.
	{
		sql := fmt.Sprintf("create database %s_shift_test", shift.cfg.FromDatabase)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Drop database event.
	{
		sql := fmt.Sprintf("drop database %s_shift_test", shift.cfg.FromDatabase)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Truncate db.table event.
	{
		sql := fmt.Sprintf("truncate table %s.%s", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Alter db.table event.
	{
		sql := fmt.Sprintf("alter table %s.%s add xxx int", shift.cfg.FromDatabase, shift.cfg.FromTable)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Alter table event.
	{
		sql := fmt.Sprintf("use %s", shift.cfg.FromDatabase)
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		sql = fmt.Sprintf("alter table %s engine=MyISAM", shift.cfg.FromTable)
		_, err = fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Create table xx event.
	{
		sql := fmt.Sprintf("create table if not exists xx(a int)")
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Drop table xx event.
	{
		sql := fmt.Sprintf("drop table xx")
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)
	}

	// Checksum check.
	{
		assertChecksumNotEqual(t, shift)
	}
}
*/

func TestShiftWithRadonReadonlyError(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftWithRadonReadonlyError(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}
}

func TestShiftWithRadonShardShiftError(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShiftWithRadonShardRuleError(log, true)
	defer cleanup()

	step := 7
	begin := 0
	// Inserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i++ {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
	}
	// To fix bug:https://github.com/BohuTANG/radon-shift/issues/25
	// before test case run exit and call "defer cleanup()", reserve
	// enough time to make sure shift done.
	time.Sleep(1 * time.Second)
}

func TestShiftStart(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, _ := NewShift(log, mockCfg).(*Shift)
	defer shift.close()

	err := shift.Start()
	assert.Nil(t, err)
}

func TestShiftRadonThrottle(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, cleanup := MockShift(log, true)
	defer cleanup()
	shift.cfg.Behinds = 0

	step := 100
	begin := 0

	// MultiInserts.
	{
		fromConn := shift.fromPool.Get()

		for i := begin; i < begin+step; i += 2 {
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43'),(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", shift.cfg.FromDatabase, shift.cfg.FromTable, i, i, i, i+1, i+1, i+1)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		shift.fromPool.Put(fromConn)
		log.Debug("test.shift.multi.insert.done")
	}

	// Checksum check.
	{
		assertChecksumEqual(t, shift)
	}
}

// Fix bug for issue #4
func TestShiftCanalClose(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, _ := NewShift(log, mockCfg).(*Shift)
	// Replace time ticker every 5s to 1s, 5s is to long for us to test
	shift.behindsTicker = time.NewTicker(time.Duration(1000) * time.Millisecond)
	h := mockHttp(log, restfulPort, mockRadonReadonly, mockRadonShift, mockRadonThrottle)
	defer func() {
		ctx, _ := context.WithTimeout(context.Background(), 2*time.Second)
		h.Shutdown(ctx)
	}()

	fromPool, errfrom := NewPool(log, 4, mockCfg.From, mockCfg.FromUser, mockCfg.FromPassword)
	assert.Nil(t, errfrom)
	toPool, errto := NewPool(log, 4, mockCfg.To, mockCfg.ToUser, mockCfg.ToPassword)
	assert.Nil(t, errto)

	var c = make(chan bool, 1)
	var wg sync.WaitGroup
	wg.Add(1)
	// Keep continuing insert so that canal will be always running and we
	// have enough time to use shift.close() to simulate signal like kill
	go func() {
		begin := 0
		step := 5000

		fromConn := fromPool.Get()
		if fromConn == nil {
			logPanicHandler(log, "test.get.from.toconn.but.return.nil")
		}
		defer fromPool.Put(fromConn)
		toConn := toPool.Get()
		if toConn == nil {
			logPanicHandler(log, "test.get.from.toconn.but.return.nil")
		}
		defer toPool.Put(toConn)

		// Cleanup From table first.
		sql := fmt.Sprintf("drop table if exists `%s`.`%s`", mockCfg.FromDatabase, mockCfg.FromTable)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.drop.from.table.error:%+v", err)
		}

		// Cleanup To table.
		sql = fmt.Sprintf("drop table if exists `%s`.`%s`", mockCfg.ToDatabase, mockCfg.ToTable)
		if _, err := toConn.Execute(sql); err != nil {
			log.Panicf("test.drop.to.table.error:%+v", err)
		}

		// Create database on from.
		sql = fmt.Sprintf("create database if not exists `%s`", mockCfg.FromDatabase)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.prepare.database.error:%+v", err)
		}

		// Create table on from.
		sql = fmt.Sprintf("create table `%s`.`%s`(a int primary key, b int, c varchar(200), d DOUBLE NULL DEFAULT NULL, e json DEFAULT NULL, f INT UNSIGNED DEFAULT NULL, g BIGINT DEFAULT NULL, h BIGINT UNSIGNED DEFAULT NULL, i TINYINT NULL, j TINYINT UNSIGNED DEFAULT NULL, k SMALLINT DEFAULT NULL, l SMALLINT UNSIGNED DEFAULT NULL, m MEDIUMINT DEFAULT NULL, n INT UNSIGNED DEFAULT NULL, o bit(1) default NULL, p text COLLATE utf8_bin, q longblob, r datetime DEFAULT NULL)", mockCfg.FromDatabase, mockCfg.FromTable)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.prepare.from.table.error:%+v", err)
		}

		for i := begin; i < begin+step; i++ {
			select {
			case <-c:
				log.Info("test.gets.signal.done.and.insert.nums:%v", i-1)
				i = 5000
			default:
			}
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", mockCfg.FromDatabase, mockCfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		log.Debug("test.shift.insert.done")
		wg.Done()
	}()

	// Sleep 1s so that tb1 has some rows before we start canal and dump
	time.Sleep(time.Second * 1)

	// Start
	var errstart error
	errstart = shift.Start()
	assert.Nil(t, errstart)

	// Sleep 2s to make sure we have enough time that shift.Start() having executed
	// canal.Run(), then we can use shift.close() to simulate signal like kill
	time.Sleep(time.Second * 2)
	log.Debug("sleep 2s")
	errClose := shift.close()
	assert.Nil(t, errClose)
	c <- true
	wg.Wait()
	assert.False(t, shift.allDone.Get())

	// No matter what exeption happens, the target table should be cleaned up
	// so the table we selected is not exist
	toConn := toPool.Get()
	if toConn == nil {
		logPanicHandler(log, "test.get.from.toconn.but.return.nil")
	}
	defer toPool.Put(toConn)
	sql := fmt.Sprintf("select count(*) from `%s`.`%s`", mockCfg.ToDatabase, mockCfg.ToTable)
	_, err := toConn.Execute(sql)
	errStr := fmt.Sprintf("%s", err)
	errWant := fmt.Sprintf("ERROR 1146 (42S02): Table '%s.%s' doesn't exist", mockCfg.ToDatabase, mockCfg.ToTable)
	b := strings.EqualFold(errStr, errWant)
	assert.True(t, b)
}

// Fix bug for issue #14
func TestShiftParseBOM(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))

	fromPool, errfrom := NewPool(log, 4, mockCfg.From, mockCfg.FromUser, mockCfg.FromPassword)
	assert.Nil(t, errfrom)
	toPool, errto := NewPool(log, 4, mockCfg.To, mockCfg.ToUser, mockCfg.ToPassword)
	assert.Nil(t, errto)

	fromConn := fromPool.Get()
	toConn := toPool.Get()
	// Clean before exit
	defer func() {
		sql := `DROP DATABASE IF EXISTS DTtest`
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.drop.databse.error:%+v", err)
		}
		fromPool.Put(fromConn)
		toPool.Put(toConn)
		fromPool.Close()
		toPool.Close()
	}()

	// Drop test DTtest DB
	sql := `DROP DATABASE IF EXISTS DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.drop.from.databse.error:%+v", err)
	}
	if _, err := toConn.Execute(sql); err != nil {
		log.Panicf("test.drop.from.databse.error:%+v", err)
	}

	// Create test DTtest DB
	sql = `CREATE DATABASE DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.create.databse.error:%+v", err)
	}
	sql = `USE DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.use.databse.error:%+v", err)
	}
	// Create test table
	sql = `CREATE TABLE normal(
            id BIGINT(64) UNSIGNED  NOT NULL AUTO_INCREMENT,
            str VARCHAR(256),
            f FLOAT,
            d DOUBLE,
            de DECIMAL(10,2),
            i INT,
            bi BIGINT,
            e enum ("e1", "e2"),
            b BIT(8),
            y YEAR,
            da DATE,
            ts TIMESTAMP,
            dt DATETIME,
            tm TIME,
            t TEXT,
            bb BLOB,
            se SET('a', 'b', 'c'),
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8`
	if _, err := fromConn.Execute(sql); err == nil {
		fromConn.Execute(`INSERT INTO normal(str, f, i, e, b, y, da, ts, dt, tm, de, t, bb, se)
        VALUES ("3", -3.14, 10, "e1", 0b0011, 1985,
        "2012-05-07", "2012-05-07 14:01:01", "2012-05-07 14:01:01",
        "14:01:01", -45363.64, "abc", "12345", "a,b")`)
	}

	{
		// If MySQL supports JSON, it must supports GEOMETRY.
		sql = `CREATE TABLE test_geo (g GEOMETRY)`
		_, err := fromConn.Execute(sql)
		assert.Nil(t, err)

		tbls := []string{
			`INSERT INTO test_geo VALUES (POINT(1, 1))`,
			`INSERT INTO test_geo VALUES (LINESTRING(POINT(0,0), POINT(1,1), POINT(2,2)))`,
			// TODO: add more geometry tests
		}

		for _, query := range tbls {
			fromConn.Execute(query)
		}
	}

	// Must allow zero time.
	fromConn.Execute(`SET sql_mode=''`)
	str := `CREATE TABLE test_parse_time (
            a1 DATETIME,
            a2 DATETIME(3),
            a3 DATETIME(6),
            b1 TIMESTAMP)`
	if _, err := fromConn.Execute(str); err != nil {
		logPanicHandler(log, "execute.create.table.[%+v].fail.error[%+v]", str, err)
	}

	fromConn.Execute(`INSERT INTO test_parse_time VALUES
        ("2014-09-08 17:51:04.123456", "2014-09-08 17:51:04.123456", "2014-09-08 17:51:04.123456",
        "2014-09-08 17:51:04.123456","2014-09-08 17:51:04.123456","2014-09-08 17:51:04.123456"),
        ("0000-00-00 00:00:00.000000", "0000-00-00 00:00:00.000000", "0000-00-00 00:00:00.000000",
        "0000-00-00 00:00:00.000000", "0000-00-00 00:00:00.000000", "0000-00-00 00:00:00.000000"),
        ("2014-09-08 17:51:04.000456", "2014-09-08 17:51:04.000456", "2014-09-08 17:51:04.000456",
        "2014-09-08 17:51:04.000456","2014-09-08 17:51:04.000456","2014-09-08 17:51:04.000456")`)

	// shift table normal
	{
		shift, _ := NewShift(log, mockCfg).(*Shift)
		h := mockHttp(log, restfulPort, mockRadonReadonly, mockRadonShift, mockRadonThrottle)

		shift.cfg.FromDatabase = "DTtest"
		shift.cfg.ToDatabase = "DTtest"
		shift.cfg.FromTable = "normal"
		shift.cfg.ToTable = "normal"
		err := shift.Start()
		assert.Nil(t, err)
		// wait shift done
		<-shift.getDoneCh()
		assert.True(t, shift.allDone.Get())
		shift.close()

		ctx, _ := context.WithTimeout(context.Background(), 2*time.Second)
		h.Shutdown(ctx)
		log.Info("shift DTtest.normal done.")
	}

	// shift table test_geo
	{
		shift, _ := NewShift(log, mockCfg).(*Shift)
		h := mockHttp(log, restfulPort, mockRadonReadonly, mockRadonShift, mockRadonThrottle)

		shift.cfg.FromDatabase = "DTtest"
		shift.cfg.ToDatabase = "DTtest"
		shift.cfg.FromTable = "test_geo"
		shift.cfg.ToTable = "test_geo"
		err := shift.Start()
		assert.Nil(t, err)
		// wait shift done
		<-shift.getDoneCh()
		assert.True(t, shift.allDone.Get())
		shift.close()

		ctx, _ := context.WithTimeout(context.Background(), 2*time.Second)
		h.Shutdown(ctx)
		log.Info("shift DTtest.test_geo done.")
	}

	// shift table test_parse_time
	{
		shift, _ := NewShift(log, mockCfg).(*Shift)
		h := mockHttp(log, restfulPort, mockRadonReadonly, mockRadonShift, mockRadonThrottle)

		shift.cfg.FromDatabase = "DTtest"
		shift.cfg.ToDatabase = "DTtest"
		shift.cfg.FromTable = "test_parse_time"
		shift.cfg.ToTable = "test_parse_time"
		err := shift.Start()
		assert.Nil(t, err)
		// wait shift done
		<-shift.getDoneCh()
		assert.True(t, shift.allDone.Get())
		shift.close()

		ctx, _ := context.WithTimeout(context.Background(), 2*time.Second)
		h.Shutdown(ctx)
		log.Info("shift DTtest.test_parse_time done.")
	}
}

// This test is also suit for issue https://github.com/radondb/radon/issues/556
func TestDataRaceOnCanalStatus(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, _ := NewShift(log, mockCfg).(*Shift)

	// The detection is opportunistically, but we want exit normally
	// on run out of a million times.
	var wg1 sync.WaitGroup
	wg1.Add(1)
	go func() {
		defer wg1.Done()
		for i := 0; i < 1e6; i++ {
			_ = shift.getCanalStatus()
		}
	}()
	var wg2 sync.WaitGroup
	wg2.Add(1)
	go func() {
		defer wg2.Done()
		for i := 0; i < 1e6; i++ {
			shift.setCanalStatus(true)
		}
	}()
	wg1.Wait()
	wg2.Wait()

	log.Info("data race not happend")
}

// for issue #40
func TestFatalExit(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))
	shift, _ := NewShift(log, mockCfg).(*Shift)

	// The detection is opportunistically, but we want exit normally
	// on run out of a million times.
	var wg1 sync.WaitGroup
	wg1.Add(1)
	shift.setCanalStatus(true)
	shift.allDone.Set(false)
	go func() {
		defer wg1.Done()
		for {
			if shift.allDone.Get() {
				shift.setCanalStatus(false)
				break
			} else {
				shift.setCanalStatus(true)
			}
		}
	}()
	var wg2 sync.WaitGroup
	wg2.Add(1)
	go func() {
		defer wg2.Done()
		for i := 0; i < 1e7; i++ {
			if shift.allDone.Get() {
				break
			}
			if shift.getCanalStatus() {
				if i == 1e6 {
					shift.allDone.Set(true)
				}
			} else {
				// This should be never happen
				panic(0)
			}
		}
	}()
	wg1.Wait()
	wg2.Wait()

	log.Info("panic not happend")
}

// For issue #38
func TestSupportShiftToRadonDB(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))

	fromPool, errfrom := NewPool(log, 4, mockRadonDBCfg.From, mockRadonDBCfg.FromUser, mockRadonDBCfg.FromPassword)
	assert.Nil(t, errfrom)
	toPool, errto := NewPool(log, 4, mockRadonDBCfg.To, mockRadonDBCfg.ToUser, mockRadonDBCfg.ToPassword)
	assert.Nil(t, errto)

	fromConn := fromPool.Get()
	toConn := toPool.Get()
	// Clean before exit
	defer func() {
		sql := `DROP DATABASE IF EXISTS DTtest`
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.drop.databse.error:%+v", err)
		}
		fromPool.Put(fromConn)
		toPool.Put(toConn)
		fromPool.Close()
		toPool.Close()
	}()

	// Drop test DTtest DB
	sql := `DROP DATABASE IF EXISTS DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.drop.from.databse.error:%+v", err)
	}
	if _, err := toConn.Execute(sql); err != nil {
		log.Panicf("test.drop.from.databse.error:%+v", err)
	}

	// Create test DTtest DB
	sql = `CREATE DATABASE DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.create.databse.error:%+v", err)
	}
	sql = `USE DTtest`
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.use.databse.error:%+v", err)
	}
	// Create test table
	sql = `CREATE TABLE normal(
            id BIGINT(64) UNSIGNED  NOT NULL AUTO_INCREMENT,
            str VARCHAR(256),
            f FLOAT,
            d DOUBLE,
            i INT,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8`
	if _, err := fromConn.Execute(sql); err == nil {
		fromConn.Execute(`INSERT INTO normal(str, f, d, i)
        VALUES ("3", -3.14, 3.14, 10)`)
	}

	// shift table normal
	{
		shift, _ := NewShift(log, mockRadonDBCfg).(*Shift)
		shift.cfg.FromDatabase = "DTtest"
		shift.cfg.ToDatabase = "DTtest"
		shift.cfg.FromTable = "normal"
		shift.cfg.ToTable = "normal"
		err := shift.Start()
		assert.Nil(t, err)
		// wait shift done
		<-shift.getDoneCh()
		assert.True(t, shift.allDone.Get())
		shift.close()

		log.Info("shift to radondb done.")
	}
}

// Fix bug for issue #560
func TestShiftColOrTblNameWithoutQuote(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.ERROR))

	fromPool, errfrom := NewPool(log, 4, mockCfg.From, mockCfg.FromUser, mockCfg.FromPassword)
	assert.Nil(t, errfrom)
	fromConn := fromPool.Get()
	// Clean before exit
	defer func() {
		sql := "DROP DATABASE IF EXISTS `a-b_db`"
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.drop.databse.error:%+v", err)
		}
		fromPool.Put(fromConn)
		fromPool.Close()
	}()

	// Drop test DB
	sql := "DROP DATABASE IF EXISTS `a-b_db`"
	if _, err := fromConn.Execute(sql); err != nil {
		log.Panicf("test.drop.from.databse.error:%+v", err)
	}

	// Create DB error
	_, err := fromConn.Execute("CREATE DATABASE a-b_db")
	assert.NotNil(t, err)

	// Create DB success
	_, err = fromConn.Execute("CREATE DATABASE `a-b_db`")
	assert.Nil(t, err)

	// Create table error
	_, err = fromConn.Execute("CREATE TABLE `a-b_db`.a-b_tbl(`char` int, `key` bigint) ENGINE=InnoDB DEFAULT CHARSET=utf8")
	assert.NotNil(t, err)

	// Create table success
	_, err = fromConn.Execute("CREATE TABLE `a-b_db`.`a-b_tbl`(`char` int, `key` bigint) ENGINE=InnoDB DEFAULT CHARSET=utf8")
	assert.Nil(t, err)

	_, err = fromConn.Execute("INSERT INTO `a-b_db`.`a-b_tbl` VALUES (3, -3.14)")
	assert.Nil(t, err)

	// Update col fail
	_, err = fromConn.Execute("UPDATE `a-b_db`.`a-b_tbl` set char=1 where char=3")
	assert.NotNil(t, err)
	// Update col success
	_, err = fromConn.Execute("UPDATE `a-b_db`.`a-b_tbl` set `char`=1 where `char`=3")
	assert.Nil(t, err)

	// DELETE fail
	_, err = fromConn.Execute("DELETE FROM `a-b_db`.`a-b_tbl` where char=3")
	assert.NotNil(t, err)

	// DELETE success
	_, err = fromConn.Execute("DELETE FROM `a-b_db`.`a-b_tbl` where `char`=3")
	assert.Nil(t, err)
}

func TestShiftSetStop(t *testing.T) {
	log := xlog.NewStdLog(xlog.Level(xlog.DEBUG))
	shift, _ := NewShift(log, mockCfg).(*Shift)
	// Replace time ticker every 5s to 1s, 5s is to long for us to test
	shift.behindsTicker = time.NewTicker(time.Duration(1000) * time.Millisecond)
	h := mockHttp(log, restfulPort, mockRadonReadonly, mockRadonShift, mockRadonThrottle)
	defer func() {
		ctx, _ := context.WithTimeout(context.Background(), 2*time.Second)
		h.Shutdown(ctx)
	}()

	fromPool, errfrom := NewPool(log, 4, mockCfg.From, mockCfg.FromUser, mockCfg.FromPassword)
	assert.Nil(t, errfrom)
	toPool, errto := NewPool(log, 4, mockCfg.To, mockCfg.ToUser, mockCfg.ToPassword)
	assert.Nil(t, errto)

	var c = make(chan bool, 1)
	var wg sync.WaitGroup
	wg.Add(1)
	// Keep continuing insert so that canal will be always running and we
	// have enough time to use shift.close() to simulate signal like kill
	go func() {
		begin := 0
		step := 50000

		fromConn := fromPool.Get()
		if fromConn == nil {
			logPanicHandler(log, "test.get.from.toconn.but.return.nil")
		}
		defer fromPool.Put(fromConn)
		toConn := toPool.Get()
		if toConn == nil {
			logPanicHandler(log, "test.get.from.toconn.but.return.nil")
		}
		defer toPool.Put(toConn)

		// Cleanup From table first.
		sql := fmt.Sprintf("drop table if exists `%s`.`%s`", mockCfg.FromDatabase, mockCfg.FromTable)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.drop.from.table.error:%+v", err)
		}

		// Cleanup To table.
		sql = fmt.Sprintf("drop table if exists `%s`.`%s`", mockCfg.ToDatabase, mockCfg.ToTable)
		if _, err := toConn.Execute(sql); err != nil {
			log.Panicf("test.drop.to.table.error:%+v", err)
		}

		// Create database on from.
		sql = fmt.Sprintf("create database if not exists `%s`", mockCfg.FromDatabase)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.prepare.database.error:%+v", err)
		}

		// Create table on from.
		sql = fmt.Sprintf("create table `%s`.`%s`(a int primary key, b int, c varchar(200), d DOUBLE NULL DEFAULT NULL, e json DEFAULT NULL, f INT UNSIGNED DEFAULT NULL, g BIGINT DEFAULT NULL, h BIGINT UNSIGNED DEFAULT NULL, i TINYINT NULL, j TINYINT UNSIGNED DEFAULT NULL, k SMALLINT DEFAULT NULL, l SMALLINT UNSIGNED DEFAULT NULL, m MEDIUMINT DEFAULT NULL, n INT UNSIGNED DEFAULT NULL, o bit(1) default NULL, p text COLLATE utf8_bin, q longblob, r datetime DEFAULT NULL)", mockCfg.FromDatabase, mockCfg.FromTable)
		if _, err := fromConn.Execute(sql); err != nil {
			log.Panicf("test.prepare.from.table.error:%+v", err)
		}

		for i := begin; i < begin+step; i++ {
			select {
			case <-c:
				log.Info("test.gets.signal.done.and.insert.nums:%v", i-1)
				i = 50000
			default:
			}
			sql := fmt.Sprintf("insert into `%s`.`%s`(a,b,c,o,p,q,r) values(%d,%d,'%d', B'1', 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, 0x6B313134363020666638303831383135646534373733633031356465343762353138653030303020E799BDE4BAAC2031302E3131362E32352E3137322C31312E312E31302E313420737061636520636F6E66696775726174696F6E207570646174656420737061636573207479706520676C6F62616C207374617475732063757272656E74206E616D65206B65792073706320686F6D65207061676520706167653A20762E31202833323831383229, '2019-4-19 18:03:43')", mockCfg.FromDatabase, mockCfg.FromTable, i, i, i)
			_, err := fromConn.Execute(sql)
			assert.Nil(t, err)
		}
		log.Debug("test.shift.insert.done")
		wg.Done()
	}()

	// Sleep 1s so that tb1 has some rows before we start canal and dump
	time.Sleep(time.Second * 1)

	// Start
	var errstart error
	errstart = shift.Start()
	assert.Nil(t, errstart)

	// Wait
	errorch := make(chan error)
	var wg2 sync.WaitGroup
	wg2.Add(1)
	go func() {
		defer wg2.Done()
		err := shift.WaitFinish()
		errorch <- err
	}()

	// get stop signal and exit with error
	shift.SetStopSignal()
	// send true to stop writing
	c <- true
	err := <-errorch
	assert.NotNil(t, err)
}

// Test Leaked Goroutine about Ticker.
type ShiftDemon struct {
	behindsTicker *time.Ticker
	done       chan bool
}

func (sd *ShiftDemon) behindsCheckStart() error{
	fmt.Println("ticker start")
	go func(s *ShiftDemon)  {
		// the older leaked goroutine
		//for range s.behindsTicker.C {
		//	fmt.Println("ticker is on.")
		//}
		for {
			select {
			case <-s.behindsTicker.C:
				fmt.Println("ticker is on.")
			case <-s.done:
				return
			}
			fmt.Println("ticker is out for.")
		}
	}(sd)
	return nil
}

func (shift *ShiftDemon) close() {
	shift.behindsTicker.Stop()
	close(shift.done)
}

func NewShiftDemon() *ShiftDemon {
	return &ShiftDemon{
		behindsTicker: time.NewTicker(1000 * time.Millisecond),
		done:          make(chan bool),
	}
}

func TestLeakedGoroutineTicker(t *testing.T) {
	defer leaktest.Check(t)()
	new := NewShiftDemon()
	new.behindsCheckStart()
	new.close()
}
