/**
 * axios封装
 * 请求拦截、响应拦截、错误统一处理
 */
import axios from 'axios'
import router from '../router'
import store from '../store/index'
import { Toast } from 'vant'
// import JSEncrypt from 'encryptlong'
//
// const publicKey = `-----BEGIN PUBLIC KEY-----
// MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCia5R96VU1dRx1JXjmhE0pmltR
// 0nR2P1YUpMqtpuW5unCoqRD2J2RYxzS2/JqNXDRsj1u/bswl5LjOFNZEVoNeCAzw
// UGX11EkSD1VOCjtYvoytn2PWrEeZoSIGehkeXXukCJrBXU2PfZhqgLGx6JO3oBSf
// 2zx2S1agMTd+OA50dwIDAQAB
// -----END PUBLIC KEY-----`

/**
 * 提示函数
 * 禁止点击蒙层、显示一秒后关闭
 */
const tip = msg => {
  Toast({
    message: msg,
    duration: 1000,
    forbidClick: true
  })
}

/**
 * 跳转登录页
 * 携带当前页面路由，以期在登录页面完成登录后返回当前页面
 */
const toLogin = () => {
  router.replace({
    path: '/login',
    query: {
      redirect: router.currentRoute.fullPath
    }
  })
}

// 参数加密
// const _encrypt = (data) => {
//   const _JSEncrypt = new JSEncrypt()
//   // 初始化公钥
//   _JSEncrypt.setPublicKey(publicKey)
//   // 通过 公钥 加密
//   return _JSEncrypt.encryptLong(data)
// }

/**
 * 请求失败后的错误统一处理
 * @param {Number} status 请求失败的状态码
 */
const errorHandle = (status, other) => {
  // 状态码判断
  switch (status) {
    // 401: 未登录状态，跳转登录页
    case 401:
      toLogin()
      break
    // 403 token过期
    // 清除token并跳转登录页
    case 403:
      tip('登录过期，请重新登录')
      localStorage.removeItem('token')
      store.commit('loginSuccess', null)
      setTimeout(() => {
        toLogin()
      }, 1000)
      break
    // 404请求不存在
    case 404:
      tip('请求的资源不存在')
      break
    case 400:
      tip('非法请求')
      break
    default:
      // console.log(other)
  }
}

// 创建axios实例
const instance = axios.create({ baseURL: '/api/v1', timeout: 1000 * 60 })
// 设置post请求头
instance.defaults.headers.post['Content-Type'] = 'application/x-www-form-urlencoded'
/**
 * 请求拦截器
 * 每次请求前，如果存在token则在请求头中携带token
 */
instance.interceptors.request.use(
  config => {
    const token = store.state.token
    token && (config.headers.authorization = token)
    // if (!config._type) {
    //   // console.log(JSON.stringify({ ...config.data, timestamp: parseInt(Date.now() / 1000) }))
    //   config.data = { datas: _encrypt(JSON.stringify({ ...config.data, timestamp: parseInt(Date.now() / 1000) })) }
    // }
    if (!config._type) {
      config.data = { ...config.data, timestamp: parseInt(Date.now() / 1000) }
    }
    return config
  },
  error => Promise.error(error))

// 响应拦截器
instance.interceptors.response.use(
  // 请求成功
  res => {
    if (res.status === 200) {
      const _authorization = res.headers.authorization
      const { code, msg } = res.data
      if (code === 400) {
        Toast.fail({
          message: msg,
          forbidClick: true
        })
      }
      if (_authorization) {
        store.commit('setToken', _authorization)
      }
      if (code === 500) {
        toLogin()
      }
      return Promise.resolve(res.data)
    } else {
      return Promise.reject(res)
    }
  },
  // 请求失败
  error => {
    const { response } = error
    if (response) {
      // 请求已发出，但是不在2xx的范围
      errorHandle(response.data.code, response.data.message)
      return Promise.reject(response)
    } else {
      // 处理断网的情况
      // eg:请求超时或断网时，更新state的network状态
      // network状态在app.vue中控制着一个全局的断网提示组件的显示隐藏
      // 关于断网组件中的刷新重新获取数据，会在断网组件中说明
      if (!window.navigator.onLine) {
        store.commit('changeNetwork', false)
      } else {
        return Promise.reject(error)
      }
    }
  })

export default instance
