#!/bin/bash
# Copyright (c) Meta Platforms, Inc. and affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# run this script from the project root using `./scripts/build_website.sh`

usage() {
  echo "Usage: $0 [-b]"
  echo ""
  echo "Build Opacus documentation."
  echo ""
  echo "  -b   Build static version of documentation (otherwise start server)"
  echo ""
  exit 1
}

BUILD_STATIC=false

while getopts 'hb' flag; do
  case "${flag}" in
    h)
      usage
      ;;
    b)
      BUILD_STATIC=true
      ;;
    *)
      usage
      ;;
  esac
done

echo "-----------------------------------"
echo "Generating API reference via Sphinx"
echo "-----------------------------------"
cd sphinx || exit
make html
cd .. || exit

echo "-----------------------------------"
echo "Building Opacus Docusaurus site"
echo "-----------------------------------"
yarn || exit

# run script to parse html generated by sphinx
echo "--------------------------------------------"
echo "Parsing Sphinx docs and moving to Docusaurus"
echo "--------------------------------------------"
cd ..
mkdir -p "website/pages/api/"

cwd=$(pwd)
python website/scripts/parse_sphinx.py -i "${cwd}/website/sphinx/build/html/" -o "${cwd}/website/pages/api/" || exit 1

SPHINX_JS_DIR='website/sphinx/build/html/_static/'
DOCUSAURUS_JS_DIR='website/static/js/'

mkdir -p $DOCUSAURUS_JS_DIR

# move JS files from /sphinx/build/html/_static/*:
cp "${SPHINX_JS_DIR}documentation_options.js" "${DOCUSAURUS_JS_DIR}documentation_options.js"
cp "${SPHINX_JS_DIR}jquery.js" "${DOCUSAURUS_JS_DIR}jquery.js"
cp "${SPHINX_JS_DIR}underscore.js" "${DOCUSAURUS_JS_DIR}underscore.js"
cp "${SPHINX_JS_DIR}doctools.js" "${DOCUSAURUS_JS_DIR}doctools.js"
cp "${SPHINX_JS_DIR}language_data.js" "${DOCUSAURUS_JS_DIR}language_data.js"
cp "${SPHINX_JS_DIR}searchtools.js" "${DOCUSAURUS_JS_DIR}searchtools.js"

# searchindex.js is not static util
cp "website/sphinx/build/html/searchindex.js" "${DOCUSAURUS_JS_DIR}searchindex.js"

# copy module sources
cp -r "website/sphinx/build/html/_sources/" "website/static/_sphinx-sources/"

echo "-----------------------------------"
echo "Generating tutorials"
echo "-----------------------------------"
mkdir -p "website/static/files"
mkdir "website/tutorials"
python website/scripts/parse_tutorials.py -w "${cwd}" || exit 1

cd website || exit

if [[ $BUILD_STATIC == true ]]; then
  echo "-----------------------------------"
  echo "Building static site"
  echo "-----------------------------------"
  yarn build
else
  echo "-----------------------------------"
  echo "Starting local server"
  echo "-----------------------------------"
  yarn start
fi
