#include "SegmentedBlock.h"
#include "core/util/prelude.h"

namespace torch_tensorrt {
namespace core {
namespace partitioning {

SegmentedBlock::SegmentedBlock(BlockID id, SegmentedBlockTarget blk_target, const std::vector<torch::jit::Node*>& nodes)
    : id_(id), target_(blk_target), g_(std::make_shared<torch::jit::Graph>()) {
  for (auto& node : nodes) {
    nodes_.push_back(node);
    appendNode(node);
  }
}

SegmentedBlock::SegmentedBlock(SegmentedBlockTarget blk_target, const std::vector<torch::jit::Node*>& nodes)
    : target_(blk_target), g_(std::make_shared<torch::jit::Graph>()) {
  for (auto& node : nodes) {
    nodes_.push_back(node);
    appendNode(node);
  }
}

void SegmentedBlock::registerOutput(torch::jit::Value* raw_output) {
  outputs_.push_back(raw_output);
  g_->registerOutput(old_to_new_[raw_output]);
}

void SegmentedBlock::eraseInput(size_t i) {
  inputs_.erase(inputs_.begin() + i);
  g_->eraseInput(i);
}

void SegmentedBlock::eraseOutput(size_t i) {
  outputs_.erase(outputs_.begin() + i);
  g_->eraseOutput(i);
}

torch::jit::Value* SegmentedBlock::getOrAddInputForValue(torch::jit::Value* old_value) {
  if (old_to_new_.count(old_value) == 0) {
    auto node = old_value->node();

    if (node->kind() == torch::jit::prim::Constant) {
      auto new_const = g_->createClone(node, {nullptr});
      g_->block()->prependNode(new_const);
      old_to_new_[old_value] = new_const->output();
      return new_const->output();
    }
    auto new_value = g_->block()->addInput();
    // every time when we addInput, we push back the corresponding lowering graph torch::jit::Value to our raw_inputs
    inputs_.push_back(old_value);
    old_to_new_[old_value] = new_value;
    new_value->copyMetadata(old_value);
    return new_value;
  } else {
    return old_to_new_[old_value];
  }
}

std::vector<ir::Input> SegmentedBlock::construct_inputs_spec() const {
  std::vector<ir::Input> inputs;
  if (min_shapes_.size() == opt_shapes_.size() && opt_shapes_.size() == max_shapes_.size()) {
    for (uint64_t i = 0; i < opt_shapes_.size(); i++) {
      auto in = ir::Input(min_shapes_[i], opt_shapes_[i], max_shapes_[i]);
      in.dtype = in_types_[i];
      inputs.push_back(in);
    }
  } else {
    for (uint64_t i = 0; i < opt_shapes_.size(); i++) {
      auto in = ir::Input(opt_shapes_[i]);
      in.dtype = in_types_[i];
      inputs.push_back(in);
    }
  }
  return inputs;
}

torch::jit::Node* SegmentedBlock::cloneNode(torch::jit::Node* node) {
  auto* block = g_->block();
  auto env = [&](torch::jit::Value* v) { return getOrAddInputForValue(v); };

  // create node for current graph by using the metadata in node and input Values in env
  auto new_node = block->appendNode(g_->createClone(node, env));
  for (size_t i = 0; i < node->outputs().size(); ++i) {
    auto oo = node->outputs()[i];
    auto no = new_node->outputs()[i];
    old_to_new_[oo] = no;
  }
  return new_node;
}

std::ostream& operator<<(std::ostream& os, const SegmentedBlock& b) {
  os << "Segment Block @" << b.id_ << ":" << std::endl;
  os << "    Target: " << b.target_ << std::endl;
  os << "    Graph: " << *b.g_ << std::endl;
  return os;
}

std::ostream& operator<<(std::ostream& os, const SegmentedBlock::SegmentedBlockTarget& t) {
  os << SegmentedBlock::target_to_str(t) << std::endl;
  return os;
}

} // namespace partitioning
} // namespace core
} // namespace torch_tensorrt
