import { Component } from 'preact';
import cx from '../../lib/cx';
import { InvertedLogo } from '../logo';
import Search from './search';
import style from './style.module.css';
import config from '../../config.json';
import { useCallback, useEffect } from 'preact/hooks';
import ReleaseLink from './gh-version';
import Corner from './corner';
import { useOverlayToggle } from '../../lib/toggle-overlay';
import { useLocation } from 'preact-iso';
import { useLanguage } from '../../lib/i18n';
import { prefetchContent } from '../../lib/use-resource';

const LINK_FLAIR = {
	logo: InvertedLogo
};

export default function Header() {
	const { url } = useLocation();
	const [open, setOpen] = useOverlayToggle();
	const toggle = useCallback(() => setOpen(!open), [open]);

	useEffect(() => {
		if (open) setOpen(false);
	}, [url]);

	return (
		<header class={cx(style.header, open && style.open)}>
			<div class={style.banner}>
				<a href="https://www.stopputin.net/">
					We stand with Ukraine. <b>Show your support</b> 🇺🇦
				</a>
			</div>
			<div class={style.outer}>
				<div class={style.inner}>
					<Nav class={style.nav} routes={config.nav} current={url} />
					<Search />
					<div class={style.social}>
						<ReleaseLink class={cx(style.socialItem, style.release)} />
						<SocialIcon
							label="Browse the code on GitHub"
							href="https://github.com/preactjs/preact"
							viewbox="0 0 24 24"
							id="github"
						/>
						<SocialIcon
							label="Follow us on Twitter"
							href="https://twitter.com/preactjs"
							viewbox="0 0 34 27.646"
							id="twitter"
						/>
						<SocialIcon
							label="Follow us on Bluesky"
							href="https://bsky.app/profile/preactjs.com"
							viewbox="0 0 568 501"
							id="bluesky"
						/>
						<SocialIcon
							label="Chat with us on Slack"
							href="http://chat.preactjs.com/"
							viewbox="0 0 512 512"
							id="slack"
						/>
					</div>
					<div class={style.translation}>
						<NavMenu language />
					</div>
					<HamburgerMenu open={open} onClick={toggle} />
				</div>
			</div>
			<Corner />
		</header>
	);
}

const SocialIcon = ({ label, href, viewbox, id }) => (
	<a
		class={style.socialItem}
		aria-label={label}
		href={href}
		target="_blank"
		rel="noopener noreferrer"
	>
		<svg aria-hidden viewBox={viewbox}>
			<use href={`/icons.svg#${id}`} />
		</svg>
	</a>
);

const HamburgerMenu = ({ open, ...props }) => (
	<div class={style.hamburger} open={open} {...props}>
		<div class={style.hb1} />
		<div class={style.hb2} />
		<div class={style.hb3} />
	</div>
);

// nested nav renderer
const Nav = ({ routes, current, ...props }) => (
	<nav {...props}>
		{routes.map(route =>
			route.routes ? (
				<NavMenu
					to={route}
					current={current}
					data-route={getRouteIdent(route)}
				/>
			) : (
				<NavLink
					to={route}
					class={cx(
						route.class,
						(pathMatchesRoute(current, route) ||
							(route.content === 'guide' && /^\/guide\//.test(current)) ||
							(route.content === 'blog' && /^\/blog\//.test(current))) &&
							style.current
					)}
				/>
			)
		)}
	</nav>
);

// nav items are really the only complex bit for menuing, since they handle click events.
class NavMenu extends Component {
	state = { open: false };

	close = () => (this.setState({ open: false }), false);

	toggle = () => (this.setState({ open: !this.state.open }), false);

	handleClickOutside = ({ target }) => {
		if (this.state.open) {
			do {
				if (target === this.base) return;
			} while ((target = target.parentNode));
			this.close();
		}
	};

	componentDidMount() {
		addEventListener('click', this.handleClickOutside);
	}

	componentWillUnmount() {
		removeEventListener('click', this.handleClickOutside);
	}

	componentDidUpdate({ current }) {
		if (current !== this.props.current && this.state.open) {
			this.close();
		}
	}

	render({ to, current, language, ...props }, { open }) {
		return (
			<div {...props} data-open={open} class={style.navGroup}>
				{language ? (
					<LanguageSelectorMenu
						isOpen={open}
						toggle={this.toggle}
						close={this.close}
						{...props}
					/>
				) : (
					<>
						<NavLink
							to={to}
							onClick={this.toggle}
							aria-haspopup
							isOpen={open}
						/>
						<Nav
							routes={to.routes}
							current={current}
							aria-label="submenu"
							aria-hidden={'' + !open}
						/>
					</>
				)}
			</div>
		);
	}
}

// depending on the type of nav link, use <a>
const NavLink = ({ to, isOpen, route, ...props }) => {
	const location = useLocation();
	const [lang] = useLanguage();
	let Flair = to.flair && LINK_FLAIR[to.flair];

	if (to.skipHeader) return;

	if (!to.path) {
		return (
			<button
				{...props}
				aria-haspopup="true"
				aria-expanded={isOpen}
				data-route={route}
			>
				{getRouteName(to, lang)}
			</button>
		);
	}

	function BrandingRedirect(e) {
		e.preventDefault();
		location.route('/branding');
	}

	const href = to.href || to.path;
	const prefetchHref = href == '/tutorial'
		? '/tutorial/index'
		: href == '/'
			? '/index'
			: href;
	const homeProps = to.href == '/' || to.path == '/'
		? { onContextMenu: BrandingRedirect, 'aria-label': 'Home' }
		: {};

	return (
		<a
			href={href}
			onMouseOver={() => prefetchContent(prefetchHref)}
			{...props}
			data-route={route}
			{...homeProps}
		>
			{Flair && <Flair title="Preact Logo" />}
			{getRouteName(to, lang)}
		</a>
	);
};

const LanguageSelectorMenu = ({ isOpen, toggle, close, ...props }) => {
	const [lang, setLang] = useLanguage();
	const onClick = useCallback(
		e => {
			setLang(e.target.dataset.value);
			close();
		},
		[setLang]
	);

	return (
		<>
			<button
				{...props}
				onClick={toggle}
				aria-label="Select your language"
				aria-haspopup
				aria-expanded={isOpen}
			>
				<svg aria-hidden viewBox="0 0 24 24">
					<use href="/icons.svg#i18n" />
				</svg>
			</button>
			<nav aria-label="submenu" aria-hidden={!isOpen}>
				{typeof window !== 'undefined' && Object.keys(config.languages).map(id => (
					<span
						class={cx(id == lang && style.current)}
						data-value={id}
						onClick={onClick}
					>
						{config.languages[id]}
					</span>
				))}
			</nav>
		</>
	);
};

export function getRouteName(route, lang) {
	return typeof route.name === 'object'
		? route.name[lang] || route.name.en
		: route.name || route.title;
}

function pathMatchesRoute(path, route) {
	if (!route || !route.path) return false;
	if (path === route.path) return true;
	let segs = path.replace(/(^\/|\/$)/g, '').split('/');
	let psegs = route.path.replace(/(^\/|\/$)/g, '').split('/');
	let len = Math.max(psegs.length, segs.length);
	for (let i = 0; i < len; i++) {
		let p = psegs[i];
		let s = segs[i];
		if (!p || (p[0] !== ':' && s !== p)) return false;
		if (!s) return /[?*]$/g.test(p);
		if (/[*+]$/g.test(p)) return true;
	}
	return true;
}

// get a CSS-addressable identifier for a given route
const getRouteIdent = route =>
	(getRouteName(route, 'en') || route.url)
		.toLowerCase()
		.replace(/[^a-z0-9]/i, '');
