---
prev: /tutorial
next: /tutorial/02-events
solvable: true
---

# Virtual DOM

Возможно, вы слышали, как люди упоминают «Virtual DOM», и задавались вопросом: что делает его «виртуальным»? Чем «виртуальный» DOM отличается от реального DOM, который мы используем при программировании для браузера?

[Virtual DOM](https://doka.guide/tools/react-and-alternatives/#virtual-dom) — это простое описание древовидной структуры с использованием объектов:

```js
let vdom = {
  type: 'p', // элемент <p>
  props: {
    class: 'big', // с классом "big"
    children: [
      'Привет, мир!', // и с текстом "Привет, мир!"
    ],
  },
};
```

Такие библиотеки, как Preact, предоставляют возможность создавать описания, которые затем сравниваются с деревом DOM браузера. В итоге это дерево обновляется в соответствии с описанной структурой виртуального дерева.

Это полезный инструмент, поскольку он позволяет нам составлять пользовательские интерфейсы _декларативно_, а не _императивно_. Вместо того чтобы описывать _как_ обновлять DOM в реакцию на такие вещи, как ввод с клавиатуры или мыши, нам нужно только описать _как_ DOM должен выглядеть после получения входных данных. Это означает, что мы можем неоднократно давать Preact описание древовидных структур, и он будет обновлять DOM браузера для соответствия каждому новому описанию — независимо от его текущей структуры.

В этой главе мы узнаем, как создавать деревья Virtual DOM и как указывать Preact обновлять DOM в соответствии с этими деревьями.

### Создание деревьев Virtual DOM

Существует несколько способов создания таких деревьев:

- `createElement()`: функция, предоставляемая Preact
- [JSX]: HTML-подобный синтаксис, который может быть скомпилирован в JavaScript
- [HTM]: HTML-подобный синтаксис, который можно использовать непосредственно в JavaScript

Начать стоит с самого простого подхода, который заключается в непосредственном вызове функции Preact `createElement()`:

```jsx
import { createElement, render } from 'preact';

let vdom = createElement(
  'p', // элемент <p>
  { class: 'big' }, // с классом "big"
  'Привет, мир!' // и с текстом "Привет, мир!"
);

render(vdom, document.body);
```

Приведённый выше код создает в Virtual DOM «описание» элемента абзаца. Первым аргументом createElement является имя HTML-элемента. Второй аргумент — «реквизит» элемента — объект, содержащий атрибуты (или свойства) для установки на элемент. Любые дополнительные аргументы являются дочерними для элемента, и могут быть строками (например `'Привет, мир!'`) или элементами Virtual DOM от вложенных вызовов `createElement()`.

Последняя строка диктует Preact построить реальное дерево DOM, соответствующее «описанию» Virtual DOM, и вставить это DOM-дерево в `<body>` веб-страницы.

### Добавим JSX!

Мы можем переписать предыдущий пример, используя [JSX], не меняя его функциональность. JSX позволяет нам описывать наш элемент абзаца, используя синтаксис, подобный HTML, что может помочь сохранить читабельность при описании более сложных деревьев. Недостаток JSX в том, что наш код больше не пишется на JavaScript и должен компилироваться с помощью такого инструмента, как [Babel]. Компиляторы преобразуют приведённый ниже пример JSX в точный код `createElement()`, который мы видели в предыдущем примере.

```jsx
import { createElement, render } from 'preact';

let vdom = <p class='big'>Привет, мир!</p>;

render(vdom, document.body);
```

Теперь это гораздо больше похоже на HTML!

И последнее, что следует помнить о JSX: код внутри JSX-элемента (в угловых скобках) — это специальный синтаксис, а не JavaScript. Для использования JavaScript синтаксиса, например, чисел или переменных, сначала нужно «перепрыгнуть» обратно из JSX с помощью `{выражения}` — аналогично полям в шаблоне. В приведённом ниже примере показаны два
выражения: одно для установки `class` в произвольную строку, а другое для вычисления числа.

```jsx
let maybeBig = Math.random() > 0.5 ? 'big' : 'small';

let vdom = <p class={maybeBig}>Привет, {40 + 2}!</p>;
//                  ^---JS---^         ^--JS--^
```

Если мы выполним команду `render(vdom, document.body)`, то на экране появится текст «Привет, 42!».

### Ещё раз с HTM

[HTM] — это альтернатива JSX, использующая стандартные шаблоны с тегами JavaScript, тем самым устраняя необходимость в компиляторе. Если вы ещё не сталкивались с шаблонами тегов, они представляют собой специальный тип литерала String, который может содержать поля `${выражений}`:

```js
let str = `Количество: ${40 + 2} единиц(ы)`; // "Количество: 42 единиц(ы)"
```

В HTM используется `${выражение}` вместо синтаксиса `{выражение}` из JSX, облегчая понимание, какие части вашего кода являются элементами HTM/JSX, а какие части являются обычным JavaScript:

```js
import { html } from 'htm/preact';

let maybeBig = Math.random() > 0.5 ? 'big' : 'small';

let vdom = html`<p class=${maybeBig}>Привет, ${40 + 2}!</p>`;
//                        ^---JS---^          ^--JS--^
```

Все эти примеры приводят к одному и тому же результату: дерево Virtual DOM, которое может быть передано в Preact для создания или обновления существующего DOM-дерева.

---

### Окольный путь: Компоненты

Более подробно о компонентах мы поговорим чуть позже, но пока важно знать, что такие HTML-элементы, как `<p>`, являются лишь одним из _двух_ типов элементов Virtual DOM. Другой тип — компонент. Он представляет собой элемент Virtual DOM, в котором типом является функция, а не строка вида `p`.

Компоненты — это строительные блоки приложений Virtual DOM. Сейчас мы создадим очень простой компонент, переместив наш JSX в функцию:

```jsx
import { createElement, render } from 'preact';

export default function App() {
  return <p class='big'>Привет, мир!</p>;
}

render(<App />, document.getElementById('app'));
```

При передаче компонента в `render()` важно позволить Preact сделать инстанцирование вместо прямого обращения к компоненту:

```jsx
const App = () => <div>foo</div>;

// НЕЛЬЗЯ: Прямой вызов компонентов означает, что они не будут считаться
// VNode и, следовательно, не смогут использовать функциональность, связанную с vnodes.
render(App(), rootElement); // ОШИБКА
render(App, rootElement); // ОШИБКА

// МОЖНО: Передача компонентов с помощью createElement() или JSX позволяет корректно отображать Preact:
render(createElement(App), rootElement); // успешно
render(<App />, rootElement); // успешно
```

## Попробуйте!

В правой части этой страницы вы увидите код из нашего предыдущего примера. Ниже расположено окно с результатом выполнения этого кода. Вы можете отредактировать код и посмотреть, как ваши изменения влияют (или ломают!) на результат в процессе работы.

Чтобы проверить, чему вы научились в этой главе, попробуйте придать тексту больше пикантности! Выделите слово `мир`, обернув его в HTML-теги: `<em>` и `</em>`.

Затем сделайте весь текст <span style="color:purple">фиолетовым</span>, добавив атрибут `style`. Атрибут `style` является специальным и позволяет использовать значение объекта с одним или несколькими CSS-свойствами для установки на элемент. Чтобы передать объект в качестве значения атрибута, необходимо использовать `{выражение}`, например `style={{ color: 'purple' }}`.

<solution>
  <h4>🎉 Поздравляем!</h4>
  <p>Мы заставили вещи появиться на странице. Далее мы сделаем их интерактивными.</p>
</solution>

```js:setup
useResult(function(result) {
  var hasEm = result.output.innerHTML.match(/<em>мир\!?<\/em>/gi);
  var p = result.output.querySelector('p');
  var hasColor = p && p.style && p.style.color === 'purple';
  if (hasEm && hasColor) {
    solutionCtx.setSolved(true);
  }
}, []);
```

```jsx:repl-initial
import { createElement, render } from 'preact';

function App() {
  return (
    <p class="big">Привет, мир!</p>
  );
}

render(<App />, document.getElementById("app"));
```

```jsx:repl-final
import { createElement, render } from 'preact';

function App() {
  return (
    <p class="big" style={{ color: 'purple' }}>
      Привет, <em>мир</em>!
    </p>
  );
}

render(<App />, document.getElementById("app"));
```

[JSX]: https://reactdev.ru/learn/writing-markup-with-jsx/#html-jsx
[HTM]: https://github.com/developit/htm
[Babel]: https://babeljs.io
