---
name: Типы компонентов
permalink: '/guide/types-of-components'
---

# Типы компонентов

В Preact есть два типа компонентов:

- Классические компоненты с _методами жизненного цикла_ и состоянием.
- Функциональные компоненты без сохранения состояния, представляющие собой функции, которые принимают `props` и возвращают [JSX].

В рамках этих двух типов также существует несколько разных способов реализации компонентов.

---

<div><toc></toc></div>

---

## Пример

Давайте воспользуемся примером: простой компонент `<Link>`, который создает элемент HTML `<a>`:

```js
class Link extends Component {
	render(props, state) {
		return <a href={props.href}>{ props.children }</a>;
	}
}
```

Мы можем создать/отобразить этот компонент следующим образом:

```xml
<Link href="http://example.com">Некоторый текст</Link>
```


### Деструктуризация входных параметров и состояния

Поскольку это ES6/ES2015, мы можем ещё больше упростить наш компонент `<Link>`, отобразив ключи из `props` (первый аргумент `render()`) в локальные переменные с помощью [деструктуризации](https://github.com/lukehoban/es6features#destructuring):

```js
class Link extends Component {
	render({ href, children }) {
		return <a {...{ href, children }} />;
	}
}
```

Если мы хотим скопировать _все_ `props`, переданные нашему компоненту `<Link>`, на элемент `<a>`, мы можем использовать оператор [spread](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Spread_operator):

```js
class Link extends Component {
	render(props) {
		return <a {...props} />;
	}
}
```


### Функциональные компоненты без сохранения состояния

Наконец, мы видим, что этот компонент не хранит состояние — мы можем отрисовать компонент с теми же параметрами и каждый раз получать один и тот же результат. В таких случаях лучше всего использовать функциональный компонент без состояния (Stateless Functional Component). Это просто функции, которые принимают `props` в качестве аргумента и возвращают JSX.

```js
const Link = ({ children, ...props }) => (
	<a {...props}>{ children }</a>
);
```

> *Примечание ES2015:* вышеприведённое является стрелочной функцией, и поскольку в теле функции мы использовали круглые скобки вместо фигурных скобок, значение внутри скобок возвращается автоматически. Подробнее об этом можно прочитать [здесь](https://github.com/lukehoban/es6features#arrows).
