---
name: Связанное состояние
permalink: '/guide/linked-state'
---

# Связанное состояние

В одной из областей Preact идет немного дальше, чем React, — это оптимизация изменения состояния. Распространённым паттерном в коде ES2015 React является использование функций Arrow внутри метода `render()` для обновления состояния в ответ на события. Создание функций, заключённых в область видимости, при каждом рендеринге неэффективно и заставляет сборщик мусора делать больше работы, чем необходимо.

---

<div><toc></toc></div>

---

## Способ с более приятным руководством

Одно из решений — объявлять связанные методы компонентов с помощью свойств класса ES7 ([поля экземпляра класса](https://github.com/jeffmo/es-class-fields-and-static-properties)):

```js
class Foo extends Component {
	updateText = e => {
		this.setState({ text: e.target.value });
	};
	render({ }, { text }) {
		return <input value={text} onInput={this.updateText} />;
	}
}
```

Хотя это позволяет добиться гораздо лучшей производительности во время выполнения, это всё ещё много лишнего кода для соединения состояния с пользовательским интерфейсом.

> Другое решение — связывать методы компонентов _декларативно_, используя декораторы ES7, такие как `@bind` из пакета [decko](https://github.com/developit/decko):


## Связанное состояние на помощь

К счастью, есть решение в виде модуля preact [`linkState`](https://github.com/developit/linkstate).

> В предыдущих версиях Preact была встроена функция `linkState()`; Однако с тех пор она была перенесена в отдельный модуль. Если вы хотите восстановить старое поведение, смотрите [эту страницу](https://github.com/developit/linkstate#usage) для получения информации об использовании полифилла.

Вызов `linkState(this, 'text')` возвращает функцию-обработчик, которая, передавая событие, использует его значение для обновления именованного свойства в состоянии вашего компонента. Многократные вызовы `linkState(component, name)` с одним и тем же `component` и `name` кэшируются, поэтому производительность практически не снижается.

Вот предыдущий пример, переписанный с использованием **Связанного состояния**:

```js
import linkState from 'linkstate';

class Foo extends Component {
	render({ }, { text }) {
		return <input value={text} onInput={linkState(this, 'text')} />;
	}
}
```

Эта форма записи лаконична, легка для восприятия и эффективна. Она обрабатывает состояние связывания с любым типом входа. Необязательный третий аргумент `'path'` может быть использован для явного указания пути к новому значению состояния с точечной нотацией для более настраиваемых привязок (например, привязка к значению стороннего компонента).


## Пользовательские маршруты событий

По умолчанию `linkState()` попытается автоматически извлечь соответствующее значение из события. Например, элемент `<input>` установит для данного свойства state значение `event.target.value` или `event.target.checked` в зависимости от типа ввода. Для пользовательских обработчиков событий при передаче скалярных значений обработчику, сгенерированному `linkState()`, будет просто использоваться скалярное значение. В большинстве случаев такое поведение желательно.

Однако есть случаи, когда это нежелательно — пользовательские события и сгруппированные радиокнопки являются двумя такими примерами. В этих случаях в `linkState()` может быть передан третий аргумент, указывающий путь к ключу с точечной нотацией в событии, где можно найти значение.

Чтобы понять эту особенность, полезно заглянуть под капот `linkState()`. Ниже показан созданный вручную обработчик событий, который сохраняет значение из объекта Event в состояние. Функционально она эквивалентна версии `linkState()`, но не включает оптимизацию мемоизации, которая делает `linkState()` ценной.

```js
// этот обработчик, возвращенный из linkState:
handler = linkState(this, 'thing', 'foo.bar');

// ...функционально эквивалентен:
handler = event => {
  this.setState({
    thing: event.foo.bar
  });
}
```


### Иллюстрация: Сгруппированные радиокнопки

Следующий код работает не так, как ожидалось. Если пользователь нажимает «no», `noChecked` становится `true`, но `yesChecked` остается `true`, так как `onChange` не срабатывает на другой радиокнопке:

```js
import linkState from 'linkstate';

class Foo extends Component {
  render({ }, { yes, no }) {
    return (
      <div>
        <input type="radio" name="demo"
          value="yes" checked={yes}
          onChange={linkState(this, 'yes')}
        />
        <input type="radio" name="demo"
          value="no" checked={no}
          onChange={linkState(this, 'no')}
        />
      </div>
    );
  }
}
```


Здесь поможет третий аргумент `linkState`. Он позволяет указать путь к объекту события, который будет использоваться в качестве связанного значения. Обращаясь к предыдущему примеру, давайте явно укажем linkState получать значение своего нового состояния из свойства `value` в `event.target`:

```js
import linkState from 'linkstate';

class Foo extends Component {
  render({ }, { answer }) {
    return (
      <div>
        <input type="radio" name="demo"
          value="yes" checked={answer == 'yes'}
          onChange={linkState(this, 'answer', 'target.value')}
        />
        <input type="radio" name="demo"
          value="no" checked={answer == 'no'}
          onChange={linkState(this, 'answer', 'target.value')}
        />
      </div>
    );
  }
}
```

Теперь пример работает как надо!
