---
name: Что нового в Preact X
description: 'Новые возможности и изменения в Preact X'
---

# Что нового в Preact X

Preact X — это огромный шаг вперед по сравнению с Preact 8.x. Мы переосмыслили каждый бит и байт нашего кода и добавили множество важных функций. То же самое относится и к улучшению совместимости для поддержки большего количества библиотек сторонних разработчиков.

В двух словах Preact X — это то, чем мы всегда хотели видеть Preact: Маленькая, быстрая и многофункциональная библиотека. Что касается размеров, то вы будете рады узнать, что все новые возможности и улучшенный рендеринг уместились на той же площади, что и в `8.x`!

---

<div><toc></toc></div>

---

## Фрагменты

Фрагменты — основная новая возможность Preact X и одна из главных причин переосмысления архитектуры Preact. Они представляют собой особый вид компонентов, которые отображают дочерние элементы в одном ряду с родительскими, без дополнительного обертывающего DOM-элемента. Кроме того, они позволяют возвращать из `render` несколько узлов.

[Подробнее о фрагментах →](/guide/v10/components#fragments)

```jsx
// --repl
function Foo() {
  return (
    <>
      <div>A</div>
      <div>B</div>
    </>
  );
}
```

## componentDidCatch

Мы все хотели бы, чтобы ошибки не возникали в наших приложениях, но иногда они возникают. С помощью `componentDidCatch` теперь можно перехватывать и обрабатывать любые ошибки, возникающие в методах жизненного цикла, таких как `render`, включая исключения, находящиеся глубоко в дереве компонентов. Это может быть использовано для вывода удобных сообщений об ошибках или записи журнала во внешнюю службу, если что-то пошло не так.

[Подробнее о жизненном цикле →](/guide/v10/components#error-boundaries)

```jsx
// --repl
class Catcher extends Component {
  state = { errored: false };

  componentDidCatch(error) {
    this.setState({ errored: true });
  }

  render(props, state) {
    if (state.errored) {
      return <p>Что-то пошло не так</p>;
    }
    return props.children;
  }
}
```

## Хуки

`Хуки` — это новый способ упростить обмен логикой между компонентами. Они представляют собой альтернативу существующему API компонентов, основанному на классах. В Preact они находятся в аддоне, который может быть импортирован через `preact/hooks`.

[Подробнее о хуках →](/guide/v10/hooks)

```jsx
// --repl
function Counter() {
  const [value, setValue] = useState(0);
  const increment = useCallback(() => setValue(value + 1), [value]);

  return (
    <div>
      Счётчик: {value}
      <button onClick={increment}>Увеличить</button>
    </div>
  );
}
```

## createContext

API `createContext` является истинным преемником `getChildContext()`. Если `getChildContext` хорошо работает, когда вы абсолютно уверены, что никогда не измените значение, то при возврате `false` компонент, находящийся между провайдером и потребителем, блокирует обновление через `shouldComponentUpdate`. С появлением нового контекстного API эта проблема ушла в прошлое. Это настоящее решение _pub/sub_ для доставки обновлений вглубь дерева.

[Подробнее о `createContext` →](/guide/v10/context#createcontext)

```jsx
const Theme = createContext('light');

function ThemedButton(props) {
  return <Theme.Consumer>{(theme) => <div>Текущая тема: {theme}</div>}</Theme.Consumer>;
}

function App() {
  return (
    <Theme.Provider value='dark'>
      <SomeComponent>
        <ThemedButton />
      </SomeComponent>
    </Theme.Provider>
  );
}
```

## Пользовательские свойства CSS

Иногда именно мелочи имеют огромное значение. Благодаря последним достижениям в области CSS вы можете использовать [CSS-переменные](https://developer.mozilla.org/ru/docs/Web/CSS/--*) для стилизации:

```jsx
function Foo(props) {
  return <div style={{ '--theme-color': 'blue' }}>{props.children}</div>;
}
```

## Перенос пакета `preact-compat` в ядро

Несмотря на то, что мы всегда стремились добавлять новые возможности и продвигать Preact вперед, пакет `preact-compat` не получил такой большой любви. До сих пор он хранился в отдельном репозитории, что затрудняло координацию крупных изменений, охватывающих Preact и слой совместимости. Благодаря переносу compat в тот же пакет, что и сам Preact, для использования библиотек из экосистемы React не нужно ничего дополнительно устанавливать.

Слой совместимости теперь называется [preact/compat](/guide/v10/differences-to-react#features-exclusive-to-preactcompat), и в нем появилось несколько новых приемов, таких как `forwardRef`, `memo` и множество улучшений совместимости.

```js
// Preact 8.x
import React from 'preact-compat';

// Preact X
import React from 'preact/compat';
```

## Многочисленные исправления совместимости

Их слишком много, чтобы их перечислять, но мы многократно увеличили совместимость с библиотеками из экосистемы React. Мы специально включили в процесс тестирования несколько популярных пакетов, чтобы гарантировать их полную поддержку.

Если вы столкнулись с библиотекой, которая плохо работала с Preact 8, вам следует попробовать её в X. Велика вероятность, что все будет работать как надо ;)
