---
name: Обновление с Preact 8.x
description: 'Узнайте, как обновить приложение Preact 8.x до Preact X'
---

# Обновление с Preact 8.x

Этот документ предназначен для того, чтобы помочь вам выполнить обновление существующего приложения Preact 8.x до Preact X и разделен на 3 основных раздела.

Preact X предлагает множество новых интересных функций, таких как `фрагменты` и `хуки`, и значительно улучшенную совместимость с экосистемой React. Мы старались свести любые критические изменения к минимуму, но не смогли полностью исключить их все без ущерба для нашего набора функций.

---

<div><toc></toc></div>

---

## Обновление зависимостей

_Примечание. В этом руководстве мы будем использовать клиент `npm`, и команды должны быть легко применимы к другим менеджерам пакетов, таким как `yarn`._

Начнём! Сначала установите Preact X:

```bash
npm install preact
```

Поскольку совместимость перенесена в ядро, в `preact-compat` больше нет необходимости. Удалите его с помощью:

```bash
npm remove preact-compat
```

### Обновление библиотек, связанных с preact

Чтобы гарантировать стабильную экосистему для наших пользователей (особенно для корпоративных пользователей), мы выпустили основные обновления версий библиотек, связанных с Preact X. Если вы используете `preact-render-to-string`, вам необходимо обновить его до версии, работающей с X.

| Библиотека                | Preact 8.x | Preact X |
| ------------------------- | ---------- | -------- |
| `preact-render-to-string` | 4.x        | 5.x      |
| `preact-router`           | 2.x        | 3.x      |
| `preact-jsx-chai`         | 2.x        | 3.x      |
| `preact-markup`           | 1.x        | 2.x      |

### Пакет `compat` перемещён в ядро

Чтобы сторонние библиотеки React работали с Preact, мы поставляем уровень **совместимости**, который можно импортировать через `preact/compat`. Ранее он был доступен как отдельный пакет, но для упрощения координации мы переместили его в основной репозиторий. Поэтому вам нужно будет изменить существующие объявления импорта или псевдонима с `preact-compat` на `preact/compat` (обратите внимание на косую черту).

Be careful not to introduce any spelling errors here. A common one seems to be to write `compact` instead of `compat`. If you're having trouble with that, think of `compat` as the `compatibility` layer for react. That's where the name is coming from.
Будьте осторожны, чтобы не допустить здесь орфографических ошибок. Чаще всего пишут `compact` вместо `compat`. Если у вас с этим проблемы, вспомните, что `compat` от слова `compatibility` («совместимость»), а не от слова `compact` («компактный»). Отсюда и название.

### Сторонние библиотеки

Из-за характера критических изменений некоторые существующие библиотеки могут перестать работать с X. Большинство из них были обновлены уже в соответствии с нашим графиком бета-тестирования, но вы можете столкнуться с библиотекой, в которой это не так.

#### preact-redux

`preact-redux` — одна из таких библиотек, которая ещё не обновлялась. Хорошей новостью является то, что `preact/compat` гораздо более совместим с React и работает «из коробки» с привязками React, называемыми `react-redux`. Переключение на него разрешит ситуацию. Убедитесь, что вы присвоили псевдонимам `react` и `react-dom` `preact/compat` в вашем сборщике.

1. Удалите `preact-redux`
2. Установите `react-redux`

#### mobx-preact

Из-за нашей повышенной совместимости с экосистемой React этот пакет больше не нужен. Вместо этого используйте `mobx-react`.

1. Удалите `mobx-preact`
2. Установите `mobx-react`

#### styled-components

Preact 8.x работал только до `styled-components@3.x`. С Preact X этого барьера больше нет, и мы работаем с последней версией `styled-components`. Убедитесь, что вы правильно настроили [псевдонимы react для preact](/guide/v10/getting-started#aliasing-react-to-preact).

#### preact-portal

Компонент `Portal` теперь является частью `preact/compat`.

1. Удалите `preact-portal`
2. Импортируйте `createPortal` из `preact/compat`

## Подготовка кода

### Использование именованного экспорта

Для лучшей поддержки древовидных изменений мы больше не поставляем экспорт `default` в preact core. Преимущество такого подхода заключается в том, что в пакет будет включен только тот код, который вам необходим.

```js
// Preact 8.x
import Preact from 'preact';

// Preact X
import * as preact from 'preact';

// Предпочтительный способ: Именованный экспорт (работает в 8.x и Preact X)
import { h, Component } from 'preact';
```

_Примечание: Это изменение не затрагивает `preact/compat`. Он по-прежнему имеет именованный экспорт и экспорт по умолчанию, чтобы оставаться совместимым с react._.

### `render()` всегда дифференцирует существующие дочерние элементы

В Preact 8.x вызов `render()` всегда добавлял элементы в контейнер.

```jsx
// Существующая разметка:
<body>
  <div>привет</div>
</body>;

render(<p>foo</p>, document.body);
render(<p>bar</p>, document.body);

// Preact 8.x output:
<body>
  <div>привет</div>
  <p>foo</p>
  <p>bar</p>
</body>;
```

Для диффузии существующих дочерних элементов в Preact 8 необходимо было предоставить существующий узел DOM.

```jsx
// Существующая разметка:
<body>
  <div>привет</div>
</body>;

let element;
element = render(<p>foo</p>, document.body);
element = render(<p>bar</p>, document.body, element);

// Preact 8.x output:
<body>
  <div>привет</div>
  <p>bar</p>
</body>;
```

В Preact X функция `render()` всегда дифференцирует дочерние элементы DOM внутри контейнера. Таким образом, если ваш контейнер содержит DOM, который не был отрендерен Preact, Preact попытается сравнить его с элементами, которые вы ему передадите. Это новое поведение более точно соответствует поведению других библиотек Virtual DOM.

```jsx
// Существующая разметка:
<body>
  <div>привет</div>
</body>;

render(<p>foo</p>, document.body);
render(<p>bar</p>, document.body);

// Вывод Preact X:
<body>
  <p>bar</p>
  <div>привет</div>
</body>;
```

Если вы ищете поведение, которое точно соответствует тому, как работает метод `render` в React, используйте метод `render`, экспортированный из `preact/compat`.

### `props.children` не всегда является `массивом`

В Preact X мы больше не можем гарантировать, что `props.children` всегда будет иметь тип `array`. Это изменение было необходимо для устранения неоднозначностей синтаксического анализа в отношении `фрагментов` и компонентов, которые возвращают `массив` дочерних элементов. В большинстве случаев вы можете этого даже не заметить. Только в тех местах, где вы будете напрямую использовать методы массива в `props.children`, необходимо обернуть их в `toChildArray`. Эта функция всегда будет возвращать массив.

```jsx
// Preact 8.x
function Foo(props) {
  // `.length` — это метод массива. В Preact X, когда `props.children` не является массивом, эта строка выдаст исключение
  const count = props.children.length;
  return <div>У меня {count} дочерних элементов</div>;
}

// Preact X
import { toChildArray } from 'preact';

function Foo(props) {
  const count = toChildArray(props.children).length;
  return <div>У меня {count} дочерних элементов</div>;
}
```

### Не обращайтесь к `this.state` синхронно

В Preact X состояние компонента больше не будет изменяться синхронно. Это означает, что чтение из `this.state` сразу после вызова `setState` вернет предыдущие значения. Вместо этого вам следует использовать функцию обратного вызова для изменения состояния, которое зависит от предыдущих значений.

```jsx
this.state = { counter: 0 };

// Preact 8.x
this.setState({ counter: this.state.counter + 1 });

// Preact X
this.setState((prevState) => {
  // В качестве альтернативы верните здесь `null`, чтобы прервать обновление состояния
  return { counter: prevState.counter + 1 };
});
```

### `dangerouslySetInnerHTML` пропустит сравнение дочерних элементов

Если у `vnode` установлено свойство `dangerousSetInnerHTML`, Preact пропустит сравнение дочерних элементов `vnode`.

```jsx
<div dangerouslySetInnerHTML='foo'>
  <span>Меня пропустят</span>
  <p>И меня тоже</p>
</div>
```

## Примечания для авторов библиотек

Этот раздел предназначен для авторов библиотек, которые поддерживают пакеты для использования с Preact X. Вы можете спокойно пропустить этот раздел, если вы ничего такого не пишете.

### Форма `VNode` изменилась

Мы переименовали/переместили следующие свойства:

- `attributes` -> `props`
- `nodeName` -> `type`
- `children` -> `props.children`

Как бы мы ни старались, мы всегда сталкивались с крайними случаями со сторонними библиотеками, написанными для React. Это изменение формы нашего `vnode` устранило множество трудно обнаруживаемых ошибок и сделало наш `compat`-код намного чище.

### Соседние текстовые узлы больше не соединяются

В Preact 8.x у нас была функция, позволяющая объединять соседние текстовые примечания в целях оптимизации. Это больше не относится к X, потому что мы больше не проводим прямые различия с DOM. На самом деле мы заметили, что это снижает производительность X, поэтому удалили его. Возьмем следующий пример:

```jsx
// Preact 8.x
console.log(<div>foo{'bar'}</div>);
// Регистрирует такую ​​структуру:
//   div
//     text

// Preact X
console.log(<div>foo{'bar'}</div>);
// Регистрирует такую ​​структуру:
//   div
//     text
//     text
```
