---
name: Переход на Preact с React
permalink: '/guide/switching-to-preact'
description: 'Всё, что нужно знать для перехода с React на Preact.'
---

# Переход на Preact с React

`preact/compat` — это наш слой совместимости, который позволяет использовать множество библиотек экосистемы React и применять их в Preact. Это рекомендуемый способ опробовать Preact, если у вас уже есть приложение на React.

Это позволяет вам продолжать писать код React/ReactDOM без каких-либо изменений в рабочем процессе или кодовой базе. `preact/compat` добавляет где-то около 2 КБ к размеру вашего пакета, но имеет то преимущество, что поддерживает подавляющее большинство существующих модулей React, которые вы можете найти в npm. Пакет `preact/compat` предоставляет все необходимые настройки поверх ядра Preact, чтобы он работал так же, как `react` и `react-dom`, в одном модуле.

---

<div><toc></toc></div>

---

## Настройка `preact/compat`

Чтобы настроить `preact/compat`, вам нужно использовать псевдонимы `react` и `react-dom` для `preact/compat`. На странице [Первые шаги](/guide/v10/getting-started#aliasing-react-to-preact) подробно описывается настройка псевдонимов в различных сборщиках.

## PureComponent

Класс `PureComponent` работает аналогично `Component`. Разница в том, что `PureComponent` пропустит рендеринг, когда новые параметры будут равны старым. Для этого мы сравниваем старые и новые параметры посредством поверхностного сравнения, где мы проверяем каждое свойство параметра на ссылочное равенство. Это может значительно ускорить работу приложений, избегая ненужных повторных рендерингов. Это работает путем добавления хука жизненного цикла `shouldComponentUpdate` по умолчанию.

```jsx
import { render } from 'preact';
import { PureComponent } from 'preact/compat';

class Foo extends PureComponent {
  render(props) {
    console.log('render');
    return <div />;
  }
}

const dom = document.getElementById('root');
render(<Foo value='3' />, dom);
// Лог: "render"

// Рендеринг второй раз, ничего не логируется
render(<Foo value='3' />, dom);
```

> Обратите внимание, что преимущество `PureComponent` окупается только тогда, когда рендеринг обходится дорого. Для простых дочерних деревьев просто выполнить «рендеринг» (`render`) может быть быстрее, чем накладные расходы на сравнение параметров.

## memo

`memo` эквивалентен функциональным компонентам, как `PureComponent` — классам. Он использует ту же функцию сравнения, но позволяет вам указать собственную специализированную функцию, оптимизированную для вашего случая использования.

```jsx
import { memo } from 'preact/compat';

function MyComponent(props) {
  return <div>Hello {props.name}</div>;
}

// Использование с функцией сравнения по умолчанию
const Memoed = memo(MyComponent);

// Использование с пользовательской функцией сравнения
const Memoed2 = memo(MyComponent, (prevProps, nextProps) => {
  // Повторный рендеринг только при изменении `name'
  return prevProps.name === nextProps.name;
});
```

> Функция сравнения отличается от `shouldComponentUpdate` тем, что она проверяет, являются ли два объекта props **равными**, тогда как `shouldComponentUpdate` проверяет, различны ли они.

## forwardRef

В некоторых случаях при написании компонента вы хотите, чтобы пользователь мог получить определённую ссылку дальше по дереву. С помощью `forwardRef` вы можете своего рода «переслать» свойство `ref`:

```jsx
import { createRef, render } from 'preact';
import { forwardRef } from 'preact/compat';

const MyComponent = forwardRef((props, ref) => {
  return <div ref={ref}>Привет, мир</div>;
});

// Использование: `ref` будет содержать ссылку на внутренний `div` вместо `MyComponent`
const ref = createRef();
render(<MyComponent ref={ref} />, dom);
```

Этот компонент наиболее полезен для авторов библиотек.

## Порталы

В редких случаях вам может потребоваться продолжить рендеринг в другой узел DOM. Целевой узел DOM **должен** присутствовать перед попыткой рендеринга в него.

```html
<html>
  <body>
    <!-- App отображается здесь -->
    <div id="app"></div>
    <!-- Модальные окна отображаются здесь -->
    <div id="modals"></div>
  </body>
</html>
```

```jsx
import { createPortal } from 'preact/compat';
import MyModal from './MyModal';

function App() {
  const container = document.getElementById('modals');
  return (
    <div>
      I'm app
      {createPortal(<MyModal />, container)}
    </div>
  );
}
```

> Имейте в виду, что из-за того, что Preact повторно использует систему событий браузера, события не передаются через контейнер портала в другое дерево.

## Suspense

Основная идея `Suspense` (дословно — «Приостановка») заключается в том, чтобы позволить разделам вашего пользовательского интерфейса отображать своего рода заполнительный контент, в то время как компоненты, расположенные ниже по дереву, всё ещё загружаются. Распространённым вариантом использования этого является разделение кода, когда вам нужно загрузить компонент из сети, прежде чем вы сможете его отобразить.

```jsx
import { Suspense, lazy } from 'preact/compat';

const SomeComponent = lazy(() => import('./SomeComponent'));

// Usage
<Suspense fallback={<div>загрузка...</div>}>
  <Foo>
    <SomeComponent />
  </Foo>
</Suspense>;
```

В этом примере пользовательский интерфейс будет отображать текст `загрузка...` до тех пор, пока `SomeComponent` не будет загружен и Promise не будет разрешен.

> Suspense как в React, так и в Preact ещё не совсем завершен. Хотя команда React по-прежнему активно отговаривает пользователей взаимодействовать с ним напрямую для получения данных, это шаблон, который некоторые пользователи Preact с удовольствием используют на протяжении последних нескольких лет. Существуют несколько известных проблем (пожалуйста, смотрите [наш трекер](https://github.com/preactjs/preact/issues?q=is%3Aissue+is%3Aopen+suspense) для получения актуальной информации), но в целом он считается достаточно стабильным для использования в работе, если вы этого хотите.
>
> Этот сайт, например, построен с использованием стратегии получения данных на основе Suspense, используемой для загрузки всего контента, который вы видите.
