---
name: Сигналы
description: 'Сигналы: составное реактивное состояние с автоматическим рендерингом.'
---

# Сигналы

Сигналы — это реактивные примитивы для управления состоянием приложения.

Уникальность _сигналов_ заключается в том, что изменения состояния автоматически обновляют компоненты и пользовательский интерфейс наиболее эффективным образом. Автоматическое связывание состояний и отслеживание зависимостей позволяет _сигналам_ обеспечивать превосходную эргономику и производительность, устраняя при этом наиболее распространённые проблемы управления состояниями.

Сигналы эффективны в приложениях любого размера, их эргономика ускоряет разработку небольших приложений, а характеристики производительности гарантируют, что приложения любого размера будут работать быстро по умолчанию.

---

**Важно**

В этом руководстве мы рассмотрим использование сигналов в Preact, и хотя это в значительной степени применимо как к библиотекам Core, так и к React, есть некоторые различия в использовании. Лучшие рекомендации по их использованию — в соответствующих документациях: [`@preact/signals-core`](https://github.com/preactjs/signals), [`@preact/signals-react`](https://github.com/preactjs/signals/tree/main/packages/react)

---

<div><toc></toc></div>

---

## Введение

Большая часть проблем управления состоянием в JavaScript связана с реакцией на изменения данного значения, поскольку значения не наблюдаются напрямую. Решения обычно обходят эту проблему, сохраняя значения в переменной и постоянно проверяя, не изменились ли они, что обременительно и не идеально для производительности. В идеале нам нужен способ выразить значение, которое сообщит нам, когда оно изменится. Это то, что делают _сигналы_.

По своей сути сигнал — это объект со свойством `.value`, содержащим значение. У этого есть важная особенность: значение сигнала может меняться, но сам сигнал всегда остается неизменным:

```js
// --repl
import { signal } from '@preact/signals';

const count = signal(0);

// Получаем значение сигнала, обращаясь к .value:
console.log(count.value); // 0

// Обновляем значение сигнала:
count.value += 1;

// Значение сигнала изменилось:
console.log(count.value); // 1
```

В Preact, когда сигнал передается через дерево в качестве параметра или контекста, мы передаем только ссылки на сигнал. Сигнал можно обновить без повторной визуализации каких-либо компонентов, поскольку компоненты видят сигнал, а не его значение. Это позволяет нам пропустить всю дорогостоящую работу по рендерингу и сразу перейти к любым компонентам в дереве, которые фактически обращаются к свойству `.value` сигнала.

У сигналов есть вторая важная характеристика: они отслеживают, когда к их значению обращаются и когда оно обновляется. В Preact доступ к свойству `.value` сигнала изнутри компонента автоматически перерисовывает компонент при изменении значения этого сигнала.

```jsx
// --repl
import { render } from 'preact';
// --repl-before
import { signal } from '@preact/signals';

// Создаём сигнал, на который можно подписаться:
const count = signal(0);

function Counter() {
  // Компонент автоматически перерисовывается при доступе к .value`:
  const value = count.value;

  const increment = () => {
    // Сигнал обновляется путем присвоения значения свойству `.value`:
    count.value++;
  };

  return (
    <div>
      <p>Счётчик: {value}</p>
      <button onClick={increment}>Нажми меня</button>
    </div>
  );
}
// --repl-after
render(<Counter />, document.getElementById('app'));
```

Наконец, _сигналы_ глубоко интегрированы в Preact, чтобы обеспечить максимально возможную производительность и эргономику. В приведенном выше примере мы обратились к `count.value`, чтобы получить текущее значение сигнала `count`, однако в этом нет необходимости. Вместо этого мы можем позволить Preact сделать всю работу за нас, используя сигнал `count` непосредственно в JSX:

```jsx
// --repl
import { render } from 'preact';
// --repl-before
import { signal } from '@preact/signals';

const count = signal(0);

function Counter() {
  return (
    <div>
      <p>Счётчик: {count}</p>
      <button onClick={() => count.value++}>Нажми меня</button>
    </div>
  );
}
// --repl-after
render(<Counter />, document.getElementById('app'));
```

## Установка

Сигналы можно установить, добавив в проект пакет `@preact/signals`:

```bash
npm install @preact/signals
```

После установки через выбранный вами менеджер пакетов вы готовы импортировать сигналы в свое приложение.

## Пример использования

Давайте использовать сигналы в реальном сценарии. Мы собираемся создать приложение списка дел, в котором вы сможете добавлять и удалять элементы. Начнём с моделирования состояния. Сначала нам понадобится сигнал, содержащий список задач, который мы можем представить с помощью массива (`Array`):

```jsx
import { signal } from '@preact/signals';

const todos = signal([{ text: 'Купить продукты' }, { text: 'Выгулять собаку' }]);
```

Чтобы позволить пользователю вводить текст для нового элемента задачи, нам понадобится ещё один сигнал, который мы вскоре подключим к элементу `<input>`. На данный момент мы уже можем использовать этот сигнал для создания функции, которая добавляет элемент задачи в наш список. Помните, мы можем обновить значение сигнала, присвоив его свойству `.value`:

```jsx
// Мы будем использовать это для ввода позже.
const text = signal('');

function addTodo() {
  todos.value = [...todos.value, { text: text.value }];
  text.value = ''; // Очистить входное значение при добавлении
}
```

> :bulb: Совет: Сигнал обновится только в том случае, если вы присвоите ему новое значение. Если значение, которое вы присваиваете сигналу, равно его текущему значению, оно не будет обновляться.
>
> ```js
> const count = signal(0);
>
> count.value = 0; // ничего не делает — значение уже равно 0
>
> count.value = 1; // обновляется — значение другое
> ```

Давайте проверим, верна ли наша логика на данный момент. Когда мы обновляем сигнал `text` и вызываем `addTodo()`, мы должны увидеть новый элемент, добавляемый к сигналу `todos`. Мы можем смоделировать этот сценарий, вызывая эти функции напрямую — пользовательский интерфейс пока не нужен!

```jsx
// --repl
import { signal } from '@preact/signals';

const todos = signal([{ text: 'Купить продукты' }, { text: 'Выгулять собаку' }]);

const text = signal('');

function addTodo() {
  todos.value = [...todos.value, { text: text.value }];
  text.value = ''; // Сбросить входное значение при добавлении
}

// Проверим, работает ли наша логика
console.log(todos.value);
// Лог: [{text: "Купить продукты"}, {text: "Выгулять собаку"}]

// Имитируем добавление новой задачи
text.value = 'Прибраться';
addTodo();

// Убеждаемся, что задача добавлена в массив, а сигнал `text` очищен:
console.log(todos.value);
// Лог: [{text: "Купить продукты"}, {text: "Выгулять собаку"}, {text: "Прибраться"}]

console.log(text.value); // Лог: ""
```

Последняя функция, которую мы хотели бы добавить, — это возможность удалять элемент задачи из списка. Для этого мы добавим функцию, которая удаляет заданный элемент задачи из массива задач:

```jsx
function removeTodo(todo) {
  todos.value = todos.value.filter((t) => t !== todo);
}
```

## Создание пользовательского интерфейса

Теперь, когда мы смоделировали состояние нашего приложения, пришло время создать красивый пользовательский интерфейс, с которым смогут взаимодействовать пользователи.

```jsx
function TodoList() {
  const onInput = (event) => (text.value = event.currentTarget.value);

  return (
    <>
      <input value={text.value} onInput={onInput} />
      <button onClick={addTodo}>Добавить</button>
      <ul>
        {todos.value.map((todo) => (
          <li>
            {todo.text} <button onClick={() => removeTodo(todo)}>❌</button>
          </li>
        ))}
      </ul>
    </>
  );
}
```

И теперь у нас есть полностью работающее приложение todo! Вы можете опробовать полную версию приложения [здесь](/repl?example=todo-list-signals) :tada:

## Получение состояния с помощью вычисляемых сигналов

Давайте добавим ещё одну функцию в наше приложение задач: каждый элемент задачи можно пометить как выполненный, и мы покажем пользователю количество выполненных элементов. Для этого мы импортируем функцию [`computed(fn)`](#computedfn), которая позволяет нам создать новый сигнал, вычисляемый на основе значений других сигналов. Возвращённый вычисленный сигнал доступен только для чтения, и его значение автоматически обновляется при изменении любых сигналов, к которым осуществляется доступ из функции обратного вызова.

```jsx
// --repl
import { signal, computed } from '@preact/signals';

const todos = signal([
  { text: 'Купить продукты', completed: true },
  { text: 'Выгулять собаку', completed: false },
]);

// Создаём сигнал, вычисляемый из других сигналов
const completed = computed(() => {
  // Когда `todos` изменяется, это автоматически повторяется:
  return todos.value.filter((todo) => todo.completed).length;
});

// Лог: 1, потому что одна задача помечена как выполненная
console.log(completed.value);
```

Нашему простому приложению со списком задач не требуется много вычисляемых сигналов, но более сложные приложения, как правило, полагаются на метод `computed()`, чтобы избежать дублирования состояния в нескольких местах.

> :bulb: Совет: Получение как можно большего количества состояний гарантирует, что ваше состояние всегда будет иметь единственный источник истины. Это ключевой принцип сигналов. Это значительно упрощает отладку на случай, если в дальнейшем возникнет ошибка в логике приложения, поскольку будет меньше поводов для беспокойства.

## Управление глобальным состоянием приложения

До сих пор мы создавали сигналы только вне дерева компонентов. Это подходит для небольшого приложения, такого как список дел, но для более крупных и сложных приложений это может затруднить тестирование. Тесты обычно включают в себя изменение значений состояния вашего приложения для воспроизведения определённого сценария, а затем передачу этого состояния компонентам и утверждение в отображаемом HTML. Для этого мы можем извлечь состояние нашего списка дел в функцию:

```jsx
function createAppState() {
  const todos = signal([]);

  const completed = computed(() => {
    return todos.value.filter((todo) => todo.completed).length;
  });

  return { todos, completed };
}
```

> :bulb: Совет: Обратите внимание, что мы сознательно не включили сюда функции `addTodo()` и `removeTodo(todo)`. Отделение данных от функций, которые их изменяют, часто помогает упростить архитектуру приложения. Для получения более подробной информации ознакомьтесь со статьёй [дизайн, ориентированный на данные](https://habr.com/ru/articles/321106/).
>
> Теперь мы можем передать состояние нашего приложения todo в качестве параметра при рендеринге:

```jsx
const state = createAppState();

// ...later:
<TodoList state={state} />;
```

Это работает в нашем приложении списка дел, поскольку состояние является глобальным, однако более крупные приложения обычно содержат несколько компонентов, требующих доступа к одним и тем же частям состояния. Обычно это включает в себя «поднятие состояния» до общего компонента-предка. Чтобы избежать передачи состояния вручную через каждый компонент через параметры, состояние можно поместить в [Context](/guide/v10/context), чтобы любой компонент в дереве мог получить к нему доступ. Вот краткий пример того, как это обычно выглядит:

```jsx
import { createContext } from 'preact';
import { useContext } from 'preact/hooks';
import { createAppState } from './my-app-state';

const AppState = createContext();

render(
  <AppState.Provider value={createAppState()}>
    <App />
  </AppState.Provider>
);

// ...позже, когда вам понадобится доступ к состоянию вашего приложения
function App() {
  const state = useContext(AppState);
  return <p>{state.completed}</p>;
}
```

Если вы хотите узнать больше о том, как работает контекст, перейдите к [документации о контексте](/guide/v10/context).

## Локальное состояние с сигналами

Большая часть состояния приложения передается с использованием параметров и контекста. Однако существует множество сценариев, в которых компоненты имеют собственное внутреннее состояние, специфичное для этого компонента. Поскольку нет никаких оснований для того, чтобы это состояние существовало как часть глобальной бизнес-логики приложения, его следует ограничить компонентом, которому оно необходимо. В этих сценариях мы можем создавать сигналы, а также вычисляемые сигналы непосредственно внутри компонентов, используя хуки `useSignal()` и `useComputed()`:

```jsx
import { useSignal, useComputed } from '@preact/signals';

function Counter() {
  const count = useSignal(0);
  const double = useComputed(() => count.value * 2);

  return (
    <div>
      <p>
        {count} x 2 = {double}
      </p>
      <button onClick={() => count.value++}>Нажми меня</button>
    </div>
  );
}
```

Эти два хука представляют собой тонкие оболочки вокруг [`signal()`](#signalinitialvalue) и [`computed()`](#computedfn), которые создают сигнал при первом запуске компонента и просто используют этот же сигнал при последующих рендерингах.

> :bulb: За кулисами реализация выглядит так:
>
> ```js
> function useSignal(value) {
>   return useMemo(() => signal(value), []);
> }
> ```

## Расширенное использование сигналов

Темы, которые мы рассмотрели до сих пор, — это всё, что вам нужно для начала. Следующий раздел предназначен для читателей, которые хотят получить ещё большую выгоду от моделирования состояния своего приложения, полностью используя сигналы.

### Реакция на сигналы вне компонентов

Работая с сигналами за пределами дерева компонентов, вы, возможно, заметили, что вычисленные сигналы не пересчитываются, если вы активно не читаете их значение. Это связано с тем, что сигналы по умолчанию являются ленивыми: они вычисляют новые значения только тогда, когда к их значению осуществляется доступ.

```js
const count = signal(0);
const double = computed(() => count.value * 2);

// Несмотря на обновление сигнала `count`, от которого зависит сигнал `double`,
// `double` ещё не обновляется, поскольку его значение ещё не использовалось.
count.value = 1;

// Чтение значения `double` приводит к его перерасчёту:
console.log(double.value); // Лог: 2
```

Возникает вопрос: как мы можем подписаться на сигналы вне дерева компонентов? Возможно, мы хотим регистрировать что-то в консоли при каждом изменении значения сигнала или сохранять состояние в [LocalStorage](https://developer.mozilla.org/ru/docs/Web/API/Window/localStorage).

Чтобы запустить произвольный код в ответ на изменение сигнала, мы можем использовать [`effect(fn)`](#effectfn). Подобно вычисляемым сигналам, эффекты отслеживают, к каким сигналам осуществляется доступ, и повторно запускают обратный вызов при изменении этих сигналов. В отличие от вычисляемых сигналов, [`effect()`](#effectfn) не возвращает сигнал — это конец последовательности изменений.

```js
import { signal, computed, effect } from '@preact/signals';

const name = signal('Джейн');
const surname = signal('Доу');
const fullName = computed(() => `${name.value} ${surname.value}`);

// Отслеживание `name` при каждом изменении:
effect(() => console.log(fullName.value));
// Лог: "Джейн Доу"

// Обновление `name` обновляет `fullName`, что снова запускает эффект:
name.value = 'Джон';
// Лог: "Джон Доу"
```

Вы можете уничтожить эффект и отказаться от подписки на все сигналы, к которым он получил доступ, вызвав возвращаемую функцию.

```js
import { signal, effect } from '@preact/signals';

const name = signal('Джейн');
const surname = signal('Доу');
const fullName = computed(() => name.value + ' ' + surname.value);

const dispose = effect(() => console.log(fullName.value));
// Лог: "Джейн Доу"

// Уничтожить эффект и подписки:
dispose();

// Обновление `name` не приводит к эффекту, поскольку оно было удалено.
// Он также не пересчитывает `fullName` теперь, когда за ним никто не наблюдает.
name.value = 'Джон';
```

> :bulb: Совет: не забудьте очистить эффекты, если вы их часто используете. В противном случае ваше приложение будет потреблять больше памяти, чем необходимо.

## Чтение сигналов без подписки на них

В тех редких случаях, когда вам нужно записать сигнал внутри [`effect(fn)`](#effectfn), но вы не хотите, чтобы эффект повторно запускался при изменении этого сигнала, вы можете использовать `.peek()`, чтобы получить текущее значение сигнала без подписки.

```js
const delta = signal(0);
const count = signal(0);

effect(() => {
  // Обновляем `count` без подписки на `count`:
  count.value = count.peek() + delta.value;
});

// Установка значения `delta` повторно запускает эффект:
delta.value = 1;

// Это не приведет к повторному запуску эффекта, поскольку он не получил доступ к `.value`:
count.value = 10;
```

> :bulb: Совет: Сценарии, в которых вы не хотите подписываться на сигнал, встречаются редко. В большинстве случаев вы хотите, чтобы ваш эффект подписывался на все сигналы. Используйте `.peek()` только тогда, когда это вам действительно нужно.

В качестве альтернативы `.peek()` у нас есть функция `untracked`, которая принимает функцию в качестве аргумента и возвращает результат выполнения этой функции. В `untracked` вы можете ссылаться на любой сигнал с помощью `.value` без создания подписки. Это может быть полезно, когда у вас есть многоразовая функция, которая обращается к `.value`, или вам нужно получить доступ к более чем одному сигналу.

```js
const delta = signal(0);
const count = signal(0);

effect(() => {
  // Обновляем `count` без подписки на `count` или `delta`:
  count.value = untracked(() => {
    count.value + delta.value
  });
});
```

## Объединение нескольких обновлений в одно

Помните функцию `addTodo()`, которую мы использовали ранее в нашем приложении todo? Вот как это выглядело:

```js
const todos = signal([]);
const text = signal('');

function addTodo() {
  todos.value = [...todos.value, { text: text.value }];
  text.value = '';
}
```

Обратите внимание, что функция запускает два отдельных обновления: одно при установке `todos.value`, а другое при установке значения `text`. Иногда это может быть нежелательно и требует объединения обоих обновлений в одно по соображениям производительности или по другим причинам. Функцию [`batch(fn)`](#batchfn) можно использовать для объединения нескольких обновлений значений в одну «фиксацию» в конце обратного вызова:

```js
function addTodo() {
  batch(() => {
    todos.value = [...todos.value, { text: text.value }];
    text.value = '';
  });
}
```

Доступ к сигналу, который был изменен в пакете, отразит его обновлённое значение. Доступ к вычисленному сигналу, который был признан недействительным другим сигналом в пакете, приведёт к повторному вычислению только необходимых зависимостей для возврата актуального значения для этого вычисленного сигнала. Любые другие недействительные сигналы остаются неизменными и обновляются только в конце пакетного обратного вызова.

```js
// --repl
import { signal, computed, effect, batch } from '@preact/signals';

const count = signal(0);
const double = computed(() => count.value * 2);
const triple = computed(() => count.value * 3);

effect(() => console.log(double.value, triple.value));

batch(() => {
  // Устанавливаем `count`, делая недействительными `double` и `triple`:
  count.value = 1;

  // Несмотря на пакетную обработку, `double` отражает новое вычисленное значение.
  // Однако `triple` будет обновляться только после завершения обратного вызова.
  console.log(double.value); // Лог: 2
});
```

> :bulb: Совет: Пакеты также могут быть вложенными, и в этом случае пакетные обновления сбрасываются только после завершения самого внешнего пакетного обратного вызова.

### Оптимизация рендеринга

С помощью сигналов мы можем обойти рендеринг Virtual DOM и связать изменения сигналов непосредственно с мутациями DOM. Если вы передаете сигнал в JSX в текстовой позиции, он будет отображаться как текст и автоматически обновляться на месте без различия Virtual DOM:

```jsx
const count = signal(0);

function Unoptimized() {
  // Перерисовывает компонент при изменении `count`:
  return <p>{count.value}</p>;
}

function Optimized() {
  // Текст автоматически обновляется без повторной отрисовки компонента:
  return <p>{count}</p>;
}
```

Чтобы включить эту оптимизацию, передайте весь сигнал в JSX вместо доступа к его свойству `.value`.

Аналогичная оптимизация рендеринга также поддерживается при передаче сигналов в качестве атрибута в элементах DOM.

## API

В этом разделе представлен обзор API сигналов. Он призван стать кратким справочником для людей, которые уже знают, как использовать сигналы, и нуждаются в напоминании о том, что доступно.

### signal(initialValue)

Создает новый сигнал с аргументом `initialValue` в качестве начального значения:

```js
const count = signal(0);
```

При создании сигналов внутри компонента используйте вариант с хуком: `useSignal(initialValue)`.

Возвращенный сигнал имеет свойство `.value`, которое можно получить или установить для чтения и записи его значения. Чтобы прочитать сигнал без подписки на него, используйте `signal.peek()`.

### computed(fn)

Создает новый сигнал, который вычисляется на основе значений других сигналов. Возвращенный вычисленный сигнал доступен только для чтения, и его значение автоматически обновляется при изменении любых сигналов, к которым осуществляется доступ из функции обратного вызова.

```js
const name = signal('Джейн');
const surname = signal('Доу');

const fullName = computed(() => `${name.value} ${surname.value}`);
```

При создании вычисляемых сигналов внутри компонента используйте вариант с хуком: `useComputed(fn)`.

### effect(fn)

Чтобы запустить произвольный код в ответ на изменение сигнала, мы можем использовать `effect(fn)`. Подобно вычисляемым сигналам, эффекты отслеживают, к каким сигналам осуществляется доступ, и повторно запускают обратный вызов при изменении этих сигналов. В отличие от вычисляемых сигналов, `effect()` не возвращает сигнал — это конец последовательности изменений.

```js
const name = signal('Джейн');

// Отображаем сообщение в консоли при изменении `name`:
effect(() => console.log('Привет, ', name.value));
// Лог: "Привет, Джейн"

name.value = 'Джон';
// Лог: "Привет, Джон"
```

При реагировании на изменения сигнала внутри компонента используйте вариант с хуком: `useSignalEffect(fn)`.

### batch(fn)

Функцию `batch(fn)` можно использовать для объединения нескольких обновлений значений в одну «фиксацию» в конце предоставленного обратного вызова. Пакеты могут быть вложенными, а изменения сбрасываются только после завершения обратного вызова самого внешнего пакета. Доступ к сигналу, который был изменен в пакете, отразит его обновлённое значение.

```js
const name = signal('Джейн');
const surname = signal('Доу');

// Объединяем обе записи в одно обновление
batch(() => {
  name.value = 'Джон';
  surname.value = 'Смит';
});
```

### untracked(fn)

Функция `untracked(fn)` может быть использована для доступа к значению нескольких сигналов без подписки на них.

```js
const name = signal("Jane");
const surname = signal("Doe");

effect(() => {
  untracked(() => {
    console.log(`${name.value} ${surname.value}`)
  })
})
```