---
name: Отладка приложений Preact
description: 'Как отлаживать приложения Preact, если что-то идёт не так.'
---

# Отладка приложений Preact

Preact поставляется с большим количеством инструментов, облегчающих отладку. Они помещены в один пакет — `preact/debug`.

Они включают интеграцию с нашим собственным расширением [Preact Devtools] для Chrome и Firefox.

Мы будем выводить предупреждение или ошибку всякий раз, когда обнаружим что-то неправильное, например, неправильную вложенность элементов `<table>`.

---

<div><toc></toc></div>

---

## Установка

[Preact Devtools] можно установить через магазин расширений вашего браузера.

- [Для Chrome](https://chrome.google.com/webstore/detail/preact-developer-tools/ilcajpmogmhpliinlbcdebhbcanbghmd)
- [Для Firefox](https://addons.mozilla.org/en-US/firefox/addon/preact-devtools/)
- [Для Edge](https://microsoftedge.microsoft.com/addons/detail/hdkhobcafnfejjieimdkmjaiihkjpmhk)

После установки нам нужно куда-то импортировать `preact/debug`, чтобы инициализировать соединение с расширением. Убедитесь, что этот импорт является **первым** импортом во всем вашем приложении.

> `@preact/preset-vite` автоматически включает пакет `preact/debug`. Вы можете смело пропустить следующий шаг, если используете его!

Вот пример того, как может выглядеть ваш основной файл приложения.

```jsx
// Должен быть первый импорт
import 'preact/debug';
import { render } from 'preact';
import App from './components/App';

render(<App />, document.getElementById('root'));
```

### Удаление инструментов разработчика из производственной сборки

Большинство упаковщиков позволяют вам удалять код, когда они обнаруживают, что ветвь внутри оператора `if` никогда не будет задействована. Мы можем использовать это, чтобы включать `preact/debug` только во время разработки и сохранять эти драгоценные байты при сборке.

```jsx
// Должен быть первый импорт
if (process.env.NODE_ENV === 'development') {
  // Здесь необходимо использовать require, так как на верхнем уровне разрешены только операторы импорта
  require('preact/debug');
}

import { render } from 'preact';
import App from './components/App';

render(<App />, document.getElementById('root'));
```

Убедитесь, что переменная `NODE_ENV` имеет правильное значение в вашем инструменте сборки.

## Предупреждения и ошибки отладки

Иногда могут появляться предупреждения или ошибки, когда Preact обнаруживает недействительный код. Их необходимо исправить, чтобы обеспечить безупречную работу приложения.

### `undefined` родитель передан в `render()`

Это означает, что код пытается превратить ваше приложение в ничто, а не в узел DOM. Это разница между:

```jsx
// Что получил Preact
render(<App />, undefined);

// против того, что он ожидал
render(<App />, actualDomNode);
```

Основная причина возникновения этой ошибки заключается в том, что узел DOM не присутствует в момент вызова функции `render()`. Убедитесь в его существовании.

### `undefined` компонент передан в `createElement()`

Preact будет выдавать эту ошибку, если вместо компонента передать `undefined`. Частой причиной этого является смешение `дефолтного` и `именованного` экспорта.

```jsx
// app.js
export default function App() {
  return <div>Hello World</div>;
}

// index.js: Неверно, поскольку в файле `app.js` нет именованного экспорта
import { App } from './app';
render(<App />, dom);
```

Такая же ошибка будет возникать и в обратном случае. Когда вы объявляете `именованный` экспорт и пытаетесь использовать его в качестве `дефолтного` экспорта. Один из быстрых способов проверить это (в случае, если ваш редактор этого не делает) — просто выйти из импорта:

```jsx
// app.js
export function App() {
  return <div>Hello World</div>;
}

// index.js
import App from './app';

console.log(App);
// Лог: { default: [Function] } вместо компонента
```

### Передача JSX-литерала как JSX дважды

Повторная передача JSX-литерала или компонента в JSX является некорректной и приведет к возникновению данной ошибки.

```jsx
const Foo = <div>foo</div>;
// Неверно: Foo уже содержит JSX-элемент
render(<Foo />, dom);
```

Чтобы исправить это, мы можем просто передать переменную напрямую:

```jsx
const Foo = <div>foo</div>;
render(Foo, dom);
```

### Некорректное вложение таблицы

В HTML есть очень чёткие указания о том, как должны быть структурированы таблицы. Отклонение от этого требования приводит к ошибкам рендеринга, которые очень трудно отладить. В Preact мы обнаружим это и выведем ошибку. Для получения более подробной информации о том, как должны быть структурированы таблицы, мы можем настоятельно рекомендовать [документацию MDN](https://developer.mozilla.org/ru/docs/Learn/HTML/Tables/Basics)

### Неверное `ref`-свойство

Если свойство `ref` содержит что-то неожиданное, мы выдадим эту ошибку. Сюда входят строковые `refs`, которые уже давно устарели.

```jsx
// Верно
<div ref={e => {/* ... */)}} />

// Верно
const ref = createRef();
<div ref={ref} />

// Неверно
<div ref="ref" />
```

### Неверный обработчик события

Иногда можно случайно передать обработчику события неверное значение. Они всегда должны быть `функцией` или `null`, если вы хотите его удалить. Все остальные типы недействительны.

```jsx
// Верно
<div onClick={() => console.log("клик")} />

// Неверно
<div onClick={console.log("клик")} />
```

### Хук может быть вызван только из методов рендеринга

Эта ошибка возникает при попытке использовать хук вне компонента. Они поддерживаются только внутри функционального компонента.

```jsx
// Неверно, должны использоваться внутри компонента
const [value, setValue] = useState(0);

// Верно
function Foo() {
  const [value, setValue] = useState(0);
  return <button onClick={() => setValue(value + 1)}>{value}</button>;
}
```

### Получение `vnode.[property]` устарело

В Preact X мы внесли некоторые изменения в нашу внутреннюю форму `vnode`.

| Preact 8.x         | Preact 10.x            |
| ------------------ | ---------------------- |
| `vnode.nodeName`   | `vnode.type`           |
| `vnode.attributes` | `vnode.props`          |
| `vnode.children`   | `vnode.props.children` |

### Дочерние элементы с одинаковым ключом

Уникальность библиотек, основанных на Virtual DOM, заключается в том, что они должны определять, когда дочерний элемент перемещается. Однако для того, чтобы знать, кто есть кто из дочерних элементов, нам необходимо их как-то помечать. _Это необходимо только при динамическом создании дочерних элементов_.

```jsx
// Оба элемента будут иметь одинаковый ключ "A"
<div>
  {['A', 'A'].map((char) => (
    <p key={char}>{char}</p>
  ))}
</div>
```

Правильный способ — дать им уникальные ключи. В большинстве случаев данные, над которыми выполняется итерация, имеют тот или иной вид `id`.

```jsx
const persons = [
  { name: 'Джон', age: 22 },
  { name: 'Сара', age: 24 },
];

// Где-то позже в вашем компоненте
<div>
  {persons.map(({ name, age }) => {
    return (
      <p key={name}>
        {name}, Age: {age}
      </p>
    );
  })}
</div>;
```

[Preact Devtools]: https://preactjs.github.io/preact-devtools/
