---
title: Упрощение архитектуры островов
date: 2024-10-27
authors:
  - reaper
---

> Это слегка изменённая версия оригинальной статьи по адресу https://barelyhuman.github.io/preact-islands-diy

# Острова

## Введение

Перед вами простое руководство, которое должно помочь вам понять, как работает островная архитектура, и как настроить свою собственную, используя инструменты, которые у вас уже есть.

Прежде всего, что такое острова? Вы можете прочитать больше об их происхождении в статье

[Архитектура островов - Джейсон Миллер &rarr;](https://jasonformat.com/islands-architecture/)

## Почему?

Для многих разработчиков, которые работали с серверным рендерингом в течение некоторого времени, мы как бы ожидали, что фронтенд-технологии в какой-то момент сделают поворот в сторону серверного рендеринга, поскольку получение и обработка данных почти всегда быстрее на сервере, где вы ближе к данным.

Это одна из многих причин, но есть и другие, о которых весь веб ведёт дискуссии, поэтому мы оставим это умным людям.

Давайте перейдем к реализации концепции.

# Начало работы

## Базовая реализация

Базовая реализация может быть обобщена для большинства приложений SSR + клиентской гидратации.

Вот общий обзор:

1. Изначально отрендерить представление на сервере как статическую страницу.
2. Гидратировать приложение на клиенте.

Теперь перейдем к деталям каждого из этих этапов.

### Начальный серверный рендер

На этом этапе вы всё ещё строите дерево компонентов с помощью любой библиотеки пользовательского интерфейса, которую вы используете: Vue, React, Preact, Solid и т. д. Затем вы упрощаете дерево компонентов, чтобы оставить только статические и немедленно вычисляемые данные. В этом случае не выполняется код с побочными эффектами и управлением состоянием.

Результатом является статический HTML-документ, который вы можете отправить клиенту.

Поскольку это руководство связано с [preact](https://preactjs.com/), мы будем использовать библиотеку от команды Preact, которая помогает нам достичь этой цели.

Вот как будет выглядеть очень примитивная реализация рендеринга компонента на сервере.

Мы используем `express.js` здесь в качестве примера, так как это первый выбор большинства новичков; процесс в основном такой же для любого другого веб-серверного движка, который вы выберете: Hapi, Koa, Fastify и т. д.

```js
// server.js
import { h } from 'preact'
import preactRenderToString from 'preact-render-to-string'

// ...оставшаяся настройка express.js

const HomePage = () => {
  return h('h1', {}, 'hello')
}

app.get('/', async (req, res) => {
  res.send(preactRenderToString(h(HomePage, {})))
})
```

Здесь основная работа выполняется `preactRenderToString`, а всё, что мы делаем, — это пишем компоненты. С небольшим количеством магии бандлинга мы должны быть в состоянии писать в JSX, чтобы сделать работу с ним немного более удобной.

### Гидратация

Итак, термин, который, как вы увидите, умные люди часто используют в интернете.

- Частичная гидратация
- Прогрессивная гидратация
- добавляйте больше по мере нахождения таких способов и т. д.

Проще говоря, это привязка интерактивности к элементу DOM с помощью _существующего_ состояния/эффектов/событий.

Это _существующее_ состояние/эффекты/события могут быть отправлены с сервера, но если вы работаете с компонентом, который может обрабатывать свое собственное состояние, и логика хорошо содержится в нем, вы просто монтируете компонент в DOM с необходимыми привязками.

В качестве примера это может выглядеть примерно так:

```js
// client.js
import { hydrate } from 'preact'
import Counter from './Counter'

const main = () => {
  // предполагаем, что сервер отрендерил компонент со следующим идентификатором:
  const container = document.getElementById('counter')
  hydrate(h(Counter, {}), container)
}

main()
```

Аналогично фазе серверного рендеринга, мы используем вспомогательную функцию из `preact`, чтобы помочь гидратировать компонент. Вы можете использовать `render`, но тогда фактический элемент уже будет чем-то, что было отрендерено сервером, и повторный рендеринг не имеет смысла, поэтому мы просто просим Preact попытаться добавить необходимые события и данные состояния.

То, что я объяснил выше, называется частичной гидратацией, поскольку вы не гидратируете всё приложение, а только определённые его части.

## Вглубь

Вам не нужно больше ничего знать, чтобы понять, как создать приложение на основе архитектуры островов, но давайте теперь перейдем к реализации этого.

# Код

Архитектура на уровне кода для этого очень похожа на большинство моделей SSR, и Vite имеет хорошее объяснение того, как написать свой собственный SSR с помощью Vite.

[&rarr; Документация Vite - Серверный рендеринг](https://vitejs.dev/guide/ssr.html)

Мы использовали webpack вместо этого, чтобы сделать его немного более многословным, что легче объяснить.

> Примечание: Вы можете получить упомянутый код из [barelyhuman/preact-islands-diy](http://github.com/barelyhuman/preact-islands-diy/)

## `server/app.js`

Начнем с файла `server/app.js`. Если у вас есть кодовая база, открытая локально, она пригодится при чтении.

Ниже приведённый фрагмент кода только подчеркивает необходимые области:

```js
import preactRenderToString from 'preact-render-to-string'
import HomePage from '../pages/HomePage.js'
import { h } from 'preact'
import { withManifestBundles } from '../lib/html.js'

const app = express()

app.get('/', async (req, res) => {
  res.send(
    withManifestBundles({
      body: preactRenderToString(h(HomePage, {})),
    })
  )
})
```

Если посмотреть на импорты, у нас те же импорты, что и в разделе [Первые шаги](#getting-started), и не так много изменилось.

Единственное дополнение здесь — это вспомогательная функция `withManifestBundles`, о которой мы поговорим далее.

## `lib/html.js`

Хелпер HTML различается в разных вариантах шаблона, но мы будем рассматривать только версию для `webpack`, которая находится в основной ветке.

Основное назначение этой вспомогательной функции — иметь возможность просматривать манифест JSON, который перечисляет, какие файлы упаковываются с помощью webpack, и их хэшированные пути при использовании в производстве.

Это необходимо, поскольку мы не будем знать хэш, и нам нужен программный способ его узнать.

Этот манифест генерируется клиентской конфигурацией webpack, которую мы рассмотрим через минуту.

```js
// получаем манифест из клиентского вывода
import manifest from '../../dist/js/manifest.json'

export const withManifestBundles = ({ styles, body }) => {
  // проходимся по каждому ключу в манифесте и создаем
  // тег скрипта для каждого.
  const bundledScripts = Object.keys(manifest).map(key => {
    const scriptPath = `/public/js/${manifest[key]}`
    return `<script src=${scriptPath}></script>`
  })

  return `<html lang="en">
    <head>
      <meta charset="UTF-8" />
      <meta name="viewport" content="width=device-width, initial-scale=1.0" />
      <style id="_goober">
        ${styles}
      </style>
    </head>

    <body>
      ${body}
    </body>
    ${bundledScripts.join('')}
  </html>`
}
```

Как объясняется в комментариях, мы просто берем все необходимые файлы из манифеста и вставляем их в виде тегов скриптов в финальный HTML, который отправляется с сервера.

Теперь перейдем к конфигурации, которая делает возможным сборку этого.

## `webpack.config.*.js`

Я старался сделать конфигурацию webpack как можно более минималистичной, чтобы не отпугнуть людей от всей идеи, поэтому давайте рассмотрим конфигурацию:

```js
// webpack.config.server.js
const path = require('path')
const nodeExternals = require('webpack-node-externals')

module.exports = {
  mode: process.env.NODE_ENV != 'production' ? 'development' : 'production',
  target: 'node',
  entry: path.resolve(__dirname, './src/server/app.js'),
  output: {
    filename: 'server.js',
    path: path.resolve(__dirname, './dist'),
  },
  stats: 'errors-warnings',
  resolve: {
    extensions: ['.js', '.jsx'],
  },
  module: {
    rules: [{ test: /\.jsx?$/, loader: 'babel-loader' }],
  },
  externals: [nodeExternals()],
}
```

Большинство из них не требуют объяснений, и единственный загрузчик, который у нас есть, — это `babel-loader`, поскольку мы используем решение CSS-IN-JS для стилизации.

Здесь нет ничего магического, мы просто указываем точку входа `server/app.js` и позволяем ему собираться в ту же папку, что и выходные данные клиента.

Теперь перейдем к конфигурации клиентской стороны, которая добавляет несколько дополнительных вещей, помимо простого указания точки входа и получения выходных данных.

Это сокращено, чтобы объяснить соответствующие части:

```js
// webpack.config.client.js

const entryPoints = glob
  .sync(path.resolve(__dirname, './src/client') + '/**/*.js', {
    absolute: true,
  })
  .reduce((acc, path) => {
    const entry = path.match(/[^\/]+\.jsx?$/gm)[0].replace(/.jsx?$/, '')
    acc[entry] = path
    return acc
  }, {})
```

Итак, первый раздел в основном находит все файлы в `src/client` и создает объект записей для webpack.

Пример: если `src/client/app.client.js` — это файл, то вывод выше будет следующим:

```json
{
  "app.client": "./src/client/app.client.js"
}
```

Здесь ничего особенного, это просто то, как webpack ожидает, чтобы записи были определены.

Всё остальное — это общая конфигурация, которая также присутствует на стороне сервера:

```js
{
  plugins: [
    new WebpackManifestPlugin({
      publicPath: '',
      basePath: '',
      filter: file => {
        return /\.mount\.js$/.test(file.name)
      },
    }),
  ]
}
```

Затем у нас есть плагин манифеста, который проверяет файлы, содержащие строку `mount` в своем имени. Это делается для того, чтобы убедиться, что загружаются только файлы входа, а не случайные файлы, и мы делаем это, указывая конкретный тип расширения для файла.

Некоторые фреймворки используют папку `islands`, чтобы отделить острова от файлов входа. Мы, вместо этого, отделяем файлы входа от островов и позволяем пользователю решать, что монтируется как остров, а что нет.

Вышеупомянутый `WebpackManifestPlugin` генерирует файл `manifest.json` в `dist/public/js`, который содержит имена упакованных файлов, которые мы использовали в файле `lib/html.js`.

## `.babelrc`

Это последняя часть конфигурации, где вы просите Babel убедиться, что используемый им JSX-движок — это Preact, а не React.

Довольно самоочевидно, но если вам нужны подробности об этой опции, пожалуйста, ознакомьтесь с документацией [babel](https://babeljs.io/) и [@babel/plugin-transform-react-jsx](https://babeljs.io/docs/en/babel-plugin-transform-react-jsx).

```json
// .babelrc
{
  "plugins": [
    [
      "@babel/plugin-transform-react-jsx",
      { "runtime": "automatic", "importSource": "preact" }
    ]
  ]
}
```

# Папки

Теперь мы можем перейти к обсуждению значения каждой папки здесь.

> **Примечание**: Пожалуйста, имейте в виду, что вы можете смешивать и сочетать папки по мере необходимости, просто убедитесь, что конфигурации отредактированы для обработки внесённых вами изменений. Если нет, текущая структура достаточно хороша для большинства приложений.

## `client`

Папка `src/client` в этой основной ветке используется для написания кода точки монтирования, который отправляется с отрендеренным HTML.

Вы добавляете выборочное монтирование в зависимости от страниц и селекторов, которые вы хотите использовать. Хотя это может загружать несколько JS-файлов, эти файлы не должны содержать ничего, кроме кода монтирования; ваши острова должны быть самодостаточными и независимыми. Тем не менее, вы можете отправить начальный набор данных с сервера в виде атрибута `data-*`, но это должны быть сериализуемые данные, иначе они будут потеряны.

Вы также можете добавить обёртку для создания острова вручную, но веб-компоненты не имеют широкой поддержки, поэтому, если вы используете их для системы с поддержкой устаревших технологий, вам лучше монтировать вручную, как упоминалось выше.

Пример:

```js
// src/client/index.mount.js

import { h, hydrate } from 'preact'

// настройка goober
import { setup } from 'goober'
setup(h)

// может быть перемещено в файл утилит и использоваться оттуда,
// в этом файле как пример на данный момент.
const mount = async (Component, elm) => {
  if (elm?.dataset?.props) {
    const props = JSON.parse(elm.dataset.props)
    delete elm.dataset.props
    hydrate(<Component {...props} />, elm)
  }
}

const main = async () => {
  // ленивая загрузка и повторное монтирование счётчика как клиентского компонента, если это необходимо
  // Лучший способ — проверить, существует ли элемент `counter` в
  // DOM, прежде чем импортировать компонент, чтобы избежать ненужных
  // загрузок JS.

  const Counter = (await import('../components/Counter.js')).default
  mount(Counter, document.getElementById('counter'))
}

main()
```

## `components`

Название довольно самоочевидно, поскольку мы не проводим никакой сегрегации между тем, что является островом, а что нет. Вы можете разместить все свои компоненты здесь, как обычно.

## `layouts`

Эти элементы разделены, поскольку мне нравится держать макеты подальше от компонентов, так как иногда у них есть не только условия рендеринга. Это не требуется в этом конкретном случае, потому что в большинстве случаев вы будете запускать свои макеты на сервере, а не на клиенте.

## `lib`

Содержит общие вспомогательные функции как для клиента, так и для сервера, поскольку оба они упаковываются отдельно, и зависимости будут встроены по мере необходимости.

## `pages`

Эта папка служит хранилищем для шаблонов. Таким образом, все, что сервер будет рендерить как страницу, помещается сюда. Возможность использовать макеты и другие компоненты, как в обычном приложении Preact, помогает в создании составных шаблонов, но все же проще держать их отдельно от фактического кода компонентов.

## `public`

Вещи, которые необходимо доставить статически через Express, помещаются сюда; Webpack заботится о копировании всего этого в конечную папку.

## `server`

Самоочевидно, файлы на стороне сервера. В большинстве случаев вы захотите переместить маршруты в отдельный файл и, возможно, добавить промежуточные программные модули, чтобы добавить вспомогательную функцию для рендеринга компонентов Preact для вас.

Что-то подобное определённо является частью сервера и не будет находиться на стороне клиента, поэтому просто держите это в этой папке.

Пример:

```js
app.use((req, res, next) => {
  res.render = (comp, data) => {
    return res.write(preactRenderToString(h(comp, { ...data })))
  }
})

// и где-то ещё в приложении

const handler = (req, res) => {
  return res.status(200).render(Homepage, { username: 'reaper' })
}
```

На самом деле, это весь код, который способствует настройке вашей собственной частичной гидратации / гидратации в стиле островов с использованием Node.js.

Большинство из этого можно достичь с помощью почти всех бандлеров, и небольшие изменения в том, как генерируются конфигурации, могут помочь вам достичь аналогичного опыта разработки (DX) с Astro, хотя вам будет лучше использовать Astro, если вы не любите поддерживать конфигурации.