---
title: Усиление сигналов
date: 2022-09-24
authors:
  - Joachim Viide
---

# Усиление сигналов

В новом выпуске Preact Signals существенно обновлены характеристики основ реактивной системы. Читайте далее, чтобы узнать, какие приемы мы использовали для этого.

Недавно мы [анонсировали](https://twitter.com/jviide/status/1572570215350964224) новые версии пакетов Preact Signals:

- [@preact/signals-core](https://www.npmjs.com/package/@preact/signals-core) 1.2.0 для общей функциональности ядра
- [@preact/signals](https://www.npmjs.com/package/@preact/signals) 1.1.0 для привязок Preact
- [@preact/signals-react](https://www.npmjs.com/package/@preact/signals-react) 1.1.0 для привязок React

В этом посте будут описаны шаги, которые мы предприняли для оптимизации **@preact/signals-core**. Это пакет, который служит основой для привязок, специфичных для платформы, но его также можно использовать независимо.

Сигналы — это подход команды Preact к реактивному программированию. Если вы хотите получить краткое представление о том, что такое _сигналы_ и как они связаны с Preact, вам поможет [сообщение в блоге с анонсом Signals](/blog/introducing-signals). Для более глубокого погружения ознакомьтесь с [официальной документацией](/guide/v10/signals).

Следует отметить, что ни одно из этих понятий нами не придумано. Реактивное программирование имеет достаточно долгую историю и уже получило широкое распространение в мире JavaScript благодаря [Vue.js](https://vuejs.org/), [Svelte](https://svelte.dev/), [SolidJS](https://www.solidjs.com/), [RxJS](https://rxjs.dev/) и многим другим, перечислять которые не имеет смысла. Всем им огромное спасибо!

## Ураганный тур по ядру сигналов

Начнем с обзора основных функций пакета **@preact/signals-core**.

В приведённых ниже фрагментах кода используются функции, импортированные из пакета. Операторы импорта показываются только в том случае, если в систему вводится новая функция.

### Сигналы

Простые _сигналы_ — это фундаментальные корневые величины, на которых базируется наша реактивная система. В других библиотеках они могут называться, например, «observables» ([MobX](https://mobx.js.org/observable-state.html), [RxJS](https://rxjs.dev/guide/observable)) или «refs» ([Vue](https://vuejs.org/guide/extras/reactivity-in-depth.html#how-reactivity-works-in-vue)). Команда Preact приняла термин «signal», используемый в [SolidJS](https://www.solidjs.com/tutorial/introduction_signals).

Сигналы представляют собой произвольные значения JavaScript, обернутые в реактивную оболочку. Вы предоставляете сигналу начальное значение, а в дальнейшем можете считывать и обновлять его по мере выполнения.

```js
// --repl
import { signal } from '@preact/signals-core';

const s = signal(0);
console.log(s.value); // Консоль: 0

s.value = 1;
console.log(s.value); // Консоль: 1
```

Сами по себе сигналы не представляют особого интереса до тех пор, пока не будут объединены с двумя другими примитивами — _вычисляемыми сигналами_ и _эффектами_.

### Вычисляемые сигналы

_Вычисляемые сигналы_ получают новые значения из других сигналов с помощью _вычисляемых функций_.

```js
// --repl
import { signal, computed } from '@preact/signals-core';

const s1 = signal('Привет');
const s2 = signal('мир');

const c = computed(() => {
  return s1.value + ', ' + s2.value;
});
```

Вычисляемая функция, переданная в `computed(...)`, не будет запущена немедленно. Это связано с тем, что вычисляемые сигналы оцениваются _лениво_, т. е. только при считывании их значений.

```js
// --repl
import { signal, computed } from '@preact/signals-core';

const s1 = signal('Привет');
const s2 = signal('мир');

const c = computed(() => {
  return s1.value + ', ' + s2.value;
});
// --repl-before
console.log(c.value); // Консоль: Привет, мир
```

Вычисленные значения также _кэшируются_. Их вычисляемые функции потенциально могут быть очень дорогими, поэтому мы хотим запускать их повторно только в тех случаях, когда это важно. Работающая вычисляемая функция отслеживает, какие значения сигналов действительно считываются во время её выполнения. Если ни одно из значений не изменилось, то повторный расчёт можно не производить. В приведённом выше примере мы можем просто использовать ранее вычисленное `c.value` до бесконечности, пока `a.value` и `b.value` остаются неизменными. Именно для облегчения отслеживания этой _зависимости_ и требуется в первую очередь обернуть примитивные значения в сигналы.

```js
// --repl
import { signal, computed } from '@preact/signals-core';

const s1 = signal('Привет');
const s2 = signal('мир');

const c = computed(() => {
  return s1.value + ', ' + s2.value;
});

console.log(c.value); // Консоль: Привет, мир
// --repl-before
// s1 и s2 не изменились, пересчёт не требуется
console.log(c.value); // Консоль: Привет, мир

s2.value = 'тьма, мой старый друг';

// s2 изменилось, поэтому вычисляемая функция запускается снова
console.log(c.value); // Консоль: Привет, тьма, мой старый друг
```

Как водится, вычисляемые сигналы сами являются сигналами. Вычисляемый сигнал может зависеть от других вычисляемых сигналов.

```js
// --repl
import { signal, computed } from '@preact/signals-core';
// --repl-before
const count = signal(1);
const double = computed(() => count.value * 2);
const quadruple = computed(() => double.value * 2);

console.log(quadruple.value); // Консоль: 4
count.value = 20;
console.log(quadruple.value); // Консоль: 80
```

Набор зависимостей не должен оставаться статичным. Вычисляемый сигнал будет реагировать только на изменения в последнем наборе зависимостей.

```js
// --repl
import { signal, computed } from '@preact/signals-core';
// --repl-before
const choice = signal(true);
const funk = signal('Группа');
const purple = signal('герой');

const c = computed(() => {
  if (choice.value) {
    console.log(funk.value, 'крови');
  } else {
    console.log('Звезда', purple.value);
  }
});
console.log(c.value); // Консоль: Группа крови

purple.value = 'по имени солнце'; // purple не является зависимостью, поэтому
console.log(c.value); // эффект не работает

choice.value = false;
console.log(c.value); // Консоль: Звезда по имени солнце

funk.value = 'Последний'; // funk больше не является зависимостью, поэтому
console.log(c.value); // эффект не работает
```

Эти три вещи — отслеживание зависимостей, ленивые вычисления и кэширование — являются общими чертами библиотек реактивности. В Vue _вычисляемые свойства_ являются [одним из ярких примеров](https://dev.to/linusborg/vue-when-a-computed-property-can-be-the-wrong-tool-195j).

### Эффекты

Вычисляемые сигналы хорошо поддаются [чистым функциям](https://ru.wikipedia.org/wiki/%D0%A7%D0%B8%D1%81%D1%82%D0%BE%D1%82%D0%B0_%D1%84%D1%83%D0%BD%D0%BA%D1%86%D0%B8%D0%B8) без побочных эффектов. Они также ленивы. Что же делать, если мы хотим реагировать на изменения значений сигналов, не опрашивая их постоянно? Эффекты в помощь!

Как и вычисляемые сигналы, эффекты также создаются с помощью функции (_effect_) и также отслеживают свои зависимости. Однако вместо того, чтобы быть ленивыми, эффекты являются _нетерпеливыми_. Функция эффекта запускается сразу при создании эффекта, а затем снова и снова при изменении значений зависимостей.

```js
// --repl
import { signal, computed, effect } from '@preact/signals-core';

const count = signal(1);
const double = computed(() => count.value * 2);
const quadruple = computed(() => double.value * 2);

effect(() => {
  console.log('четырехкратное значение теперь равно', quadruple.value);
}); // Консоль: четырехкратное значение теперь равно 4

count.value = 20; // Консоль: четырехкратное значение теперь равно 80
```

Эти реакции вызываются _уведомлениями_. Когда простой сигнал изменяется, он оповещает об этом свои зависимости. Они, в свою очередь, уведомляют свои зависимости и т. д. Как это [принято][https://mobx.js.org/computeds.html] в реактивных системах, вычисляемые сигналы по пути следования уведомления отмечают, что они устарели и готовы к повторному вычислению. Если уведомление доходит до какого-либо эффекта, то этот эффект планирует свой запуск, как только завершатся все ранее запланированные эффекты.

Когда вы закончили работу с эффектом, вызовите _disposer_, который был возвращен при первом создании эффекта:

```js
// --repl
import { signal, computed, effect } from '@preact/signals-core';
// --repl-before
const count = signal(1);
const double = computed(() => count.value * 2);
const quadruple = computed(() => double.value * 2);

const dispose = effect(() => {
  console.log('четырехкратное значение теперь равно', quadruple.value);
}); // Консоль: четырехкратное значение теперь равно 4

dispose();
count.value = 20; // ничего не выводится
```

Существуют и другие функции, например [`batch`](/guide/v10/signals/#batchfn), но эти три являются наиболее значимыми для последующих замечаний по внедрению.

# Замечания по внедрению

Когда мы взялись за реализацию более производительных версий описанных выше примитивов, нам пришлось искать быстрые способы выполнения всех следующих подзадач:

- Отслеживание зависимостей: Вести учёт используемых сигналов (простых или вычисляемых). Зависимости могут меняться динамически.
- Ленивые вычисления: Вычисляемые функции должны выполняться только по требованию.
- Кэширование: Вычисляемый сигнал должен повторно вычисляться только в том случае, если его зависимости могли измениться.
- Стремление: Эффект должен запускаться как можно скорее, когда что-то в его цепочке зависимостей меняется.

Реактивная система может быть реализована миллиардом различных способов. Первая выпущенная версия **@preact/signals-core** была основана на коллекциях (_Sets_), поэтому мы и дальше будем использовать этот подход для сравнения и сопоставления.

### Отслеживание зависимостей

Всякий раз, когда функция вычисления/эффекта начинает вычисление, ей требуется способ перехвата сигналов, считанных во время её работы. Для этого вычисляемый сигнал или эффект устанавливает себя в качестве текущего _оценочного контекста_. При чтении свойства `.value` сигнала вызывается [геттер](https://developer.mozilla.org/ru/docs/Web/JavaScript/Reference/Functions/get). Геттер добавляет сигнал в качестве зависимости, _источника_, контекста оценки. Контекст также добавляется как зависимая _цель_ сигнала.

В итоге сигналы и эффекты всегда имеют актуальное представление о своих зависимостях и зависимых элементах. Затем каждый сигнал может уведомлять свои зависимые элементы об изменении своего значения. Эффекты и вычисляемые сигналы могут ссылаться на свои наборы зависимостей, чтобы отписаться от этих уведомлений, когда, например, эффект утилизируется.

![Сигналы и эффекты всегда имеют актуальное представление о своих зависимостях (источниках) и зависимых объектах (целях)](/signals/signal-boosting-01.png)

Один и тот же сигнал может быть прочитан несколько раз в одном и том же контексте оценки. В таких случаях было бы удобно сделать какую-то дедупликацию для зависимых и не зависимых записей. Нам также необходим способ обработки изменяющихся наборов зависимостей: чтобы либо перестраивать набор зависимостей при каждом запуске, либо инкрементально добавлять/удалять зависимости.

Коллекции [Set](https://developer.mozilla.org/ru/docs/Web/JavaScript/Reference/Global_Objects/Set) хорошо подходят для всего этого. Как и многие другие реализации, они использовались в исходной версии Preact Signals. Коллекции позволяют добавлять _и_ удалять элементы за [константное время O(1)](https://ru.wikipedia.org/wiki/%D0%92%D1%80%D0%B5%D0%BC%D0%B5%D0%BD%D0%BD%D0%B0%D1%8F_%D1%81%D0%BB%D0%BE%D0%B6%D0%BD%D0%BE%D1%81%D1%82%D1%8C_%D0%B0%D0%BB%D0%B3%D0%BE%D1%80%D0%B8%D1%82%D0%BC%D0%B0#%D0%9A%D0%BE%D0%BD%D1%81%D1%82%D0%B0%D0%BD%D1%82%D0%BD%D0%BE%D0%B5_%D0%B2%D1%80%D0%B5%D0%BC%D1%8F) (амортизировано), а также перебирать текущие элементы за [линейное время O(n)](https://ru.wikipedia.org/wiki/%D0%92%D1%80%D0%B5%D0%BC%D0%B5%D0%BD%D0%BD%D0%B0%D1%8F_%D1%81%D0%BB%D0%BE%D0%B6%D0%BD%D0%BE%D1%81%D1%82%D1%8C_%D0%B0%D0%BB%D0%B3%D0%BE%D1%80%D0%B8%D1%82%D0%BC%D0%B0#%D0%9B%D0%B8%D0%BD%D0%B5%D0%B9%D0%BD%D0%BE%D0%B5_%D0%B2%D1%80%D0%B5%D0%BC%D1%8F). Дубликаты также обрабатываются автоматически! Неудивительно, что многие системы реагирования используют преимущества наборов (или [Карт](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Map)). Правильный инструмент для работы и всё такое.

Однако нам было интересно, существуют ли какие-то альтернативные подходы. Создание коллекций может быть относительно дорогостоящим, и, по крайней мере, для вычисляемых сигналов могут потребоваться две отдельные коллекции: одна для зависимостей и одна для зависимых. Джейсон снова был _total Джейсоном_ и [проверял производительность](https://esbench.com/bench/6317fc2a6c89f600a5701bc9) для сравнения итерации коллекций и массивов. Будет много итераций, так что всё складывается.

![Итерация Set немного медленнее, чем итерация Array](/signals/signal-boosting-01b.png)

У коллекций также есть свойство: они повторяются в порядке вставки. И это круто — это как раз то, что нам понадобится позже, когда мы будем иметь дело с кэшированием. Но есть вероятность, что порядок не всегда останется прежним. Обратите внимание на следующий сценарий:

```js
// --repl
import { signal, computed } from '@preact/signals-core';
// --repl-before
const s1 = signal(0);
const s2 = signal(0);
const s3 = signal(0);

const c = computed(() => {
  if (s1.value) {
    s2.value;
    s3.value;
  } else {
    s3.value;
    s2.value;
  }
});
```

В зависимости от `s1` порядок зависимостей может быть `s1, s2, s3` или `s1, s3, s2`. Чтобы поддерживать коллекции в порядке, необходимо предпринять специальные шаги: либо удалить, а затем снова добавить элементы, очистить коллекцию перед запуском функции или создать новую для каждого запуска. Каждый подход может вызвать отток памяти. И всё это только для того, чтобы объяснить теоретический, но, вероятно, редкий случай изменения порядка зависимостей.

Есть множество других способов справиться с этим. Например, нумерация и последующая сортировка зависимостей. В итоге мы изучили [связанные списки](https://ru.wikipedia.org/wiki/%D0%A1%D0%B2%D1%8F%D0%B7%D0%BD%D1%8B%D0%B9_%D1%81%D0%BF%D0%B8%D1%81%D0%BE%D0%BA).

### Связные списки

Связные списки часто считаются довольно примитивными, но для наших целей они обладают некоторыми очень полезными свойствами. Если у вас есть узлы двусвязного списка, то следующие операции могут быть очень дешёвыми:

- Вставка элемента в один конец списка за время O(1)
- Удаление узла (на который у вас уже есть указатель) из любого места списка за время O(1)
- Перебор списка за время O(n) (O(1) на узел)

Оказывается, эти операции — всё, что нам нужно для управления списками зависимостей/зависимых.

Начнём с создания «исходного узла» для каждого отношения зависимости. Атрибут `source` узла указывает на сигнал, от которого зависит. Каждый узел имеет свойства `nextSource` и `prevSource`, указывающие на следующий и предыдущий исходные узлы в списке зависимостей соответственно. Эффекты или вычисляемые сигналы получают атрибут `sources`, указывающий на первый узел списка. Теперь мы можем перебирать зависимости, вставлять новую зависимость и удалять зависимости из списка для изменения порядка.

![Эффекты и вычисляемые сигналы сохраняют свои зависимости в двусвязном списке](/signals/signal-boosting-02.png)

Теперь давайте сделаем то же самое, но наоборот: для каждого зависимого создадим «целевой узел». Атрибут `target` узла указывает на зависимый эффект или вычисляемый сигнал. `nextTarget` и `prevTarget` создают двусвязный список. Обычный и вычисляемый сигнал получает атрибут `target` , указывающий на первый целевой узел в их зависимом списке.

![Сигналы хранят свои зависимости в двусвязном списке](/signals/signal-boosting-03.png)

Но эй, зависимости и зависимые бывают парами. Для каждого исходного узла **должен** существовать соответствующий целевой узел. Мы можем воспользоваться этим фактом и объединить «исходные узлы» и «целевые узлы» в просто «узлы». Каждый узел становится своего рода четырёхсвязным чудовищем, которое зависимый может использовать как часть своего списка зависимостей, и наоборот.

![Каждый узел становится своего рода четырёхсвязным чудовищем, которое зависимый может использовать как часть своего списка зависимостей, и наоборот](/signals/signal-boosting-04.png)

К каждому узлу могут быть прикреплены дополнительные элементы в целях подсчёта. Перед каждой функцией вычисления/эффекта мы перебираем предыдущие зависимости и устанавливаем флаг «неиспользуемый» для каждого узла. Мы также временно сохраняем узел в его свойстве `.source.node` для дальнейшего использования. После этого функция может начать свое выполнение.

Во время выполнения каждый раз, когда считывается зависимость, значения подсчёта можно использовать для определения того, наблюдалась ли эта зависимость уже во время этого или предыдущего запуска. Если зависимость связана с предыдущим запуском, мы можем перезапустить её узел. Для ранее невиданных зависимостей мы создаем новые узлы. Затем узлы перемешиваются, чтобы сохранить их в обратном порядке использования. В конце прогона мы снова просматриваем список зависимостей, очищая узлы, которые всё ещё болтаются с установленным флагом «неиспользуемый». Затем мы переворачиваем список оставшихся узлов, чтобы сохранить его в чистоте для последующего использования.

Этот деликатный танец смерти позволяет нам выделить только один узел для каждой пары, зависящей от зависимости, а затем использовать этот узел бесконечно, пока существуют отношения зависимости. Если дерево зависимостей остается стабильным, потребление памяти также остается стабильным после начальной фазы сборки. При этом списки зависимостей остаются актуальными и располагаются в порядке использования. При постоянном объёме работы O(1) на узел. Отлично!

### Нетерпеливые эффекты

Если позаботиться об отслеживании зависимостей, нетерпеливые эффекты относительно легко реализовать с помощью уведомлений об изменениях. Сигналы оповещают своих иждивенцев об изменении значений. Если сам зависимый сигнал является вычисляемым сигналом, имеющим зависимые элементы, он передает уведомление вперёд и так далее. Эффекты, которые сами получают расписание уведомлений для запуска.

Мы добавили сюда пару оптимизаций. Если получающая сторона уведомления уже была уведомлена ранее и у нее ещё не было возможности запуститься, то она не будет передавать уведомление вперед. Это уменьшает количество каскадных уведомлений, когда дерево зависимостей расширяется или сужается. Обычные сигналы также не уведомляют свои зависимые объекты, если значение сигнала фактически не меняется (например, `s.value = s.value`). Но это просто вежливость.

Чтобы эффекты могли планировать себя, необходим своего рода список запланированных эффектов. Мы добавили специальный атрибут `.nextBatchedEffect` к каждому экземпляру _Effect_, позволяя экземплярам _Effect_ выполнять двойную функцию в качестве узлов в односвязном списке планирования. Это уменьшает отток памяти, поскольку повторное планирование одного и того же эффекта не требует дополнительного выделения или освобождения памяти.

### Интерлюдия: Подписки на уведомления против GC

Мы не были полностью правдивы. Вычисляемые сигналы на самом деле _всегда_ не получают уведомления от своих зависимостей. Вычисляемый сигнал подписывается на уведомления о зависимостях только тогда, когда есть что-то, например эффект, прослушивающий сам сигнал. Это позволяет избежать проблем в таких ситуациях:

```js
const s = signal(0);

{
  const c = computed(() => s.value);
}
// c вышел из области видимости
```

Если бы `c` всегда подписывался на уведомления от `s`, тогда `c` не мог бы собирать мусор до тех пор, пока `s` тоже не выйдет из области видимости. Это потому, что `s` будет продолжать использовать ссылку на `c`.

Существует несколько решений этой проблемы, например, использование [WeakRefs](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/WeakRef) или требование удаления вычисляемых сигналов вручную. В нашем случае связные списки предоставляют очень удобный способ динамической подписки и отказа от подписки на уведомления о зависимостях на лету, благодаря всем этим возможностям O(1). Конечным результатом является то, что вам не нужно обращать особого внимания на висящие ссылки на вычисляемые сигналы. Мы посчитали, что это наиболее эргономичный и эффективный подход.

В тех случаях, когда вычисляемый сигнал **подписался** на уведомления, мы можем использовать эти знания для дополнительной оптимизации. Это приводит нас к ленивым вычислениям и кэшированию.

### Ленивые и кэшированные вычисляемые сигналы

Самый простой способ реализовать ленивый вычисляемый сигнал — просто пересчитывать каждый раз, когда его значение считывается. Хотя это было бы не очень эффективно. Вот где очень помогают кеширование и отслеживание зависимостей.

Каждый простой и вычисляемый сигнал имеет свой собственный _номер версии_. Они увеличивают номера версий каждый раз, когда замечают изменение собственного значения. Когда запускается вычисляемая функция, она сохраняет в узлах последние замеченные номера версий своих зависимостей. Мы могли бы выбрать сохранение предыдущих значений зависимостей в узлах вместо номеров версий. Однако, поскольку вычисляемые сигналы ленивы, они могут использовать устаревшие и потенциально дорогостоящие значения бесконечно. Поэтому мы посчитали, что нумерация версий — безопасный компромисс.

В итоге мы получили следующий алгоритм определения того, когда вычисляемый сигнал может взять выходной и повторно использовать свое кэшированное значение:

1. Если ни один сигнал нигде не изменил значения с момента последнего запуска, то выбросить и вернуть кэшированное значение.

> Каждый раз, когда изменяется простой сигнал, он также увеличивает _глобальный номер версии_, общий для всех простых сигналов. Каждый вычисляемый сигнал отслеживает последний глобальный номер версии, который он видел. Если глобальная версия не изменилась с момента последнего расчёта, то повторный расчёт можно пропустить досрочно. В любом случае в этом случае не может быть никаких изменений ни в одном вычисленном значении.

1. Если вычисляемый сигнал прослушивает уведомления, и не был уведомлён с момента последнего запуска, то выбросить и вернуть кэшированное значение

> Когда вычисляемый сигнал получает уведомление от своих зависимостей, он помечает кэшированное значение как устаревшее. Как описано ранее, вычисляемые сигналы не всегда получают уведомления. Но когда они это сделают, мы сможем этим воспользоваться.

1. Пересмотреть зависимости по порядку. Проверить их номера версий. Если при отсутствии зависимости номер версии изменился даже после переоценки, то выбросить и вернуть кэшированное значение.

> Именно этот шаг является причиной того, что мы с особой любовью и заботой следим за тем, чтобы зависимости находились в порядке их использования. Если зависимость изменится, мы не хотим повторно оценивать зависимости, которые появятся позже в списке, потому что это может быть просто ненужной работой. Кто знает, возможно, изменение этой первой зависимости приведёт к тому, что следующая вычисляемая функция отбросит последние зависимости.

1. Запустить функцию вычисления. Если возвращаемое значение отличается от кэшированного, то увеличить номер версии вычисляемого сигнала. Закэшировать и вернуть новое значение.

> Это крайняя мера! Но, по крайней мере, если новое значение равно кэшированному, номер версии не изменится, и зависимые в дальнейшем смогут использовать его для оптимизации собственного кэширования.

Последние два шага часто возвращаются к зависимостям. Вот почему предыдущие шаги предназначены для того, чтобы попытаться обойти рекурсию.

# Финал

В типичной манере Preact по пути было добавлено множество более мелких оптимизаций. [Исходный код](https://github.com/preactjs/signals/tree/main/packages/core/src) содержит некоторые комментарии, которые могут оказаться полезными, а могут и нет. Ознакомьтесь с [тестами](https://github.com/preactjs/signals/tree/main/packages/core/test), если вам интересно, какие крайние случаи мы придумали, чтобы обеспечить надежность нашей реализации.

Этот пост был своего рода свалкой мозгов. В нём изложены основные шаги, которые мы предприняли, чтобы сделать **@preact/signals-core** в версии 1.2.0 лучше — до некоторого определения слова «лучше». Надеемся, что некоторые из перечисленных здесь идей найдут отклик и будут повторно использованы и переработаны другими. По крайней мере, это мечта!

Огромное спасибо всем, кто внёс свой вклад. И спасибо вам, что дочитали до этого места! Это было путешествие.
