---
title: Предварительный рендеринг с помощью `@preact/preset-vite`
date: 2024-08-06
authors:
  - Ryan Christian
---

# Предварительный рендеринг с пресетом Vite

Прошло полгода с тех пор, как наш плагин для пререндеринга стал доступен в `@preact/preset-vite`, так что давайте поговорим о нем, немного о нашей истории и об экосистеме в целом.

Те, кто уже давно в нашем сообществе, знают, как сильно мы любим пререндеринг; Это была первоклассная функция в Preact-CLI, затем в WMR, а теперь и в нашем пресете Vite. Если все сделано правильно, это безболезненное дополнение к типичному SPA, которое значительно улучшает пользовательский опыт, и наш плагин пререндеринга призван способствовать именно этому.

## Что такое пререндеринг?

«Предрендеринг», в контексте этой статьи, — это акт генерации HTML из вашего приложения во время сборки с помощью рендеринга на стороне сервера (SSR); Иногда это также называют генерацией статических сайтов (SSG).

Хотя мы не будем углубляться в достоинства SSR или даже утверждать, что вам следует его использовать, в целом выгоднее отправлять пользователю полностью заполненный HTML-документ при первой навигации (и, возможно, при последующих, в зависимости от стратегии маршрутизации), а не пустую оболочку, в которую в конечном итоге превратится JS на стороне клиента. Пользователи быстрее получат доступ к документу и смогут начать пользоваться страницей (пусть и с урезанной функциональностью), пока JS будет загружаться в фоновом режиме.

## Наша история в космосе

С тех пор как Preact-CLI впервые вышел в свет в мае 2017 года, встроенный пререндер стал одним из основных инструментов сборки; Мы с радостью перенесли его в WMR в 2020 году, и нам и членам сообщества очень не хватало этого, когда мы перешли на предложение Vite.

Хотя каждая итерация немного отличается от другой, все они построены на одной и той же основной идее: пользователи с большей готовностью примут пререндеринг, чем проще его настройка, включая ограниченные изменения в существующей кодовой базе. В Preact-CLI это означало предоставление экспорта корневого компонента по умолчанию с некоторыми JSON-данными для его наполнения; В WMR, а теперь и в Vite, это означает экспорт простой функции `prerender()`, которая возвращает HTML для маршрута, при этом пререндер проходит через само приложение, заменяя необходимость в JSON на начальном этапе.

Любой, кто много работал с SSR в масштабе, знает, что существует гора сложностей, от которых невозможно полностью абстрагироваться, и мы не станем утверждать обратное. Тем не менее, почти все SPA обеспечивают лучший опыт, если они предварительно рендеризованы, поэтому мы хотим привлечь как можно больше пользователей — снижение барьера для входа показало себя очень успешным в нашем сообществе, так что это было ключевой частью нашей философии дизайна, чтобы стремиться к «drop-in», как можно больше.

## Существующая экосистема Vite

Прежде чем мы создали собственную реализацию пререндеринга для нашего пресета Vite, мы изучили существующую экосистему Vite, чтобы посмотреть, что предлагается, но не нашли в предложенных вариантах того, что нам нужно. Пререндеринг лучше всего работает, когда он максимально приближен к «drop-in», насколько это возможно, взять существующее приложение с минимальными изменениями и сгенерировать из него HTML, но существующие решения были ещё одним шагом на пути к «drop-in» и разделились на две основные категории:

1. Многочисленные сборки
	- Отдельные клиентские и серверные сборки, часто также отдельные точки входа
	- Меньше изоморфности, разные ветви в приложении для разных сред
2. Фреймворки / Обёртки Vite
	- Теперь Vite используется не напрямую, а как абстракция
	- Некоторое количество buy-in/lock-in
	- Матрица поддержки различных опций конфигурации Vite, плагинов и т. д. может быть сложной и не совсем понятной

Хотя у этих решений есть свои достоинства и свое место в экосистеме, ни одно из них не было настолько хорошо, насколько оно могло бы подойти для нашей экосистемы, учитывая наши исторические предложения в этой области. «Лучший сценарий» DX часто приносился в жертву более сложным или специфическим потребностям — и это вполне оправданный компромисс.

Однако для пререндеринга мы решили предложить нечто отличное от существующих вариантов, или, по крайней мере, что-то более привычное для наших пользователей.

## Реализация в `@preact/preset-vite`

Спустя годы мы всё ещё были влюблены в простоту и расширяемость пререндеринга WMR и чувствовали, что его очень не хватает в нашем пресете Vite, поэтому мы решили перенести его с небольшими изменениями, чтобы устранить имеющиеся недостатки. Немного работы и вуаля, пререндеринг через плагин!

Для начала приведем пример пререндеринга «Hello World». Приложение.

> Подсказка: Наш инициализатор Vite (`$ npm create preact`) может настроить его для вас вместе с несколькими другими дополнительными опциями, такими как маршрутизация, TypeScript и т. д. Если вы хотите попробовать наш пререндеринг, это самый быстрый способ войти в курс дела.

Во-первых, включите пререндеринг, установив `prerender: { enabled: true }` в опциях плагина `@preact/preset-vite`:

```diff
// vite.config.js
import { defineConfig } from 'vite';
import preact from '@preact/preset-vite';

// https://vitejs.dev/config/
export default defineConfig({
	plugins: [
		preact({
+			prerender: { enabled: true }
		}),
	],
});
```

...затем добавьте атрибут `prerender` к скрипту, содержащему нашу функцию `prerender()` — это позволит плагину узнать, где её найти. Хотя вы можете установить этот параметр для любого скрипта, который вам нравится, в наших примерах он всегда будет находиться в корне приложения.

```diff
// index.html
-<script type="module" src="/src/index.jsx"></script>
+<script prerender type="module" src="/src/index.jsx"></script>
```

...Наконец, внесите пару изменений в корень нашего приложения:

1. Переключите `render` на `hydrate`.
	* `hydrate` из `preact-iso` — это очень маленькая утилита, которая решает, рендерить ли приложение или гидратировать, в зависимости от того, может ли она найти существующую разметку в документе. В dev он будет использовать `render`, но в production, с предварительно отрендеренным HTML, он будет использовать `hydrate`.
	* Нам нужно добавить проверку объекта `window` (`typeof window !== undef`), чтобы гарантировать, что мы не пытаемся получить доступ к `document`, глобальному браузеру, в Node во время SSR.

2. Добавьте наш экспорт `prerender()`.
	* Это средство пререндеринга, и оно полностью контролируется пользователем. Вы решаете, как должно выглядеть ваше приложение, какие реквизиты передавать корневому компоненту, вносите любые изменения в HTML, выполняете любую постобработку и т. д. Всё, что нужно плагину, это вернуть объект, содержащий свойство `html` с вашей HTML-строкой.
	* В наших примерах мы будем использовать prerender из `preact-iso`, который представляет собой тонкую оболочку `renderToStringAsync` из `preact-render-to-string` с одним главным преимуществом: он автоматически собирает и возвращает относительные ссылки, которые он находит на страницах, которые вы предварительно визуализируете. Плагин предварительного рендеринга затем может использовать эти ссылки для «прогулки» по вашему приложению, самостоятельно обнаруживая страницы. Мы покажем это позже.

```diff
// src/index.jsx
-import { render } from 'preact';
+import { hydrate, prerender as ssr } from 'preact-iso';

function App() {
    return <h1>Привет, мир!</h1>
}

-render(<App />, document.getElementById('app'));
+if (typeof window !== 'undefined') {
+	hydrate(<App />, document.getElementById('app'));
+}

+export async function prerender(data) {
+    return await ssr(<App {...data} />)
+}
```

При такой настройке у вас будет приложение с предварительным рендерингом. Однако ни одно приложение не бывает таким простым, поэтому давайте рассмотрим несколько более сложных примеров.

### Пример полного API

```jsx
// src/index.jsx

// ...

export async function prerender(data) {
	const { html, links: discoveredLinks } = ssr(<App />);

	return {
		html,
		// По желанию добавьте дополнительные ссылки, которые должны быть
		// пререндеренные (если они еще не были — они будут дедупированы)
		links: new Set([...discoveredLinks, '/foo', '/bar']),
		// Опционально настройте и добавьте элементы в `<head>`
		// пререндеренного HTML-документа
		head: {
			// Устанавливает значение атрибута "lang": `<html lang="en">`
			lang: 'en',
			// Устанавливает заголовок текущей страниці: `<title>My cool page</title>`
			title: 'My cool page',
			// Устанавливает любые дополнительные элементы, которые вы хотите включить в `<head>`:
			//   <link rel="stylesheet" href="foo.css">
			//   <meta property="og:title" content="Social media title">
			elements: new Set([
				{ type: 'link', props: { rel: 'stylesheet', href: 'foo.css' } },
				{ type: 'meta', props: { property: 'og:title', content: 'Social media title' } }
			])
		}
	}
}
```

### Изоморфное извлечение контента с помощью выборки на основе ожидания

```jsx
// src/use-fetch.js
import { useState } from "preact/hooks";

const cache = new Map();

async function load(url) {
	const res = await fetch(url);
	if (res.ok) return await res.text();
	throw new Error(`Failed to fetch ${url}!`);
}

// Простой механизм выборки на основе ожидания с кэшированием
export function useFetch(url) {
	const [_, update] = useState({});

	let data = cache.get(url);
	if (!data) {
		data = load(url);
		cache.set(url, data);
		data.then(
			(res) => update((data.res = res)),
			(err) => update((data.err = err)),
		);
	}

	if (data.res) return data.res;
	if (data.err) throw data.err;
	throw data;
}
```

```jsx
// src/index.jsx
import { hydrate, prerender as ssr } from 'preact-iso';
import { useFetch } from './use-fetch.js';

function App() {
    return (
        <div>
            <Suspense fallback={<p>Loading...</p>}>
                <Article />
            </Suspense>
        </div>
    );
}

function Article() {
	const data = useFetch("/my-local-article.txt");
	return <p>{data}</p>;
}

if (typeof window !== 'undefined') {
	hydrate(<App />, document.getElementById('app'));
}

export async function prerender(data) {
    return await ssr(<App {...data} />)
}
```

### Использование `globalThis` для передачи данных

```js
// src/title-util.js
import { useEffect } from 'preact/hooks';

/**
 * Настройка `document.title` или `globalThis.title`
 * @param {string} title
 */
export function useTitle(title) {
	if (typeof window === 'undefined') {
		globalThis.title = createTitle(title);
	}
	useEffect(() => {
		if (title) {
			document.title = createTitle(title);
		}
	}, [title]);
}
```

```jsx
// src/index.jsx
import { LocationProvider, Router, hydrate, prerender as ssr } from 'preact-iso';

import { useTitle } from './title-util.js'

function App() {
    return (
        <LocationProvider>
            <main>
                <Home path="/" />
                <NotFound default />
            </main>
        </LocationProvider>
    );
}

function Home() {
    useTitle('Preact - Home');
    return <h1>Привет, мир!</h1>;
}

function NotFound() {
    useTitle('Preact - 404');
    return <h1>Page Not Found</h1>;
}

if (typeof window !== 'undefined') {
    hydrate(<App />, document.getElementById('app'));
}

export async function prerender(data) {
    const { html, links } = await ssr(<App {...data} />);

    return {
        html,
        links,
        head: {
            title: globalThis.title,
            elements: new Set([
                { type: 'meta', props: { property: 'og:title', content: globalThis.title } },
            ])
        }
    };
}
```

Хотя этот паттерн встречается реже, вы также можете использовать его разновидность для инициализации и передачи данных пререндера вглубь приложения, минуя необходимость в глобальном хранилище/контексте.

```jsx
// src/some/deep/Component.jsx
function MyComponent({ myFetchData }) {
	const [myData, setMyData] = useState(myFetchData || 'some-fallback');
	...
}
```

```js
let initialized = false;
export async function prerender(data) {
    const init = async () => {
        const res = await fetch(...);
        if (res.ok) globalThis.myFetchData = await res.json();

        initialized = true;
    }
    if (!initialized) await init();

    const { html, links } = await ssr(<App {...data} />);
    ...
}
```

---

Для любопытных, спрашивающих «Как все это работает?», все можно разбить на три простых шага:

1. Настройка

	Мы устанавливаем скрипт с экспортированной вами функцией `prerender()` в качестве дополнительного входа и говорим Rollup сохранять сигнатуры записей, что позволяет нам получить доступ и вызвать эту функцию после сборки.
2. Сборка

	Мы позволяем Vite создавать ваше приложение, как обычно: компиляция JSX, запуск плагинов, оптимизация ресурсов и т. д.
3. Пререндеринг

  На этапе `generateBundle` плагина мы начинаем генерировать HTML. Начиная с `/`, мы начинаем выполнять собранные JS-пакеты в Node, вызываем функцию `prerender()` и вставляем HTML, который она возвращает, в документ `index.html`, а затем записываем результат в указанную директорию вывода. Все новые ссылки, которые возвращает функция `prerender()`, ставятся в очередь для последующей обработки.

  Пререндеринг завершается, когда у нас заканчиваются URL-адреса для передачи в ваше приложение.

После этого Vite продолжит завершать процесс сборки, запуская все остальные плагины, которые у вас есть. Ваше приложение с предварительным рендерингом будет доступно сразу же, без последующих сборок и скриптов.

### Некоторые интересные функции

- Реализация `fetch()` на базе файловой системы (как показано в примере «Изоморфная выборка»)
	- Прежде чем бежать за вилами, выслушайте нас! Во время пререндеринга (и только во время пререндеринга) мы исправляем `fetch()`, чтобы разрешить чтение файлов непосредственно из файловой системы. Это позволяет использовать статические файлы (текст, JSON, Markdown и т. д.) во время пререндеринга без необходимости запускать сервер для их использования. При предварительной визуализации можно использовать те же пути к файлам, что и в браузере.
	- Фактически, именно так мы создали ту самую страницу, которую вы сейчас читаете! `fetch('/content/blog/preact-prerender.json')`, который срабатывает при переходе на эту страницу, примерно переводится как `new Response(await fs.readFile('/content/blog/preact-prerender.json'))` при пререндеринге. Мы считываем файл, оборачиваем его в `Response`, чтобы имитировать сетевой запрос, и отправляем его обратно в ваше приложение. Ваше приложение может использовать тот же самый запрос `fetch()` во время пререндеринга и на клиенте.
	- В паре с suspense и асинхронной реализацией SSR получается действительно отличный DX.
- Ползучие ссылки
	- Частично поддерживаемый пользовательским экспортом функции `prerender()`, частично — плагином, вы можете вернуть набор ссылок при пререндеринге страницы (`preact-iso` делает это удивительно просто), которые будут добавлены в список URL для пререндеринга плагина. Это позволит плагину просматривать ваш сайт во время сборки, находя все больше и больше страниц для естественного пререндеринга.
	- Вы также можете предоставлять ссылки вручную через опции плагина или добавляя некоторые из них к тем, которые возвращает `preact-iso`, как мы показали выше в примере полного API. Это особенно полезно для страниц ошибок, таких как `/404`, на которые, возможно, не будет ссылок, но вы всё равно хотите, чтобы они были пререндерены.

...и, возможно, самое большое преимущество:

- Управление путём переключения булева значения в вашем конфигурационном файле
	- Поскольку мы не являемся обёрткой, и вам не нужно изменять свой исходный код для его поддержки (кроме некоторых проверок окон), нет никакой блокировки. Если вы решите отойти от дел или захотите провести тестирование на выходе, достаточно будет перевернуть булеву функцию, и вы вернетесь к обычному SPA с Vite.
	- Как мы уже неоднократно упоминали, пререндеринг показывает себя с лучшей стороны, когда он максимально приближен к «drop-in», как можно больше, и это включает в себя возможность вернуться назад по своей прихоти. Для нас важно, чтобы вы могли переходить от SPA к пререндерингу и обратно с минимальными усилиями.

## Итоговые замечания

Команда Vite, вероятно, хотела бы, чтобы мы упомянули, что этот плагин вносит небольшой патч в сгенерированный клиентский код, и что они (команда Vite) не обязательно одобряют запуск пакетов браузера в Node.

Патч, о котором идет речь, выглядит следующим образом:

```diff
// src/node/plugins/importAnalysisBuild.ts
-if (__VITE_IS_MODERN__ && deps && deps.length > 0) {,
+if (__VITE_IS_MODERN__ && deps && deps.length > 0 && typeof window !== 'undefined') {,
	 const links = document.getElementsByTagName('link')
	 ...
```

Поскольку попытка выполнить `document.getElementsByTagName` приведет к ошибке в Node, где нет `document`, мы просто добавляем дополнительное условие в предзагрузчик, чтобы он не делал попыток запуска в Node, и всё. Просто частичное изменение этой единственной строки, которая в любом случае не имеет особого смысла во время предварительного рендеринга.

Мы очень, очень довольны таким уровнем риска и уже некоторое время активно используем его без каких-либо проблем, но это в некоторой степени использование инструмента сверх того, для чего он был предназначен, и мы хотим об этом раскрыть.

Для всех, кто не пользуется Preact, хорошие новости: наш плагин полностью не зависит от платформы! Чтобы облегчить использование в любой другой среде, альтернативно предлагается как [`vite-prerender-plugin`](https://npm.im/vite-prerender-plugin). Та же функциональность и синхронизация с `@preact/preset-vite`, но отсутствуют другие утилиты, специфичные для Preact, которые поставляются в предустановленном плагине Preact.
