---
title: Знакомство с сигналами
date: 2022-09-06
authors:
  - Marvin Hagemeister
  - Jason Miller
---

# Знакомство с сигналами

Сигналы — это способ выражения состояния, который гарантирует, что приложения будут работать быстро, независимо от того, насколько они сложны. Сигналы основаны на реактивных принципах и обеспечивают отличную эргономику для разработчиков благодаря уникальной реализации, оптимизированной для Virtual DOM.

По своей сути сигнал — это объект со свойством `.value`, который содержит некоторое значение. Доступ к свойству значения сигнала изнутри компонента автоматически обновляет этот компонент при изменении значения этого сигнала.

Помимо простоты и легкости написания, это также обеспечивает быстрое обновление состояния независимо от количества компонентов приложения. Сигналы по умолчанию являются быстрыми, автоматически оптимизируя для вас обновления за кулисами.

```jsx
// --repl
import { render } from 'preact';
// --repl-before
import { signal, computed } from '@preact/signals';

const count = signal(0);
const double = computed(() => count.value * 2);

function Counter() {
  return (
    <button onClick={() => count.value++}>
      {count} x 2 = {double}
    </button>
  );
}
// --repl-after
render(<Counter />, document.getElementById('app'));
```

Сигналы, в отличие от хуков, могут использоваться как внутри, так и вне компонентов. Сигналы также отлично сочетаются с хуками **_и_** классовыми компонентами, что позволяет внедрять их в удобном для вас темпе и использовать уже имеющиеся знания. Опробуйте их на нескольких компонентах и постепенно внедряйте с течением времени.

И, кстати, мы остаемся верны своим корням, предлагая вам самые маленькие библиотеки из возможных. Использование сигналов в Preact увеличивает размер пакета всего на **1,6 кБ**.

Если вы хотите сразу приступить к работе, перейдите к нашей [документации](/guide/v10/signals), чтобы получить более подробную информацию о сигналах.

## Какие задачи решаются с помощью сигналов?

За последние годы мы работали над широким спектром приложений и команд: от небольших стартапов до монолитов, в которых одновременно принимали участие сотни разработчиков. За это время все члены основной команды заметили повторяющиеся проблемы с управлением состоянием приложения.

Для решения этих проблем были созданы фантастические решения, но даже самые лучшие из них всё равно требуют ручной интеграции во фреймворк. В результате мы наблюдаем нерешительность разработчиков в принятии этих решений, предпочитающих строить свои приложения с использованием примитивов состояния, предоставляемых фреймворком.

Мы создали Signals как привлекательное решение, сочетающее в себе оптимальную производительность и эргономику для разработчиков с бесшовной интеграцией с инфраструктурой.

## Проблема глобальных состояний

Состояние приложения обычно начинается с малого и простого, возможно, с нескольких простых хуков `useState`. По мере роста приложения и увеличения числа компонентов, которым требуется доступ к одному и тому же фрагменту состояния, это состояние в конечном итоге поднимается к общему компоненту-предку. Эта схема повторяется несколько раз, пока большая часть состояния не окажется вблизи корня дерева компонентов.

![Изображение, показывающее, как глубина дерева компонентов напрямую влияет на производительность рендеринга при использовании стандартных обновлений состояния.](/signals/state-updates.png)

Этот сценарий представляет собой проблему для традиционных фреймворков на основе Virtual DOM, которые должны обновлять всё дерево, затронутое недействительностью состояния. По сути, производительность рендеринга зависит от количества компонентов в этом дереве. Мы можем обойти эту проблему, мемоизируя части дерева компонентов с помощью `memo` или `useMemo`, чтобы фреймворк получал те же самые объекты. Если ничего не изменилось, это позволяет фреймворку пропустить отрисовку некоторых частей дерева.

В теории это звучит разумно, но на деле часто оказывается гораздо сложнее. На практике, по мере роста кодовых баз, становится трудно определить, где следует разместить эти оптимизации. Зачастую даже хорошо продуманная мемоизация оказывается неэффективной из-за нестабильных значений зависимостей. Поскольку хуки не имеют явного дерева зависимостей, которое можно было бы проанализировать, инструментальные средства не могут помочь разработчикам диагностировать **_причины_** нестабильности зависимостей.

## Контекстный хаос

Другой распространённый обходной путь, к которому прибегают команды при разделении состояний, заключается в том, чтобы поместить состояние в контекст. Это позволяет сократить время рендеринга, потенциально пропуская рендеринг для компонентов между поставщиком контекста и потребителями. Однако здесь есть одна загвоздка: может быть обновлено только значение, переданное поставщику контекста, и только в целом. Обновление свойства объекта, выставляемого через контекст, не приводит к обновлению потребителей этого контекста — гранулярное обновление невозможно. Возможные варианты решения этой проблемы — разделение состояния на несколько контекстов или аннулирование объекта контекста путем его клонирования при изменении любого из его свойств.

![Контекст может пропускать обновление компонентов до тех пор, пока из него не будет считано значение. Затем снова приступают к мемоизации.](/signals/context-chaos.png)

Поначалу перемещение значений в контекст кажется выгодным решением, но со временем возникают проблемы, связанные с увеличением размера дерева компонентов только для обмена значениями. Бизнес-логика неизбежно оказывается в зависимости от множества значений контекста, что может заставить её реализовывать в определённом месте дерева. Добавление компонента, подписывающегося на контекст, в середину дерева является дорогостоящим, так как уменьшает количество компонентов, которые могут быть пропущены при обновлении контекста. Более того, все компоненты, находящиеся под подписчиком, теперь должны быть выведены заново. Единственным решением этой проблемы является активное использование мемоизации, что возвращает нас к проблемам, присущим мемоизации.

## В поисках лучшего способа управления состоянием

Мы вернулись к чертёжной доске в поисках примитива состояния нового поколения. Мы хотели создать нечто такое, что одновременно решало бы проблемы существующих решений. Ручная интеграция фреймворков, чрезмерная зависимость от мемоизации, неоптимальное использование контекста и отсутствие программной наблюдаемости ощущаются как обратные явления.

Разработчикам необходимо «согласиться» на производительность с помощью этих стратегий. Что, если бы мы могли изменить ситуацию и предоставить систему, которая была бы **быстрой по умолчанию**, обеспечивая наилучшую производительность, от которой вам придется отказаться?

Наш ответ на эти вопросы – Signals. По умолчанию это быстрая система, не требующая запоминания или каких-то трюков в вашем приложении. Сигналы предоставляют преимущества детального обновления состояния независимо от того, является ли это состояние глобальным, передаваемым через входные параметры или контекст или локальным для компонента.

## Сигналы в будущее

Основная идея сигналов заключается в том, что вместо передачи значения непосредственно через дерево компонентов мы передаем объект сигнала, содержащий это значение (аналогично `ref`). Когда значение сигнала меняется, сам сигнал остается прежним. В результате сигналы можно обновлять без повторной визуализации компонентов, через которые они прошли, поскольку компоненты видят сигнал, а не его значение. Это позволяет нам пропустить всю дорогостоящую работу по рендерингу компонентов и сразу перейти к конкретным компонентам в дереве, которые фактически получают доступ к значению сигнала.

![Сигналы могут продолжать пропускать сравнение Virtual DOM, независимо от того, в каком месте дерева к ним осуществляется доступ](/signals/signals-update.png)

Мы используем тот факт, что граф состояний приложения обычно намного меньше, чем дерево его компонентов. Это приводит к более быстрому рендерингу, поскольку для обновления графа состояний требуется гораздо меньше работы по сравнению с деревом компонентов. Эта разница наиболее очевидна при измерении в браузере — на снимке экрана ниже показана трассировка DevTools Profiler для одного и того же приложения, измеренная дважды: один раз с использованием хуков в качестве примитива состояния, а второй раз с использованием сигналов:

![Сравнение профилирования обновлений Virtual DOM и обновлений с помощью сигналов, которые обходят почти все различия Virtual DOM](/signals/virtual-dom-vs-signals-update.png)

Версия сигналов значительно превосходит механизм обновления любой традиционной инфраструктуры на основе Virtual DOM. В некоторых приложениях, которые мы тестировали, сигналы работают настолько быстрее, что их вообще становится трудно найти во флеймграфе.

Сигналы меняют представление о производительности: вместо того, чтобы соглашаться на производительность с помощью мемоизации или селекторов, сигналы по умолчанию быстрые. При использовании сигналов производительность исключается (без использования сигналов).

Чтобы достичь такого уровня производительности, сигналы были построены на следующих ключевых принципах:

- **Ленивые по умолчанию:** отслеживаются и обновляются только те сигналы, которые в данный момент где-то используются. Отключенные сигналы не влияют на производительность.
- **Оптимальные обновления.** Если значение сигнала не изменилось, компоненты и эффекты, использующие это значение сигнала, не будут обновляться, даже если зависимости сигнала изменились.
- **Оптимальное отслеживание зависимостей.** Фреймворк отслеживает, что сигнализирует о том, что всё зависит от вас — никаких массивов зависимостей, таких как хуки.
- **Прямой доступ.** Доступ к значению сигнала в компоненте автоматически подписывается на обновления без необходимости использования селекторов или хуков.

Эти принципы делают сигналы хорошо подходящими для широкого спектра вариантов использования, даже для сценариев, которые не имеют ничего общего с рендерингом пользовательских интерфейсов.

## Перенос сигналов в Preact

Определив правильный примитив состояния, мы приступили к его подключению к Preact. Что нам всегда нравилось в хуках, так это то, что их можно использовать непосредственно внутри компонентов. Это эргономическое преимущество по сравнению со сторонними решениями по управлению состоянием, которые обычно полагаются на функции «селектора» или оборачивают компоненты в специальную функцию для подписки на обновления состояния.

```js
// Подписка на основе селектора :(
function Counter() {
  const value = useSelector((state) => state.count);
  // .
}

// Подписка на основе функции-обертки :(
const counterState = new Counter();

const Counter = observe((props) => {
  const value = counterState.count;
  // .
});
```

Ни один из подходов не показался нам удовлетворительным. Селекторный подход требует заключения всего доступа к состоянию в селектор, что становится утомительным для сложных или вложенных состояний. Подход к обертыванию компонентов в функцию требует ручных усилий по обёртыванию компонентов, что приводит к множеству проблем, таких как отсутствие имен компонентов и статических свойств.

За последние несколько лет у нас была возможность тесно сотрудничать со многими разработчиками. Одна из распространённых проблем, особенно для новичков в (p)react, заключается в том, что такие понятия, как селекторы и оболочки, являются дополнительными парадигмами, которые необходимо изучить, прежде чем чувствовать себя продуктивно с каждым решением по управлению состоянием.

В идеале нам не нужно было бы знать о селекторах или функциях-обертках, и мы могли бы просто получать доступ к состоянию непосредственно внутри компонентов:

```jsx
// Представьте, что это какое-то глобальное состояние, доступ к которому нужен всему приложению:
let count = 0;

function Counter() {
  return <button onClick={() => count++}>Значение: {count}</button>;
}
```

Код понятен и легко понять, что происходит, но, к сожалению, он не работает. Компонент не обновляется при нажатии кнопки, поскольку невозможно узнать, что `count` изменился.

Однако мы не могли выкинуть этот сценарий из головы. Что мы могли бы сделать, чтобы такая ясная модель стала реальностью? Мы начали создавать прототипы различных идей и реализаций, используя [подключаемый модуль рендеринга](/guide/v10/options). Это заняло время, но в конце концов мы пришли к тому, как это осуществить:

```jsx
// --repl
import { render } from 'preact';
import { signal } from '@preact/signals';
// --repl-before
// Представьте, что это какое-то глобальное состояние, доступ к которому нужен всему приложению:
const count = signal(0);

function Counter() {
  return <button onClick={() => count.value++}>Значение: {count.value}</button>;
}
// --repl-after
render(<Counter />, document.getElementById('app'));
```

Здесь нет ни селекторов, ни функций-обёрток, ничего. Доступа к значению сигнала достаточно, чтобы компонент знал, что ему необходимо обновиться при изменении значения этого сигнала. После тестирования прототипа в нескольких приложениях стало ясно, что мы чего-то добились. Написание кода таким способом казалось интуитивно понятным и не требовало какой-либо умственной гимнастики, чтобы всё работало оптимально.

## Можем ли мы пойти ещё быстрее?

Мы могли бы остановиться на этом и выпустить сигналы как есть, но это команда Preact: нам нужно было посмотреть, насколько далеко мы сможем продвинуть интеграцию Preact. В приведённом выше примере счётчика значение `count` используется только для отображения текста, что на самом деле не требует повторной визуализации всего компонента. Вместо автоматического повторного рендеринга компонента при изменении значения сигнала, что, если мы перерисовываем только текст? А ещё лучше: что, если мы полностью обойдем Virtual DOM и обновим текст непосредственно в DOM?

```jsx
const count = signal(0);

// Вместо этого:
<p>Значение: {count.value}</p>

// … мы можем передать сигнал непосредственно в JSX:
<p>Значение: {count}</p>

// … или даже передать их как свойства DOM:
<input value={count} onInput={.} />
```

Так что да, мы тоже это сделали. Вы можете передать сигнал непосредственно в JSX в любом месте, где обычно используется строка. Значение сигнала будет отображаться как текст и автоматически обновляться при изменении сигнала. Это также работает для атрибута.

## Следующие шаги

Если вам интересно и вы хотите сразу приступить к делу, перейдите к нашей [документации](/guide/v10/signals). Нам бы хотелось узнать, как вы собираетесь их использовать.

Помните, что не нужно спешить с переходом на сигналы. Хуки будут по-прежнему поддерживаться, и они также отлично работают с сигналами! Мы рекомендуем постепенно опробовать сигналы, начиная с нескольких компонентов, чтобы привыкнуть к концепциям.
