---
layout: 2ColLeft
title: Common Errors
sidebar: nav.html
---

{% include anchor.html class="h3" title="No 'Access-Control-Allow-Origin' header" hash="no_access_control_allow_origin_header" %}

If you see the error:

> XMLHttpRequest cannot load [...]
> No 'Access-Control-Allow-Origin' header is present on the requested resource.
> Origin [...] is therefore not allowed access.

or this one:

> Cross-Origin Request Blocked: The Same Origin Policy disallows reading the remote resource at http://[couchDBIP]:[couchDBPort]/[dbname]/?_nonce=[request hash]. This can be fixed by moving the resource to the same domain or enabling CORS

it's because you need to enable [CORS](https://developer.mozilla.org/en-US/docs/Web/HTTP/Access_control_CORS) on CouchDB/Cloudant/whatever you're using. Otherwise, your scripts can only access the server database if they're served from the same origin &#8212; the protocol (ex: _http://_, _https://_), domain, and port number must match.

You can enable CORS in CouchDB using `curl` or the Futon web interface, but we've saved you some time by making a Node script called [add-cors-to-couchdb](https://github.com/pouchdb/add-cors-to-couchdb). Just run:

{% highlight bash %}
$ npm install -g add-cors-to-couchdb
$ add-cors-to-couchdb
{% endhighlight %}

Or if your database is not at `127.0.0.1:5984`:

{% highlight bash %}
$ add-cors-to-couchdb http://me.example.com \
    -u myusername -p mypassword
{% endhighlight %}

You can check that CORS is now enabled by visiting [http://localhost:5984/_utils/config.html](http://localhost:5984/_utils/config.html) in your browser. You should see something like this:

{% include img.html src="cors_in_couchdb.png" alt="CORS settings in CouchDB" %}

{% include anchor.html class="h3" title="PouchDB throws a `No valid adapter found` error" hash="no_valid_adapter" %}

Reading from/writing to a local database from an `iframe` with a different origin will cause PouchDB to throw a `No valid adapter found` error in Firefox. This is due to Firefox's IndexedDB implementation.

IndexedDB has a same-origin restriction. Read/write operations from another origin will always fail, but only Firefox triggers a `No valid adapter found` error. Chrome / Opera will instead throw an [`UnknownError`](#unknown_error_chrome).

{% include anchor.html class="h3" title="iOS/Safari: \"there was not enough remaining storage space\"" hash="not_enough_space" %}

On iOS and Safari, if you expect your app to use more than 5MB of space, you will need to request the space up-front from the user.  In certain versions of Safari, you can never request more than what the user originally grants you.

{% include img.html src="safari_popup.png" alt="Safari storage quota popup" %}

To get around this, when you create your PouchDB, use the `opts.size` option for the expected _maximum_ size of the database in MB.  Valid increments are 10, 50, 100, 500, and 1000.  For instance, if you request 50, then Safari will show a popup saying "allow 50MB?" but if you request 51, then Safari will show a popup saying "allow 100MB?".

If you don't use the `size` option, then you'll be able to use up to 5MB without any popup, but then once you use more, there will be a popup asking for 10.

```js
new PouchDB('mydb', {size: 10}); // request 10 MB with a popup
new PouchDB('mydb', {size: 50}); // request 50 MB with a popup
new PouchDB('mydb'); // implicitly request 5 MB, no popup until you exceed 5MB
```

This does not affect any backend other than Web SQL. Chrome, Android, and Opera do not show the popup. On PhoneGap/Cordova apps, you can also use the [SQLite plugin][sqlite] to get around this problem. Here's [more information about storage quotas](http://www.html5rocks.com/en/tutorials/offline/quota-research) and [details on the Safari/iOS bug](https://github.com/pouchdb/pouchdb/issues/2347).

{% include anchor.html class="h3" title="PouchDB throws 404 (Object Not Found) for '_local' document" hash="404__local_document" %}

Don't worry, nothing is amiss, this is expected behaviour:
During PouchDB's initial replication PouchDB will check for a checkpoint, if it doesn't exist a 404 will be returned and a checkpoint will subsequently be written.

{% include anchor.html class="h3" title="PouchDB is throwing `InvalidStateError`" hash="throwing_invalidstateerror" %}

Are you in private browsing mode? IndexedDB is [disabled in private browsing mode](https://developer.mozilla.org/en-US/docs/IndexedDB/Using_IndexedDB) in Firefox.

{% include anchor.html class="h3" title="Can't open second database on Android WebView with Cordova/Phone Gap" hash="phonegap_cordova_second_database" %}

There is a limit of one database per app in some versions of the Android WebView. Install the [SQLite plugin][sqlite], then PouchDB will use that if it is available.

{% include anchor.html class="h3" title="Possible EventEmitter memory leak detected" hash="event_emitter_limit" %}

If you see this warning:

    (node) warning: possible EventEmitter memory leak detected. 11 listeners added.
    Use emitter.setMaxListeners() to increase limit.

This is because PouchDB uses Node-style [EventEmitters](https://nodejs.org/api/events.html) for its events. An EventEmitter is any object that has an `.on()` or `once()` method, such as `db.changes().on('change', ...`.

By default, all EventEmitters have 10 listeners, and if you exceed that limit, e.g. by attaching many `changes()` listeners or running many simultaneous `replicate()` or `sync()` events, then you may exceed this limit.

**This could indicate a memory leak in your code**. Check to make sure that you are calling `cancel()` on any `changes()`, `replicate()`, or `sync()` handlers, if you are constantly starting and stopping those events.

If you're sure it's not a memory leak, though, you can increase the limit by doing:

{% highlight javascript %}
db.setMaxListeners(20);  // or 30 or 40 or however many you need
{% endhighlight %}

In the above example, `db` refers to a database object you created using `new PouchDB('dbname')`.

{% include anchor.html class="h3" title="Database size limitation of ~5MB on iOS with Cordova/Phone Gap" hash="size_limitation_5mb" %}

If you're storing large amounts of data, such as PNG attachments, the [SQLite plugin][sqlite] is again your friend. (See [issue #1260](https://github.com/pouchdb/pouchdb/issues/1260) for details.)

{% include anchor.html class="h3" title="CouchDB returns a 404 for GETs from a CouchApp" hash="404_get_couchapp" %}

Certain URL rewrites are broken by PouchDB's cache-busting; try adding `{cache : false}` to the PouchDB constructor. (See [issue #1233](https://github.com/pouchdb/pouchdb/issues/1233) for details.)

{% include anchor.html class="h3" title="Uncaught TypeError: object is not a function" hash="typeerror_object_is_not_a_function" %}

Did you include the [es6-promise shim library](https://github.com/jakearchibald/es6-promise)?  Not every browser implements ES6 Promises correctly. (See [issue #1747](https://github.com/pouchdb/pouchdb/issues/1747) for details.)

{% include anchor.html class="h3" title="SyntaxError: Parse error (Cordova on Android)" hash="cordova_android_parse_error" %}

Did you include the [es5-shim library][es5shim]?  PouchDB is written in ES5, which is supported by modern browsers, but requires shims for older browsers (e.g. IE 9, Android 2.1 WebView).

In Android, if you're loading PouchDB directly via `webView.loadUrl('javascript://' + js')`, you should prefer the minified PouchDB javascript file to the unminified one, since code comments can also cause parse errors.

{% include anchor.html class="h3" title="PouchDB object fails silently (Safari)" hash="safari_object_silent_fail" %}

Safari requires users to confirm that they want to allow an app to store data locally ("Allow this website to use space on your disk?").  If PouchDB is loaded in an `iframe` or some other unusual way, the dialog might not be shown, and the database will silently fail.

{% include anchor.html class="h3" title="window.localStorage is not available (Chrome apps)" hash="window_localstorage_chrome_apps" %}

In Chrome apps, you'll see the warning "window.localStorage is not available in packaged apps. Use chrome.storage.local instead."  This is harmless; since PouchDB doesn't use localStorage if it's not available.

{% include anchor.html class="h3" title="Error: UnknownError (Firefox)" hash="unknown_error_ff" %}

Are you using a webserver to host and run your code? This error can be caused by running your script/file locally using the `file:///` setting in Firefox, since Firefox does not [allow access to IndexedDB locally](https://bugzilla.mozilla.org/show_bug.cgi?id=643318). You can use the SimpleHTTPServer to deploy your script by running `python -m SimpleHTTPServer` from the directory containing the script, or use the Apache webserver and then access the script by using `http://localhost/{path_to_your_script}`.

{% include anchor.html class="h3" title="Error: UnknownError (Chrome / Opera)" hash="unknown_error_chrome" %}

This can occur when attempting to read from or write to IndexedDB from a different origin. IndexedDB has a same-origin restriction. Attempting to write to the database associated with _http://example.com_ from an `iframe` served from _http://api.example.com_, for example, will fail.

In Firefox, PouchDB instead throws a [`No valid adapter found`](#no_valid_adapter) error.

{% include anchor.html class="h3" title="DataCloneError: An object could not be cloned" hash="could_not_be_cloned" %}

If you ever see:

    Uncaught DataCloneError:
      Failed to execute 'put' on 'IDBObjectStore':
      An object could not be cloned.

Or:

    DataCloneError: The object could not be cloned.

Then the problem is that the document you are trying to store is not a pure JSON object. For example, an object with its own class (`new Foo()`) or with special methods like getters and setters cannot be stored in PouchDB/CouchDB.

If you are ever unsure, then run this on the document:

```js
JSON.parse(JSON.stringify(myDocument));
```

If the object you get out is the same as the object you put in, then you are storing the right kind of object.

Note that this also means that you cannot store `Date`s in your document. You must convert them to strings or numbers first. `Date`s will be stored as-is in IndexedDB, but in the other adapters and in CouchDB, they will be automatically converted to ISO string format, e.g. `'2015-01-01T12:00:00.000Z'`. This can caused unwanted results. See [#2351](https://github.com/pouchdb/pouchdb/issues/2351) and [#2158](https://github.com/pouchdb/pouchdb/issues/2158) for details.

{% include anchor.html class="h3" title="DOM Exception 18 in Android pre-Kitkat WebView" hash="android_pre_kitkat" %}

This applies to hybrid apps designed to run in Android pre-Kitkat (i.e. before 4.4).

If you are directly using a `WebView` and not using Cordova/PhoneGap, you will probably either run into an error where PouchDB silently fails or you see `Error: SECURITY_ERR: DOM Exception 18` in the console. As a sanity test, you can run this JavaScript:

```js
openDatabase('mydatabase', 1, 'mydatabase', 5000000, function (db) { console.log('it works!'); });
```

If you see "it works" in the console, then everything's peachy. Otherwise there are a few things you have to do.

First, make sure Web SQL is enabled on your `WebView` in the first place using [setDatabaseEnabled](http://developer.android.com/reference/android/webkit/WebSettings.html#setDatabaseEnabled%28boolean%29):

```java
myWebView.getSettings().setDatabaseEnabled(true);
```

Second, specify a path for the database. Yes, you need to do this, even though it's deprecated in Kitkat:

```java
String databasePath = getContext().getApplicationContext().getDir(
  "database", Context.MODE_PRIVATE).getPath();
webView.getSettings().setDatabasePath(databasePath);
```

Third, you'll need to set an `onExceededDatabaseQuota` handler. Yes, it's also deprecated in Kitkat. Yes, you still need to do it.

```java
webView.setWebChromeClient(new WebChromeClient() {

  @Override
  public void onExceededDatabaseQuota(String url, String databaseIdentifier, long currentQuota, long estimatedSize,
                                      long totalUsedQuota, WebStorage.QuotaUpdater quotaUpdater) {
    quotaUpdater.updateQuota(estimatedSize * 2);
  }
});
```

If you skip any one of these three steps, then you will get the `DOM Exception 18` error. You need to do all three.

Alternatively, you can also load the `WebView` with a fake `http://` URL, but this may cause other errors when you try to fetch files based on a relative path:

```java
webView.loadDataWithBaseURL("http://www.example.com",
    htmlContent,
    "text/html",
    "utf-8",
    null);
```

{% include anchor.html class="h3" title="Replication with attachments is slow or fails" hash="replicating_attachments_slow" %}

The symptoms for this issues are:

1. Replicating a database that has many attachments from a CouchDB server is either slow or fails randomly.
2. You get server error message of the nature `No buffer space available`.

Chances are that your server runs inside a virtual machine. The host system, or hypervisor, imposes limits on how much data each virtual machine can use for networking. If you are on a cheap virtual server, it is possible, that the default settings for PouchDB pull-replication (10 parallel batches of 100 documents each) exhaust the narrow limit of your server. Even a single client can cause this.

The solution is to move to a better server, but if that is not an immediate option, a workaround would be reducing the `options.batch_size` and `options.batches_limit` [replication options]({{ site.baseurl }}/api.html#replication).

To find optimal values, start by setting them both to 1 (meaning that PouchDB should download one document after the other) and increase from there and stop when the symptoms begin again. Note that multiple concurrent clients can still cause an issue, if you get too many. If all your documents have one or more attachments (e.g. a photos database), setting both options to `1` is probably a good idea.

{% include alert/start.html variant="info" %}

Generally, reducing these options that replicating the database down will take more time. Please test various settings to see what works for you and your hardware.

{% include alert/end.html %}

{% include alert/start.html variant="info" %}

This issue has been found on OpenVZ systems, but other Hypervisors might also be affected. See
<a
  href="http://blog.aplikacja.info/2010/01/105-no-buffer-space-available-on-openvz-vps/"
  target="_blank"
>http://blog.aplikacja.info/2010/01/105-no-buffer-space-available-on-openvz-vps/</a>
on how to diagnose this issue.

{% include alert/end.html %}

[es5shim]: https://github.com/es-shims/es5-shim
[sqlite]: https://github.com/brodysoft/Cordova-SQLitePlugin

{% include anchor.html class="h3" title="Packaging PouchDB in an app with WebPack" hash="package_pouchdb_webpack" %}

PouchDB may have various dependencies that may not play nicely with WebPack. Here are some issues you may run into and their resolutions:

**You may need an appropriate loader to handle this file type.**

If you run into the following error (or similar):

```sh
ERROR in ./~/pouchdb/~/levelup/package.json
Module parse failed: /path/to/node_modules/pouchdb/node_modules/levelup/package.json Line 2: Unexpected token :
You may need an appropriate loader to handle this file type.
| {
|   "name": "levelup",
|   "description": "Fast & simple storage - a Node.js-style LevelDB wrapper",
|   "version": "0.18.6",
 @ ./~/pouchdb/~/levelup/lib/util.js 102:30-56

```
WebPack needs to be configured to recognize how to load json files. Simply, install `json-loader` and edit `webpack.config.js` as follows: 

```js
module: {
    loaders: [
        // https://github.com/pouchdb/pouchdb/issues/3319
        {
            test: /\.json$/,
            loader: "json-loader"
        }
    ]
}
```

{% include anchor.html class="h3" title="Failed to load resource: the server responded with a status of 400 (Bad request) " hash="couchbase_dbname" %}

If you are using Couchbase Lite to sync with PouchDB then you cannot use capital letters in your database name as Couchbase Lite has restrictions on valid database names.

{% include anchor.html class="h3" title="Live changes pass undetected (Web Extension)" hash="webextension_live_changes" %}

When using PouchDB in a WebExtension (at least in Chromium 55 and Firefox 50), but apparently only if the `manifest.json` contains the `store` permission, a [live changes feed]({{ site.baseurl }}/guides/changes.html#live-changes-feed) in one tab or background process may not detect changes made in another tab/process. It will of course still report those changes upon the next change that it does detect.
Minimal code to reproduce this can be found [here](https://gist.github.com/Treora/150dca4b57b1b881bd049303e82c5ced).

{% include anchor.html class="h3" title="PouchDB install errors on Windows" hash="pouchdb_install_errors_windows" %}

Sometimes `npm install pouchdb` fails on windows, when no prebuilt binary is available. The error looks similar to this:

```sh
C:\XXX\node_modules\project_name>if not defined npm_config_node_gyp (node "C:\XXX\node_modules\npm\bin\node-gyp-bin\....\node_modules\node-gyp\bin\node-gyp.js" rebuild ) else (node "" rebuild )
gyp ERR! configure error
gyp ERR! stack Error: Can't find Python executable "python", you can set the PYTHON env variable.
gyp ERR! stack at PythonFinder.failNoPython (C:\XXX\nodejs\node_modules\npm\node_modules\node-gyp\lib\configure.js:483:19)
```

or something like this

```sh
gyp ERR! configure error
npm ERR! code ELIFECYCLE
npm ERR! errno 1
npm ERR! leveldown@3.0.0 install: `prebuild-install || node-gyp rebuild`
npm ERR! Exit status 1
npm ERR!
npm ERR! Failed at the leveldown@3.0.0 install script.
```

If you are on windows and getting any error messages mentioning **leveldown** and **Python**, please attempt the steps below.

Node-Gyp requires Python 2.7, it does not work with Python 3.x unfortunately.

Here are a few steps you can take:

1. try `npm install --global --production windows-build-tools` (this command needs to be ran with admin privileges)
2. read other [pointers to using node-gyp on Windows here](https://github.com/nodejs/node-gyp#on-windows)
3. If you are only using PouchDB in the browser, you can install [pouchdb-browser](https://www.npmjs.com/package/pouchdb-browser) instead: `npm install --save pouchdb-browser`
