#ifndef __PDDL__DETAIL__AST_COPY_H
#define __PDDL__DETAIL__AST_COPY_H

#include <pddl/AST.h>
#include <pddl/Variant.h>

namespace pddl
{
namespace ast
{

////////////////////////////////////////////////////////////////////////////////////////////////////
//
// ASTCopy
//
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// Primitives
////////////////////////////////////////////////////////////////////////////////////////////////////

inline Constant deepCopy(Constant &other);
inline PrimitiveType deepCopy(PrimitiveType &other);
inline Variable deepCopy(Variable &other);

////////////////////////////////////////////////////////////////////////////////////////////////////
// Expressions: Base Classes
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Derived, class ArgumentLeft, class ArgumentRight = ArgumentLeft>
inline Derived deepCopy(Binary<Derived, ArgumentLeft, ArgumentRight> &other);
template<class Derived, class Argument>
inline Derived deepCopy(NAry<Derived, Argument> &other);
template<class Derived, class Argument>
inline Derived deepCopy(Quantified<Derived, Argument> &other);

////////////////////////////////////////////////////////////////////////////////////////////////////
// Expressions
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Argument>
inline At<Argument> deepCopy(At<Argument> &other);
template<class Argument>
inline Not<Argument> deepCopy(Not<Argument> &other);

////////////////////////////////////////////////////////////////////////////////////////////////////
// Variants
////////////////////////////////////////////////////////////////////////////////////////////////////

inline ast::Type deepCopy(ast::Type &other);

////////////////////////////////////////////////////////////////////////////////////////////////////
// Unique Pointers
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class T>
std::unique_ptr<T> deepCopy(std::unique_ptr<T> &other);

////////////////////////////////////////////////////////////////////////////////////////////////////
// Primitives
////////////////////////////////////////////////////////////////////////////////////////////////////

Constant deepCopy(Constant &other)
{
	return Constant(other.declaration);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

PrimitiveType deepCopy(PrimitiveType &other)
{
	return PrimitiveType(other.declaration);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

Variable deepCopy(Variable &other)
{
	return Variable(other.declaration);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Expressions: Base Classes
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Derived, class ArgumentLeft, class ArgumentRight>
Derived deepCopy(Binary<Derived, ArgumentLeft, ArgumentRight> &other)
{
	auto argumentLeft{deepCopy(other.argumentLeft)};
	auto argumentRight{deepCopy(other.argumentRight)};

	return Binary<Derived, ArgumentLeft, ArgumentRight>(std::move(argumentLeft), std::move(argumentRight));
}

////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Derived, class Argument>
Derived deepCopy(NAry<Derived, Argument> &other)
{
	typename Derived::Arguments arguments;
	arguments.reserve(other.arguments.size());

	for (auto &argument : other.arguments)
		arguments.emplace_back(deepCopy(argument));

	return Derived(std::move(arguments));
}

////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Derived, class Argument>
Derived deepCopy(Quantified<Derived, Argument> &other)
{
	auto argument{deepCopy(other.argument)};

	return Quantified<Derived, Argument>(std::move(argument));
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Expressions
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Argument>
At<Argument> deepCopy(At<Argument> &other)
{
	auto argument{deepCopy(other.argument)};

	return At<Argument>(other.timePoint, std::move(argument));
}

////////////////////////////////////////////////////////////////////////////////////////////////////

template<class Argument>
Not<Argument> deepCopy(Not<Argument> &other)
{
	auto argument{deepCopy(other.argument)};

	return Not<Argument>(std::move(argument));
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Variants
////////////////////////////////////////////////////////////////////////////////////////////////////

ast::Type deepCopy(ast::Type &other)
{
	return other.match([](auto &x) -> ast::Type {return deepCopy(x);});
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Unique Pointers
////////////////////////////////////////////////////////////////////////////////////////////////////

template<class T>
std::unique_ptr<T> deepCopy(std::unique_ptr<T> &other)
{
	return std::make_unique<T>(deepCopy(*other));
}

////////////////////////////////////////////////////////////////////////////////////////////////////

}
}

#endif
