//! Execution and scheduling

use alloc::{boxed::Box, collections::VecDeque};

use hashbrown::{HashMap, HashSet};

use super::{state::LocalContext, term::LocalTerm};

/// The starting sequence number for port identifiers
pub const PORT_START: usize = 0x8000_0000_0000_0000;

/// Status of an [Execute] object
#[derive(Clone, Copy, PartialEq, Eq)]
pub enum ExecuteStatus {
    /// Waiting for a message
    Waiting,
    /// Ready to run
    Ready,
    /// Currently running
    Running,
    /// Marked for destruction, will never run
    Exited,
}

/// Executable identifier. Refers to an executable of any kind.
/// 
/// Internally represented as
///   - ID of the scheduler it was originally created on
///   - Sequence number within the scheduler it was originally created on
/// 
/// Executables might get transferred between schedulers, in which case
/// their ID does not change. If the MSB of the sequence number is set, the
/// executable should match `is_port`. If that bit is reset, it should match
/// `is_pid`. This distinction only matters to those guard functions.
#[derive(Clone, Copy, PartialEq, Eq, Hash, Debug, PartialOrd, Ord)]
pub struct Eid(pub usize, pub usize);

/// State common shared between all executables
pub struct CommonState {
    pub id: Eid,
    pub status: ExecuteStatus,
    /// Messages that had been sent to this executable
    pub mailbox: VecDeque<LocalTerm>,
    /// Priority (lower numeric value = higher priority)
    pub priority: u8,
}

/// Trait for things that can receive messages, be scheduled and be executed.
/// Such objects are referred to as "executables".
pub trait Execute: 'static {
    /// Gets a shared reference to the common state
    fn get_common_state(&self) -> &CommonState;
    /// Gets a mutable reference to the common state
    fn get_common_state_mut(&mut self) -> &mut CommonState;
    /// Runs the executable for a specified number of reductions
    fn run_for(&mut self, context: &mut LocalContext, reductions: isize);

    /// Returns the status of an executable
    fn get_status(&self) -> ExecuteStatus {
        self.get_common_state().status
    }

    /// Pushes a message into the queue
    fn send_message(&mut self, message: LocalTerm) {
        self.get_common_state_mut().mailbox.push_back(message);
    }
}

/// Allows executables to be created by the scheduler directly using an
/// initialization argument provided from the outside.
pub trait ExecuteMake<'i>: Sized + Execute {
    /// Whether the executable is a port or not. This only matters to the
    /// `is_port` and `is_pid` BIFs.
    const IS_PORT: bool;
    /// Initialization argument needed to construct the executable
    type Init: Sized;
    /// Executable creation error
    type Error: Sized;
    /// Creates an executable given the common state (generated by a scheduler)
    /// and an initialization argument specific to the executable type. The
    /// point of this indirection is to satisfy scheduler invariants.
    fn new(common: CommonState, init: &'i Self::Init, ctx: &LocalContext) -> Result<Self, Self::Error>;
}

/// Trait for things that run executables.
pub trait Schedule {
    /// Returns the scheduler ID
    fn get_id(&self) -> usize;
    /// Runs exactly zero or one executables up to an implementation-defined
    /// reduction limit. Returns `true` if an executable got ran, `false` if the
    /// run queue is empty.
    fn step(&mut self) -> bool;
    /// Constructs an executable and adds it to the scheduler
    fn add<'i, E: ExecuteMake<'i>>(&mut self, init: &'i E::Init, priority: u8) -> Result<Eid, E::Error>;
    /// Removes an executable from the scheduler
    fn remove(&mut self, id: Eid);

    /// Runs the scheduler in an infinite loop
    fn run(&mut self) -> ! {
        loop {
            self.step();
        }
    }
}

/// Trait for things that route messages.
pub trait TransferAgent {
    /// Forwards a message to the next node in the path. That might be an
    /// executable in a scheduler itself, or another scheduler, or a scheduler
    /// on another machine.
    fn route_message(&mut self, receiver: Eid, message: LocalTerm, ttl: usize);
}

/// A primitive transfer agent that collects messages while the scheduler is
/// busy, and processes them all at once once a process runs out of reductions.
#[derive(Default)]
pub struct LocalTransferAgent {
    local_executables: HashSet<Eid>,
    mailbox: VecDeque<(Eid, LocalTerm)>,
}

impl TransferAgent for LocalTransferAgent {
    fn route_message(&mut self, receiver: Eid, message: LocalTerm, mut ttl: usize) {
        #[cfg(feature = "trace-messages")]
        log::trace!("send {:?} to {:?} (ttl {ttl})", message, receiver);
        if self.local_executables.contains(&receiver) {
            self.mailbox.push_back((receiver, message));
        } else {
            // TODO: figure out who to send the message to
            ttl -= 1;
            if ttl > 0 {
                todo!();
            }
        }
    }
}

/// A primitive round robin scheduler with priorities
pub struct PrimitiveScheduler {
    /// Scheduler ID
    id: usize,
    /// Next "process" sequence number
    next_proc: usize,
    /// Next "port" sequence number
    next_port: usize,
    /// Mapping of executable IDs to executables
    executables: HashMap<Eid, Box<dyn Execute>>,
    /// Queue of executables and priorities that are [ExecuteStatus::Ready]
    run_queue: VecDeque<(Eid, u8)>,
    /// Execution context
    context: LocalContext,
}

impl PrimitiveScheduler {
    pub fn new(mut context: LocalContext, id: usize) -> PrimitiveScheduler {
        context.messenger = Some(<LocalTransferAgent as Default>::default());
        PrimitiveScheduler {
            id,
            next_proc: 0,
            next_port: PORT_START,
            executables: HashMap::new(),
            run_queue: VecDeque::new(),
            context,
        }
    }

    /// Generates a new [Eid]
    fn new_eid(&mut self, for_port: bool) -> Eid {
        if for_port {
            let res = Eid(self.id, self.next_port);
            self.next_port += 1;
            res
        } else {
            let res = Eid(self.id, self.next_proc);
            self.next_proc += 1;
            res
        }
    }
}

impl Schedule for PrimitiveScheduler {
    fn get_id(&self) -> usize {
        self.id
    }

    fn step(&mut self) -> bool {
        const REDUCTIONS: isize = 2000;

        // find executable with lowest numerical priority
        let (to_run_idx, _) = self.run_queue
            .iter()
            .enumerate()
            .fold((-1, 256usize), |acc @ (_, hi_prio), (idx, (_, priority))| {
                let priority = *priority as usize;
                if priority < hi_prio {
                    (idx as isize, priority)
                } else {
                    acc
                }
            });
        if to_run_idx < 0 {
            // empty run queue
            return false;
        }

        // get executable
        let exec_id = self.run_queue.remove(to_run_idx as usize).unwrap().0;
        let exec = self.executables.get_mut(&exec_id);
        let Some(exec) = exec else {
            panic!("inconsistent scheduler state: Eid {exec_id:?} in run queue but not in list of executables");
        };

        // run executable
        exec.get_common_state_mut().status = ExecuteStatus::Running;
        exec.run_for(&mut self.context, REDUCTIONS);

        // update state depending on new status
        let status = exec.get_common_state().status;
        let prio = exec.get_common_state().priority;
        match status {
            ExecuteStatus::Exited => {
                self.context.messenger.as_mut().unwrap().local_executables.remove(&exec_id);
                self.executables.remove(&exec_id);
            },
            ExecuteStatus::Running => {
                exec.get_common_state_mut().status = ExecuteStatus::Ready;
                self.run_queue.push_back((exec_id, prio));
            },
            ExecuteStatus::Ready => {
                self.run_queue.push_back((exec_id, prio));
            },
            ExecuteStatus::Waiting => (),
        }

        // process incoming messages
        loop {
            let message = self.context.messenger.as_mut().unwrap().mailbox.pop_front();
            let Some((receiver, message)) = message else { break };
            let Some(receiver) = self.executables.get_mut(&receiver) else { break };
            let state = receiver.get_common_state_mut();
            state.mailbox.push_back(message);
            if state.status == ExecuteStatus::Waiting {
                state.status = ExecuteStatus::Ready;
                self.run_queue.push_back((state.id, state.priority));
            }
        }

        // an executable got ran
        true
    }

    fn add<'i, E: ExecuteMake<'i>>(&mut self, init: &'i E::Init, priority: u8) -> Result<Eid, E::Error> {
        let id = self.new_eid(E::IS_PORT);
        let common = CommonState {
            id,
            priority,
            status: ExecuteStatus::Waiting,
            mailbox: Default::default(),
        };
        let executable = E::new(common, init, &self.context)?;
        self.executables.insert(id, Box::new(executable));
        self.run_queue.push_back((id, priority));
        self.context.messenger.as_mut().unwrap().local_executables.insert(id);
        Ok(id)
    }

    fn remove(&mut self, _id: Eid) {
        todo!();
    }
}
