/*
 * Copyright 2024 NAVER Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.navercorp.pinpoint.grpc.client;

import com.navercorp.pinpoint.common.util.Assert;
import com.navercorp.pinpoint.grpc.client.config.ClientOption;
import com.navercorp.pinpoint.grpc.client.config.ClientRetryOption;
import io.grpc.ClientInterceptor;
import io.grpc.NameResolverProvider;
import io.netty.handler.ssl.SslContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.LinkedList;
import java.util.Objects;

/**
 * @author Woonduk Kang(emeroad)
 */
public class DefaultChannelFactoryBuilder implements ChannelFactoryBuilder {

    private final Logger logger = LogManager.getLogger(this.getClass());

    private final String factoryName;

    private int executorQueueSize = 1024;
    private HeaderFactory headerFactory;

    private ClientOption clientOption;
    private SslContext sslContext;
    private ClientRetryOption clientRetryOption;

    private final LinkedList<ClientInterceptor> clientInterceptorList = new LinkedList<>();
    private NameResolverProvider nameResolverProvider;

    public DefaultChannelFactoryBuilder(String factoryName) {
        this.factoryName = Objects.requireNonNull(factoryName, "factoryName");
    }

    @Override
    public void setExecutorQueueSize(int executorQueueSize) {
        Assert.isTrue(executorQueueSize > 0, "must be `executorQueueSize > 0`");
        this.executorQueueSize = executorQueueSize;
    }

    @Override
    public void setHeaderFactory(HeaderFactory headerFactory) {
        this.headerFactory = Objects.requireNonNull(headerFactory, "headerFactory");
    }

    @Override
    public void addFirstClientInterceptor(ClientInterceptor clientInterceptor) {
        Objects.requireNonNull(clientInterceptor, "clientInterceptor");
        this.clientInterceptorList.addFirst(clientInterceptor);
    }

    @Override
    public void addClientInterceptor(ClientInterceptor clientInterceptor) {
        Objects.requireNonNull(clientInterceptor, "clientInterceptor");
        this.clientInterceptorList.add(clientInterceptor);
    }

    @Override
    public void setClientOption(ClientOption clientOption) {
        this.clientOption = Objects.requireNonNull(clientOption, "clientOption");
    }

    @Override
    public void setSslContext(SslContext sslContext) {
        this.sslContext = sslContext;
    }

    @Override
    public void setClientRetryOption(ClientRetryOption clientRetryOption) {
        this.clientRetryOption = clientRetryOption;
    }

    @Override
    public void setNameResolverProvider(NameResolverProvider nameResolverProvider) {
        this.nameResolverProvider = Objects.requireNonNull(nameResolverProvider, "nameResolverProvider");
    }

    @Override
    public ChannelFactory build() {
        logger.info("build ChannelFactory:{}", factoryName);
        Objects.requireNonNull(headerFactory, "headerFactory");
        Objects.requireNonNull(clientOption, "clientOption");

        return new DefaultChannelFactory(factoryName, executorQueueSize,
                headerFactory, nameResolverProvider,
                clientOption, clientInterceptorList,
                sslContext, clientRetryOption);
    }
}
