/*
 * Copyright 2020 NAVER Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.navercorp.pinpoint.pluginit.utils;

import com.navercorp.pinpoint.common.util.IOUtils;
import com.navercorp.pinpoint.it.plugin.utils.WebServer;
import fi.iki.elonen.NanoHTTPD;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.assertj.core.api.Assertions.assertThat;

/**
 * @author WonChul Heo(heowc)
 */
public class WebServerTest {

    private static WebServer webServer;

    @BeforeClass
    public static void newTestWebServer() throws Exception {
        webServer = WebServer.newTestWebServer();
    }

    @AfterClass
    public static void cleanup() {
        WebServer.cleanup(webServer);
    }

    @Test
    public void testGetHostAndPort() {
        assertThat(webServer.getHostAndPort()).startsWith("localhost:");
    }

    @Test
    public void testGetCallHttpUrl() {
        assertThat(webServer.getCallHttpUrl()).startsWith("http://localhost:");
    }

    @Test
    public void testHostname() {
        assertThat(webServer.getHostname()).isEqualTo("localhost");
    }

    @Test
    public void testServe() {
        Map<String, List<String>> params = new HashMap<>();
        NanoHTTPD.Response response = serve(params);
        assertResponse(response, "{}");

        params = new HashMap<>();
        params.put("foo", Collections.singletonList("bar"));
        response = serve(params);
        assertResponse(response, "{foo=[bar]}");

        params = new HashMap<>();
        params.put("a", Collections.singletonList("b"));
        params.put("c", Collections.singletonList("d"));
        response = serve(params);
        assertResponse(response, "{a=[b], c=[d]}");
    }

    private NanoHTTPD.Response serve(final Map<String, List<String>> params) {
        return webServer.serve(new NanoHTTPD.IHTTPSession() {
            @Override
            public void execute() {
                // ignored
            }

            @Override
            public NanoHTTPD.CookieHandler getCookies() {
                return null;
            }

            @Override
            public Map<String, String> getHeaders() {
                return new HashMap<>();
            }

            @Override
            public InputStream getInputStream() {
                return null;
            }

            @Override
            public NanoHTTPD.Method getMethod() {
                return NanoHTTPD.Method.GET;
            }

            @Override
            public Map<String, String> getParms() {
                return null;
            }

            @Override
            public Map<String, List<String>> getParameters() {
                return params;
            }

            @Override
            public String getQueryParameterString() {
                return null;
            }

            @Override
            public String getUri() {
                return null;
            }

            @Override
            public void parseBody(Map<String, String> files) {
                // ignored
            }

            @Override
            public String getRemoteIpAddress() {
                return null;
            }

            @Override
            public String getRemoteHostName() {
                return null;
            }
        });
    }

    private static void assertResponse(NanoHTTPD.Response response, String responseData) {
        try {
            assertThat(response.getStatus().getRequestStatus()).isEqualTo(NanoHTTPD.Response.Status.OK.getRequestStatus());
            assertThat(response.getMimeType()).isEqualTo(NanoHTTPD.MIME_HTML);
            final String actualData = new String(IOUtils.toByteArray(response.getData()), Charset.defaultCharset());
            assertThat(actualData).isEqualTo(responseData);
        } catch (IOException e) {
            // ignored
        }
    }
}
