/*
 * Copyright 2020 NAVER Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.navercorp.pinpoint.bootstrap.plugin.request;

import com.navercorp.pinpoint.bootstrap.config.Filter;
import com.navercorp.pinpoint.bootstrap.config.HttpStatusCodeErrors;
import com.navercorp.pinpoint.bootstrap.config.SkipFilter;
import com.navercorp.pinpoint.bootstrap.context.TraceContext;
import com.navercorp.pinpoint.bootstrap.plugin.RequestRecorderFactory;
import com.navercorp.pinpoint.bootstrap.plugin.http.HttpStatusCodeRecorder;
import com.navercorp.pinpoint.bootstrap.plugin.proxy.DisableRequestRecorder;
import com.navercorp.pinpoint.bootstrap.plugin.proxy.ProxyRequestRecorder;
import com.navercorp.pinpoint.bootstrap.plugin.request.util.DisableParameterRecorder;
import com.navercorp.pinpoint.bootstrap.plugin.request.util.ParameterRecorder;
import com.navercorp.pinpoint.bootstrap.plugin.request.util.RemoteAddressResolverFactory;
import com.navercorp.pinpoint.common.trace.ServiceType;
import com.navercorp.pinpoint.common.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * @author Woonduk Kang(emeroad)
 */
public class ServletRequestListenerBuilder<REQ> {
    private final ServiceType serviceType;
    private final TraceContext traceContext;
    private final RequestAdaptor<REQ> requestAdaptor;

    private String realIpHeader;
    private String realIpEmptyValue;

    private ParameterRecorder<REQ> parameterRecorder;

    private Filter<String> excludeUrlFilter;
    private Filter<String> traceExcludeMethodFilter;
    private RequestRecorderFactory<REQ> requestRecorderFactory;

    private HttpStatusCodeErrors httpStatusCodeErrors;

    private List<String> recordRequestHeaders;
    private List<String> recordRequestCookies;

    public ServletRequestListenerBuilder(final ServiceType serviceType,
                                         final TraceContext traceContext,
                                         final RequestAdaptor<REQ> requestAdaptor) {
        this.serviceType = Objects.requireNonNull(serviceType, "serviceType");

        this.traceContext = Objects.requireNonNull(traceContext, "traceContext");
        this.requestAdaptor = Objects.requireNonNull(requestAdaptor, "requestAdaptor");
    }



    public void setParameterRecorder(ParameterRecorder<REQ> parameterRecorder) {
        this.parameterRecorder = parameterRecorder;
    }

    public void setRealIpSupport(String realIpHeader, String realIpEmptyValue) {
        this.realIpHeader = realIpHeader;
        this.realIpEmptyValue = realIpEmptyValue;
    }

    public void setExcludeURLFilter(Filter<String> excludeUrlFilter) {
        this.excludeUrlFilter = excludeUrlFilter;
    }

    public void setTraceExcludeMethodFilter(Filter<String> traceExcludeMethodFilter) {
        this.traceExcludeMethodFilter = traceExcludeMethodFilter;
    }

    public void setRequestRecorderFactory(RequestRecorderFactory<REQ> requestRecorderFactory) {
        this.requestRecorderFactory = requestRecorderFactory;
    }

    // refactor~~
    public void setHttpStatusCodeRecorder(final HttpStatusCodeErrors httpStatusCodeErrors) {
        this.httpStatusCodeErrors = httpStatusCodeErrors;
    }

    public void setServerHeaderRecorder(List<String> recordRequestHeaders) {
        this.recordRequestHeaders = recordRequestHeaders;
    }

    public void setServerCookieRecorder(List<String> recordRequestCookies) {
        this.recordRequestCookies = recordRequestCookies;
    }

    private <T> Filter<T> newExcludeUrlFilter(Filter<T> excludeUrlFilter) {
        return filterNonNull(excludeUrlFilter);
    }

    private <T> Filter<T> newTraceExcludeMethodFilter(Filter<T> excludeMethodFilter) {
        return filterNonNull(excludeMethodFilter);
    }

    private static <T> Filter<T> filterNonNull(Filter<T> excludeUrlFilter) {
        if (excludeUrlFilter == null) {
            return new SkipFilter<>();
        }
        return excludeUrlFilter;
    }

    public ServletRequestListener<REQ> build() {

        RequestAdaptor<REQ> requestAdaptor = RemoteAddressResolverFactory.wrapRealIpSupport(this.requestAdaptor, realIpHeader, realIpEmptyValue);

        RequestTraceReader<REQ> requestTraceReader = new RequestTraceReader<>(traceContext, requestAdaptor, true);


        ProxyRequestRecorder<REQ> proxyRequestRecorder;
        if (requestRecorderFactory == null) {
            proxyRequestRecorder = new DisableRequestRecorder<>();
        } else {
            proxyRequestRecorder = requestRecorderFactory.getProxyRequestRecorder(requestAdaptor);
        }


        Filter<String> excludeUrlFilter = newExcludeUrlFilter(this.excludeUrlFilter);
        Filter<String> traceExcludeMethodFilter = newTraceExcludeMethodFilter(this.traceExcludeMethodFilter);

        final ServerRequestRecorder<REQ> serverRequestRecorder = newServerRequestRecorder(requestAdaptor);

        ParameterRecorder<REQ> parameterRecorder = newParameterRecorder();

        // not general api : http??
        HttpStatusCodeRecorder httpStatusCodeRecorder;
        if (httpStatusCodeErrors == null) {
            HttpStatusCodeErrors httpStatusCodeErrors = new HttpStatusCodeErrors(Collections.emptyList());
            httpStatusCodeRecorder = new HttpStatusCodeRecorder(httpStatusCodeErrors);
        } else {
            httpStatusCodeRecorder = new HttpStatusCodeRecorder(httpStatusCodeErrors);
        }


        return new ServletRequestListener<>(serviceType, traceContext, requestAdaptor, requestTraceReader,
                excludeUrlFilter, traceExcludeMethodFilter, parameterRecorder, proxyRequestRecorder,
                serverRequestRecorder, httpStatusCodeRecorder);
    }


    private ServerRequestRecorder<REQ> newServerRequestRecorder(RequestAdaptor<REQ> requestAdaptor) {
        final ServerHeaderRecorder<REQ> headerRecorder = newServerHeaderRecorder(requestAdaptor);
        final ServerCookieRecorder<REQ> cookieRecorder = newServerCookieRecorder(requestAdaptor);
        return new ServerRequestRecorder<>(requestAdaptor, headerRecorder, cookieRecorder);
    }

    private ServerHeaderRecorder<REQ> newServerHeaderRecorder(RequestAdaptor<REQ> requestAdaptor) {
        if (CollectionUtils.isEmpty(recordRequestHeaders)) {
            return new BypassServerHeaderRecorder<>();
        }

        if (AllServerHeaderRecorder.isRecordAllHeaders(recordRequestHeaders)) {
            return new AllServerHeaderRecorder<>(requestAdaptor);
        }

        return new DefaultServerHeaderRecorder<>(requestAdaptor, recordRequestHeaders);
    }

    private ServerCookieRecorder<REQ> newServerCookieRecorder(RequestAdaptor<REQ> requestAdaptor) {
        if (CollectionUtils.isEmpty(recordRequestCookies)) {
            return new BypassServerCookieRecorder<>();
        }

        if (!(requestAdaptor instanceof CookieSupportAdaptor)) {
            // unsupported
            return new BypassServerCookieRecorder<>();
        }
        CookieSupportAdaptor<REQ> cookieAdaptor = (CookieSupportAdaptor<REQ>) requestAdaptor;
        return new DefaultServerCookieRecorder<>(cookieAdaptor, recordRequestCookies);
    }

    private ParameterRecorder<REQ> newParameterRecorder() {
        if (this.parameterRecorder == null) {
            return new DisableParameterRecorder<>();
        }
        return this.parameterRecorder;
    }
}
