/*
 * Copyright 2022 NAVER Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.navercorp.pinpoint.bootstrap.plugin.reactor;

import com.navercorp.pinpoint.bootstrap.async.AsyncContextAccessorUtils;
import com.navercorp.pinpoint.bootstrap.context.AsyncContext;
import com.navercorp.pinpoint.bootstrap.context.SpanEventRecorder;
import com.navercorp.pinpoint.bootstrap.context.TraceContext;
import com.navercorp.pinpoint.bootstrap.interceptor.AsyncContextSpanEventApiIdAwareAroundInterceptor;
import com.navercorp.pinpoint.common.trace.ServiceType;

public class FluxAndMonoSubscribeInterceptor extends AsyncContextSpanEventApiIdAwareAroundInterceptor {
    private final ServiceType serviceType;

    public FluxAndMonoSubscribeInterceptor(TraceContext traceContext, ServiceType serviceType) {
        super(traceContext);
        this.serviceType = serviceType;
    }

    // AsyncContext must exist in Target for tracking.
    public AsyncContext getAsyncContext(Object target, Object[] args) {
        if (checkTargetReactorContextAccessor(target, args)) {
            return AsyncContextAccessorUtils.getAsyncContext(target);
        }
        if (checkTargetAsyncContextAccessor(target, args)) {
            return AsyncContextAccessorUtils.getAsyncContext(target);
        }
        if (checkSubscriberReactorContextAccessor(target, args)) {
            return AsyncContextAccessorUtils.getAsyncContext(target);
        }
        return AsyncContextAccessorUtils.getAsyncContext(target);
    }

    private boolean checkTargetReactorContextAccessor(final Object target, final Object[] args) {
        final AsyncContext asyncContext = ReactorContextAccessorUtils.getAsyncContext(target);
        if (asyncContext != null) {
            setReactorContextToSubscriber(asyncContext, args);
            return true;
        }
        return false;
    }

    private boolean checkTargetAsyncContextAccessor(final Object target, final Object[] args) {
        final AsyncContext asyncContext = AsyncContextAccessorUtils.getAsyncContext(target);
        if (asyncContext != null) {
            setReactorContextToTarget(asyncContext, target);
            setReactorContextToSubscriber(asyncContext, args);
            return true;
        }
        return false;
    }

    private boolean checkSubscriberReactorContextAccessor(final Object target, final Object[] args) {
        final AsyncContext asyncContext = ReactorContextAccessorUtils.getAsyncContext(args, 0);
        if (asyncContext != null) {
            setReactorContextToTarget(asyncContext, target);
            return true;
        }
        return false;
    }

    protected void setReactorContextToTarget(AsyncContext asyncContext, Object target) {
        ReactorContextAccessorUtils.setAsyncContext(asyncContext, target);
        if (isDebug) {
            logger.debug("Set reactorContext to target. reactorContext={}", asyncContext);
        }
    }

    protected void setReactorContextToSubscriber(AsyncContext asyncContext, Object[] args) {
        ReactorContextAccessorUtils.setAsyncContext(asyncContext, args, 0);
        if (isDebug) {
            logger.debug("Set reactorContext to args[0]. asyncContext={}", asyncContext);
        }
    }

    @Override
    public void doInBeforeTrace(SpanEventRecorder recorder, AsyncContext asyncContext, Object target, int apiId, Object[] args) {
    }

    public AsyncContext getAsyncContext(Object target, Object[] args, Object result, Throwable throwable) {
        return AsyncContextAccessorUtils.getAsyncContext(target);
    }

    @Override
    public void doInAfterTrace(SpanEventRecorder recorder, Object target, int apiId, Object[] args, Object result, Throwable throwable) {
        recorder.recordApiId(apiId);
        recorder.recordServiceType(serviceType);
        recorder.recordException(throwable);
    }
}
